/**
 * Database.js UnitTests
 */

var assert = require('assert');
var { Instance, RenderArea, Database } = require('../database');

describe('Instance', function () {
  describe('constructor()', function () {
    var instance = new Instance("192.168.1.1", 8080, 1);
    it('should exists when created', function () {
      assert.notEqual(instance, undefined);
      assert.notEqual(instance, null);
    });

    it('should return the correct values', function () {
      assert.strictEqual(instance.ip, "192.168.1.1");
      assert.strictEqual(instance.port, 8080);
      assert.strictEqual(instance.id, 1);
    });
  });
});

describe('RenderArea', function () {
  describe('constructor()', function () {
    var renderarea = new RenderArea("TheCone");
    it('should exists when created', function () {
      assert.notEqual(renderarea, undefined);
      assert.notEqual(renderarea, null);
    });

    it('should return the correct values', function () {
      assert.strictEqual(renderarea.name, "TheCone");
    });
  });
});

describe('Database', function () {
  var database;
  beforeEach(function () {
    database = new Database(":memory:", false);
  });
  afterEach(function () {
    database.close();
  });
  describe('constructor()', function () {
    it('should exists when created', function () {
      assert.notEqual(database, undefined);
      assert.notEqual(database, null);
    });
  });
  describe('registerInstance()', function () {
    it('should be able to register an instance', function () {
      var instance = database.registerInstance("1.1.1.1", 80);
      assert.notEqual(instance, null);
    });
    it('should be able to register an instance with the same port', function () {
      var instance1 = database.registerInstance("1.1.1.1", 80);
      var instance2 = database.registerInstance("1.1.1.2", 80);
      assert.notEqual(instance1, null);
      assert.notEqual(instance2, null);
      assert.notEqual(instance1.id, instance2.id);
    });
    it('should be able to register an instance with the same ip', function () {
      var instance1 = database.registerInstance("1.1.1.2", 80);
      var instance2 = database.registerInstance("1.1.1.2", 81);
      assert.notEqual(instance1, null);
      assert.notEqual(instance2, null);
      assert.notEqual(instance1.id, instance2.id);
    });
    it('should not be able to duplicate an instance', function () {
      database.registerInstance("1.1.1.1", 80);
      assert.throws(() => database.registerInstance("1.1.1.1", 80), Error);
    });
  });
  describe('getInstanceById()', function () {
    it('should return the correct instance', function () {
      var instance = database.registerInstance("1.1.1.1", 80);
      var instanceById = database.getInstanceById(instance.id);
      assert.strictEqual(instance.id, instanceById.id);
    });
  });
  describe('getInstances()', function () {
    it('should return the correct instance', function () {
      database.registerInstance("1.1.1.1", 80);
      var instances = database.getInstances();
      assert.strictEqual(instances.length, 1);
      assert.strictEqual(instances[0].ip, "1.1.1.1");
      assert.strictEqual(instances[0].port, 80);
    });
    it('should return an empty array if empty', function () {
      var instances = database.getInstances();
      assert.deepStrictEqual(instances, []);
    });
    it('should return the correct number of instances', function () {
      const rnd = Math.floor(Math.random() * 10);
      for (let i = 0; i < rnd; i++) {
        database.registerInstance("1.1.1.1", 80 + i);
      }
      var instances = database.getInstances();
      assert.strictEqual(instances.length, rnd);
    });
  });
  describe('deleteInstance()', function () {
    it('should remove an instance', function () {
      database.registerInstance("1.1.1.1", 80);
      database.registerInstance("1.1.1.2", 80);
      var instances = database.getInstances();
      assert.strictEqual(instances.length, 2);
      // Remove the 1.1.1.1:80 instance
      if (instances[0].ip === "1.1.1.1")
        database.deleteInstance(instances[0]);
      else
        database.deleteInstance(instances[1]);
      instances = database.getInstances();
      assert.strictEqual(instances.length, 1);
      // Should have the 1.1.1.2:80 remaining
      assert.strictEqual(instances[0].ip, "1.1.1.2");
      database.deleteInstance(instances[0]);
      instances = database.getInstances();
      assert.strictEqual(instances.length, 0);
    });
  });
  describe('createRenderArea()', function () {
    it('should be able to create a renderArea and link it to an instance', function () {
      database.registerInstance("1.1.1.1", 80);
      var instances = database.getInstances();
      var renderArea = new RenderArea("testRA");
      // Base
      database.createRenderArea(renderArea, instances[0]);
    });
    it('should be able to create 2 renderAreas and link it to an instance', function () {
      database.registerInstance("1.1.1.1", 80);
      var instances = database.getInstances();
      var renderArea1 = new RenderArea("testRA1");
      var renderArea2 = new RenderArea("testRA2");
      database.createRenderArea(renderArea1, instances[0]);
      database.createRenderArea(renderArea2, instances[0]);
    });
    it('should be able to create 1 renderArea and link it to 2 instances', function () {
      database.registerInstance("1.1.1.1", 80);
      database.registerInstance("1.1.1.2", 80);
      var instances = database.getInstances();
      var renderArea = new RenderArea("testRA");
      database.createRenderArea(renderArea, instances[0]);
      database.createRenderArea(renderArea, instances[1]);
    });
    it('should not be able to create a renderArea and link it two times to one instance', function () {
      database.registerInstance("1.1.1.1", 80);
      var instances = database.getInstances();
      var renderArea = new RenderArea("testRA");
      database.createRenderArea(renderArea, instances[0]);
      assert.throws(() => database.createRenderArea(renderArea, instances[0]), Error);
    });
  });
  describe('getInstance()', function () {
    it('should return an instance', function () {
      database.registerInstance("1.1.1.1", 80);
      var instances = database.getInstances();
      var renderArea = new RenderArea("testRA");
      database.createRenderArea(renderArea, instances[0]);
      var instance = database.getInstance(renderArea);
      assert.strictEqual(instance.ip, "1.1.1.1");
      assert.strictEqual(instance.port, 80);
    });
    it('should return null if not linked', function () {
      var renderArea = new RenderArea("testRA");
      var instance = database.getInstance(renderArea);
      assert.strictEqual(instance, null);
    });
  });
  describe('deleteRenderArea()', function () {
    it('should be able to remove a renderArea', function () {
      database.registerInstance("1.1.1.1", 80);
      var instances = database.getInstances();
      var renderArea = new RenderArea("testRA");
      database.createRenderArea(renderArea, instances[0]);
      database.deleteRenderArea(renderArea);
      assert.strictEqual(database.getNumRenderAreas(instances[0]), 0);
      assert.strictEqual(database.getInstance(renderArea), null);
    });
  });
  describe('getNumRenderAreas()', function () {
    it('should return the correct amount of RA per instance', function () {
      database.registerInstance("1.1.1.1", 80);
      database.registerInstance("1.1.1.2", 80);
      database.registerInstance("1.1.1.3", 80);
      var renderArea = new RenderArea("testRA");
      var instances = database.getInstances();
      database.createRenderArea(renderArea, instances[0]);
      database.createRenderArea(renderArea, instances[1]);
      database.createRenderArea(renderArea, instances[2]);
      // One per instance, based on previous tests
      instances.forEach(instance => {
        assert.strictEqual(database.getNumRenderAreas(instance), 1);
      });
      var renderArea1 = new RenderArea("testRA1");
      database.createRenderArea(renderArea1, instances[0]);
      assert.strictEqual(database.getNumRenderAreas(instances[0]), 2);
    });
  });
});