package remoteviz.monitoring.monitoringgrafana;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.influxdb.dto.Point;
import org.influxdb.dto.Query;
import org.influxdb.dto.QueryResult;
import org.influxdb.InfluxDB;
import org.influxdb.InfluxDBFactory;

import com.openinventor.remoteviz.rendering.Client;
import com.openinventor.remoteviz.rendering.Connection;
import com.openinventor.remoteviz.rendering.MetricsListener;
import com.openinventor.remoteviz.rendering.RenderArea;

public class MonitoringMetrics extends MetricsListener
{
  private InfluxDB influxDB;
  private final static String IP_ADDRESS = "127.0.0.1";
  private final static int UDP_PORT = 4444;
  private final static int HTTP_PORT = 8086;
  private final static String DATABASE_NAME = "rv_monitoring";
  private static ExecutorService executorService = null;

  class PostMetric implements Runnable
  {
    private InfluxDB influxDB;
    private int udpPort;
    private String name;
    private long value;
    private String tagName;
    private String tagValue;

    public PostMetric(InfluxDB influxDB, int udpPort, String name, long value, String tagName, String tagValue)
    {
      this.influxDB = influxDB;
      this.udpPort = udpPort;
      this.name = name;
      this.value = value;
      this.tagName = tagName;
      this.tagValue = tagValue;
    }

    @Override
    public void run()
    {
      Point point = null;
      if (this.tagName != null)
        point = Point.measurement(this.name).tag(this.tagName, this.tagValue).addField("value", this.value).build();
      else
        point = Point.measurement(this.name).addField("value", this.value).build();
      this.influxDB.write(this.udpPort, point);
    }
  }

  public MonitoringMetrics()
  {
    try
    {
      this.influxDB = InfluxDBFactory.connect("http://" + IP_ADDRESS + ":" + HTTP_PORT);

      // Check if the database exists
      QueryResult result = this.influxDB.query(new Query("SHOW DATABASES"));
      List<List<Object>> databaseNames = result.getResults().get(0).getSeries().get(0).getValues();
      List<String> databases = new ArrayList<>();
      if (databaseNames != null)
      {
        for (List<Object> database : databaseNames)
        {
          databases.add(database.get(0).toString());
        }
      }

      // Create the database if necessary
      if (!databases.contains(DATABASE_NAME))
      {
        String createDatabaseQueryString = String.format("CREATE DATABASE \"%s\"", DATABASE_NAME);
        this.influxDB.query(new Query(createDatabaseQueryString));
      }
    }
    catch(Exception e)
    {
      System.out.println("Is the InfluxDb server running?");
      System.exit(-1);
    }

    System.out.println(String.format("Connected to InfluxDB: \"%s:%s\"", IP_ADDRESS, UDP_PORT));

    this.executorService = Executors.newSingleThreadExecutor();
  }

  @Override
  public void onMeasuredNetworkLatency(int time, Client client)
  {
    this.executorService.execute(new PostMetric(this.influxDB, UDP_PORT, "rv_network_latency", (long)time, "client_id", client.getId()));
  }

  @Override
  public void onMeasuredDecodingTime(int time, Connection connection)
  {
    this.executorService.execute(new PostMetric(this.influxDB, UDP_PORT, "rv_decoding_time", (long)time, "connection_id", connection.getId()));
  }

  @Override
  public void onMeasuredRenderingTime(int time, RenderArea renderArea)
  {
    this.executorService.execute(new PostMetric(this.influxDB, UDP_PORT, "rv_rendering_time", (long)time, "renderarea_id", renderArea.getId()));
  }

  @Override
  public void onMeasuredEncodingTime(int time, Connection connection)
  {
    this.executorService.execute(new PostMetric(this.influxDB, UDP_PORT, "rv_encoding_time", (long)time, "connection_id", connection.getId()));
  }

  @Override
  public void onChangedNumClients(int number)
  {
    this.executorService.execute(new PostMetric(this.influxDB, UDP_PORT, "rv_num_clients", (long)number, null, null));
  }

  @Override
  public void onChangedNumConnections(int number)
  {
    this.executorService.execute(new PostMetric(this.influxDB, UDP_PORT, "rv_num_connections", (long)number, null, null));
  }

  @Override
  public void onChangedNumRenderAreas(int number)
  {
    this.executorService.execute(new PostMetric(this.influxDB, UDP_PORT, "rv_num_renderareas", (long)number, null, null));
  }

}
