package volumeviz.sample.undefinedVoxel;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoSceneManager.AntialiasingModes;
import com.openinventor.inventor.actions.SoGLRenderAction.TransparencyTypes;
import com.openinventor.inventor.nodes.SoBaseColor;
import com.openinventor.inventor.nodes.SoCube;
import com.openinventor.inventor.nodes.SoDrawStyle;
import com.openinventor.inventor.nodes.SoFile;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoTranslation;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.nodes.SceneOrbiter;
import com.openinventor.inventor.viewercomponents.nodes.SoViewingCube;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.draggers.SoOrthoSliceDragger;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;
import com.openinventor.volumeviz.nodes.SoVolumeShape;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  private IRenderAreaInteractive m_viewer;
  private SoVolumeData m_volData;
  private SoOrthoSlice m_slice;
  private JCheckBox m_undefinedCheckbox;
  final byte ARBITRARY_UNDEFINED_VALUE = 0;

  @Override
  public void start()
  {

    ////////////////////////////////////////////////////////
    /// Scene Graph
    SoSeparator root = new SoSeparator();

    SoSeparator volRendSep = new SoSeparator();
    root.addChild(volRendSep);

    final int volDim = 100;
    m_volData = CreateVolumeData(volDim);
    volRendSep.addChild(m_volData);
    m_volData.undefinedValue.setValue(ARBITRARY_UNDEFINED_VALUE);

    SoTransferFunction transFunction = new SoTransferFunction();
    volRendSep.addChild(transFunction);
    transFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);
    transFunction.actualColorMap.set1Value(3, 0.0f);

    m_slice = new SoOrthoSlice();
    volRendSep.addChild(m_slice);
    m_slice.sliceNumber.setValue(volDim * 3 / 4);
    m_slice.alphaUse.setValue(SoSlice.AlphaUses.ALPHA_AS_IS);
    m_slice.clipping.setValue(true);
    m_slice.enableBorder.setValue(true);
    m_slice.borderWidth.setValue(3.0f);

    SoOrthoSliceDragger orthoSliceDragger = new SoOrthoSliceDragger();
    volRendSep.addChild(orthoSliceDragger);
    SoPath path = new SoPath(m_slice);
    orthoSliceDragger.orthoSlicePath.setValue(path);

    SoVolumeRenderingQuality volumeQuality = new SoVolumeRenderingQuality();
    volumeQuality.ambientOcclusion.setValue(true);
    volRendSep.addChild(volumeQuality);

    SoVolumeRender volumeRender = new SoVolumeRender();
    volumeRender.interpolation.connectFrom(m_slice.interpolation);
    volRendSep.addChild(volumeRender);

    // Add Bbox
    root.addChild(createBBox(m_volData.extent.getValue()));

    ////////////////////////////////////////////////////////
    /// Viewer
    m_viewer = ViewerComponentsFactory.createRenderAreaOrbiter();
    setBackground(Color.white);
    setLayout(new BorderLayout());
    add(buildOptionsPanel(), BorderLayout.NORTH);
    add(m_viewer.getComponent(), BorderLayout.CENTER);

    m_viewer.setSceneGraph(root);

    final float grayVal = 0.99f;
    final SbColor bgColor = new SbColor(grayVal, grayVal, grayVal);
    SoGradientBackground gradientBackground = new SoGradientBackground();
    gradientBackground.color0.setValue(bgColor);
    gradientBackground.color1.setValue(bgColor);
    root.insertChild(gradientBackground, 0);

    SoViewingCube vCube = ((SceneOrbiter) m_viewer.getSceneInteractor()).getViewingCube();
    vCube.opacityMin.setValue(0.0f);
    vCube.opacityMax.setValue(0.8f);
    SoFile compass = new SoFile();
    compass.name.setValue("$OIVJHOME/data/examples/ViewingCube/Compass/compass-rgb.iv");
    vCube.compass.setValue(compass);

    m_viewer.getComponent().setPreferredSize(new Dimension(1280, 720));
    m_viewer.setTransparencyType(TransparencyTypes.OPAQUE_FIRST);

    m_viewer.setAntialiasingMode(AntialiasingModes.AUTO);
    m_viewer.setAntialiasingQuality(1);
    m_viewer.viewAll(new SbViewportRegion());
  }

  private JPanel buildOptionsPanel()
  {
    JPanel panel = new JPanel();
    panel.setLayout(new FlowLayout(FlowLayout.LEFT, 5, 5));

    m_undefinedCheckbox = new JCheckBox("Undefined value");
    panel.add(m_undefinedCheckbox);
    m_undefinedCheckbox.setSelected(true);

    // Connections
    m_undefinedCheckbox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( e.getStateChange() == ItemEvent.SELECTED )
          m_volData.undefinedValue.setValue(ARBITRARY_UNDEFINED_VALUE);
        else
          m_volData.undefinedValue.setValue(Double.NaN);
      }
    });

    panel.add(new JLabel("        Interpolation"));

    final String interpolationModesStr[] = { "NEAREST", "LINEAR", "CUBIC" };
    final SoVolumeShape.Interpolations interpolationModes[] = {
      SoVolumeShape.Interpolations.NEAREST,
      SoVolumeShape.Interpolations.LINEAR,
      SoVolumeShape.Interpolations.CUBIC
    };
    JComboBox<String> interpolationModeCombo = new JComboBox<String>(interpolationModesStr);
    interpolationModeCombo.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( e.getStateChange() == ItemEvent.SELECTED )
          m_slice.interpolation.setValue(
              interpolationModes[Math.min(interpolationModeCombo.getSelectedIndex(), interpolationModes.length)]);
      }
    });
    interpolationModeCombo.setSelectedIndex(1);
    panel.add(interpolationModeCombo);

    return panel;
  }

  @Override
  public void stop()
  {
    m_viewer.dispose();
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("UndefinedVoxel");
  }

  ////////////////////////////////////////////////////////////////////////
  // Helper for Bbox creation
  private SoSeparator createBBox(SbBox3f bbox)
  {
    SoSeparator bboxSep = new SoSeparator();

    SoPickStyle pickStyle = new SoPickStyle();
    bboxSep.addChild(pickStyle);
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    SoDrawStyle drawStyle = new SoDrawStyle();
    bboxSep.addChild(drawStyle);
    drawStyle.style.setValue(SoDrawStyle.Styles.LINES);
    drawStyle.lineWidth.setValue(2.0f);

    SoBaseColor bboxMat = new SoBaseColor();
    final float grayVal = 0.1f;
    bboxMat.rgb.setValue(new SbVec3f(grayVal, grayVal, grayVal));
    bboxSep.addChild(bboxMat);

    SoLightModel lightModel = new SoLightModel();
    bboxSep.addChild(lightModel);
    lightModel.model.setValue(SoLightModel.Models.BASE_COLOR);

    SoTranslation translation = new SoTranslation();
    bboxSep.addChild(translation);
    translation.translation.setValue(bbox.getCenter());

    SoCube cube = new SoCube();
    SbVec3f bboxSize = bbox.getSize();
    cube.width.setValue(bboxSize.getValueAt(0));
    cube.height.setValue(bboxSize.getValueAt(1));
    cube.depth.setValue(bboxSize.getValueAt(2));
    bboxSep.addChild(cube);

    return bboxSep;
  }

  ////////////////////////////////////////////////////////////////////////
  // Fake volume
  private SoVolumeData CreateVolumeData(int volDim)
  {
    ByteBuffer data = ByteBuffer.allocateDirect(volDim * volDim * volDim);
    data.order(ByteOrder.nativeOrder());

    int center = volDim / 2;
    int radius = (int) (volDim / 1.8);

    for ( int i = 0; i < volDim; ++i )
      for ( int j = 0; j < volDim; ++j )
        for ( int k = 0; k < volDim; ++k )
          if ( ((i - center) * (i - center) + (j - center) * (j - center) + (k - center) * (k - center)) < radius
              * radius )
            data.put(ARBITRARY_UNDEFINED_VALUE);
          else
            data.put((byte) 150);

    SoVolumeData volData = new SoVolumeData();
    volData.data.setValue(new SbVec3i32(volDim, volDim, volDim), new SbDataType(SbDataType.DataTypes.UNSIGNED_BYTE),
        data);
    return volData;
  }
}
