package volumeviz.sample.uniformTiles;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.nio.ByteBuffer;
import javax.swing.JLabel;
import javax.swing.JPanel;
import com.openinventor.inventor.SbBox2i32;
import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbBox3i32;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2d;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoSceneManager.AntialiasingModes;
import com.openinventor.inventor.devices.SoBufferObject;
import com.openinventor.inventor.devices.SoBufferObject.AccessModes;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoComplexity;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoInteractiveComplexity;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.inventor.viewercomponents.nodes.SceneOrbiter;
import com.openinventor.ldm.nodes.SoDataSet;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.ldm.readers.SoVolumeReader;
import com.openinventor.ldm.tiles.SoCpuBufferUniform;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;
import util.Example;
import util.ViewerComponentsFactory;

/**
 * This example illustrates the benefit of providing uniform tiles in a volume
 * reader for each tile containing a constant value. 2 render areas display an
 * equivalent dataset having the same size, same extent and same tile size. The
 * datasets have only tiles with a constant value which is random. Thus, the
 * rendering of such dataset is a mosaic of small cubes with a single color on
 * each cube (a cube represents a tile). Both render areas use the same volume
 * reader class RandomReader. However the instance of RandomReader used by the
 * left render area allocates only standard buffers (SoCPUBuffer) in the
 * readTile method, while the instance on the right render area allocates only
 * uniform buffers (SoCPUBufferUniform).
 * <p>
 * The rendering are not equal on each render areas for 2 reasons:
 * <ul>
 * <li>the tile's value is defined randomly
 * <li>the use of uniform buffer might lead to many more tiles loaded on the
 * right renderarea.
 * </ul>
 * 2 sliders are provided in the user interface of this example to define the
 * maximum amount of memory allocated by the volume rendering. The first slider
 * defines the maximum memory used to load tiles on the GPU. The second slider
 * defines the maximum amount of texture memory on the CPU used to render the
 * dataset.
 * <p>
 * The goal of this example is to illustrate the benefit of uniform buffer
 * versus standard buffer when using the same maximum amount of memory. If the
 * maximum amount of memory is small, the right render area display many more
 * tiles which mimics an higher rendering quality. Moreover, if the maximum
 * amount of the memory is quite large, both render area display the same image
 * but after a different duration. The right render area displays the highest
 * resolution of the volume much earlier than the left render area.
 */

public class Main extends Example
{

  public class RandomReader extends SoVolumeReader
  {
    SoDataSet.DataTypes m_type;
    SbVec3i32 m_dim;
    SbVec3i32 m_tileDim;
    boolean m_doUniform;
    int m_tileBytes;

    public RandomReader(boolean doUniform, SbVec3i32 dim)
    {
      m_type = SoDataSet.DataTypes.UNSIGNED_BYTE;
      m_dim = dim;
      m_tileDim = new SbVec3i32(64, 64, 64);
      m_doUniform = doUniform;
    }

    @Override
    public DataInfo getDataChar()
    {
      DataInfo dataInfo = new DataInfo();
      dataInfo.type = m_type;
      dataInfo.dim = m_dim;
      dataInfo.size = new SbBox3f(0.0f, 0.0f, 0.0f, (float) m_dim.getValueAt(0), (float) m_dim.getValueAt(1),
          (float) m_dim.getValueAt(2));
      dataInfo.readError = ReadErrors.RD_NO_ERROR;
      m_tileBytes = m_tileDim.getValueAt(0) * m_tileDim.getValueAt(1) * m_tileDim.getValueAt(2);
      return dataInfo;
    }

    @Override
    public SbVec3i32 getTileSize()
    {
      return m_tileDim;
    }

    @Override
    public boolean isThreadSafe()
    {
      return true;
    }

    @Override
    public boolean isDataConverted()
    {
      return true;
    }

    @Override
    public SoBufferObject readTile(int index, SbBox3i32 tilePosition)
    {
      SoCpuBufferObject tileBuffer = null;
      double voxelValue = Math.random() * 255.0d;

      if ( m_doUniform )
      {
        tileBuffer = new SoCpuBufferUniform(voxelValue, new SbDataType(SbDataType.DataTypes.UNSIGNED_BYTE));
      }
      else
      {
        tileBuffer = new SoCpuBufferObject();
        tileBuffer.setSize(m_tileBytes);
        ByteBuffer buffer = tileBuffer.map(AccessModes.SET);
        for ( int i = 0; i < m_tileBytes; i++ )
          buffer.put((byte) voxelValue);
        tileBuffer.unmap();
      }

      return tileBuffer;
    }

    @Override
    public void getSubSlice(SbBox2i32 subSlice, int sliceNumber, SoBufferObject bufferObject)
    {}

    @Override
    public SbVec2d getTileMinMax(int fileId)
    {
      // Dummy values are returned only for the purpose of this demo
      if ( m_doUniform ) // LDM loads efficiently uniform tiles only if the min
                         // & max are equal
        return new SbVec2d(0, 0);
      else
        return new SbVec2d(-1, 1);
    }
  }

  private IRenderAreaInteractive m_viewerLeft;
  private IRenderAreaInteractive m_viewerRight;
  private SoVolumeData m_volumeDataLeft;
  private SoVolumeData m_volumeDataRight;

  @Override
  public void start()
  {
    m_viewerLeft = ViewerComponentsFactory.createRenderAreaOrbiter();
    m_viewerRight = ViewerComponentsFactory.createRenderAreaOrbiter();

    // GUI
    setBackground(Color.white);
    setLayout(new BorderLayout());
    add(buildOptionsPanel(), BorderLayout.NORTH);

    final Component componentLeft = m_viewerLeft.getComponent();
    final Container containerLeft = new Container();
    containerLeft.setLayout(new BorderLayout());
    componentLeft.setPreferredSize(new Dimension(640, 720));
    containerLeft.add(componentLeft, BorderLayout.NORTH);
    JLabel labelLeft = new JLabel(" Without uniform tiles");
    labelLeft.setFont(new Font(labelLeft.getName(), Font.PLAIN, 22));
    labelLeft.setBackground(Color.white);
    labelLeft.setOpaque(true);
    containerLeft.add(labelLeft, BorderLayout.WEST);
    add(containerLeft, BorderLayout.WEST);

    final Component componentRight = m_viewerRight.getComponent();
    final Container containerRigth = new Container();
    containerRigth.setLayout(new BorderLayout());
    componentRight.setPreferredSize(new Dimension(640, 720));
    containerRigth.add(componentRight, BorderLayout.NORTH);
    JLabel labelRight = new JLabel(" With uniform tiles");
    labelRight.setFont(new Font(labelLeft.getName(), Font.PLAIN, 22));
    labelRight.setBackground(Color.white);
    labelRight.setOpaque(true);
    containerRigth.add(labelRight, BorderLayout.WEST);
    add(containerRigth, BorderLayout.EAST);

    // Scene graph
    SoSeparator sep1 = new SoSeparator();
    SoSeparator sep2 = new SoSeparator();

    final float grayVal = 1.0f;
    final SbColor bgColor = new SbColor(grayVal, grayVal, grayVal);
    SoGradientBackground gradientBackground = new SoGradientBackground();
    gradientBackground.color0.setValue(bgColor);
    gradientBackground.color1.setValue(bgColor);
    sep1.addChild(gradientBackground);
    sep2.addChild(gradientBackground);

    final int volumeEdge = 1024;
    SbVec3i32 volumeSize = new SbVec3i32(volumeEdge, volumeEdge, volumeEdge);

    {
      SoInteractiveComplexity interactiveComplexity = new SoInteractiveComplexity();
      interactiveComplexity.fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
      sep1.addChild(interactiveComplexity);
      sep1.addChild(new SoComplexity());

      m_volumeDataLeft = new SoVolumeData();
      RandomReader reader = new RandomReader(false, volumeSize);
      m_volumeDataLeft.setReader(reader);
      sep1.addChild(m_volumeDataLeft);

      // Transfer function in a file
      SoTransferFunction transfertFunction = new SoTransferFunction();
      transfertFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);
      sep1.addChild(transfertFunction);

      sep1.addChild(new SoVolumeRenderingQuality());

      SoVolumeRender volRender = new SoVolumeRender();
      volRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
      sep1.addChild(volRender);
    }

    {
      SoInteractiveComplexity interactiveComplexity = new SoInteractiveComplexity();
      interactiveComplexity.fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
      sep2.addChild(interactiveComplexity);
      sep2.addChild(new SoComplexity());

      m_volumeDataRight = new SoVolumeData();
      RandomReader reader = new RandomReader(true, volumeSize);
      m_volumeDataRight.setReader(reader);
      sep2.addChild(m_volumeDataRight);

      SoTransferFunction transfertFunction = new SoTransferFunction();
      transfertFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);
      sep2.addChild(transfertFunction);

      sep2.addChild(new SoVolumeRenderingQuality());

      SoVolumeRender volRender = new SoVolumeRender();
      volRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
      sep2.addChild(volRender);
    }

    m_viewerLeft.setSceneGraph(sep1);
    m_viewerRight.setSceneGraph(sep2);

    m_viewerLeft.setAntialiasingMode(AntialiasingModes.AUTO);
    m_viewerLeft.setAntialiasingQuality(1);

    m_viewerRight.setAntialiasingMode(AntialiasingModes.AUTO);
    m_viewerRight.setAntialiasingQuality(1);

    ((SceneOrbiter) m_viewerLeft.getSceneInteractor()).enableViewingCube(false);
    ((SceneOrbiter) m_viewerRight.getSceneInteractor()).enableViewingCube(false);

    SoCamera cam = m_viewerLeft.getSceneInteractor().getCamera();
    cam.position.setValue(new SbVec3f(874.039f, 874.039f, 874.039f));
    cam.orientation.setValue(new SbRotation(new SbVec3f(-0.590284f, 0.769274f, 0.244504f), 0.987861f));

    m_viewerRight.getSceneInteractor().getCamera().position
        .connectFrom(m_viewerLeft.getSceneInteractor().getCamera().position);
    m_viewerRight.getSceneInteractor().getCamera().orientation
        .connectFrom(m_viewerLeft.getSceneInteractor().getCamera().orientation);
    m_viewerLeft.getSceneInteractor().getCamera().position
        .connectFrom(m_viewerRight.getSceneInteractor().getCamera().position);
    m_viewerLeft.getSceneInteractor().getCamera().orientation
        .connectFrom(m_viewerRight.getSceneInteractor().getCamera().orientation);

    m_viewerLeft.viewAll(new SbViewportRegion());
    m_viewerRight.viewAll(new SbViewportRegion());

    final int maxTexValue = 5;
    final int mainTexValue = 10;

    m_volumeDataLeft.ldmResourceParameters.getValue().maxTexMemory.setValue(maxTexValue);
    m_volumeDataRight.ldmResourceParameters.getValue().maxTexMemory.setValue(maxTexValue);

    m_volumeDataLeft.ldmResourceParameters.getValue().maxMainMemory.setValue(mainTexValue);
    m_volumeDataRight.ldmResourceParameters.getValue().maxMainMemory.setValue(mainTexValue);
  }

  @Override
  public void stop()
  {
    m_viewerLeft.dispose();
    m_viewerRight.dispose();
  }

  private JPanel buildOptionsPanel()
  {
    // Build Highlight parameters panel
    SliderPanel m_videoMemSlider = new SliderPanel(1.0f, 4096.0f, 5.0f, 1, false);

    SliderPanel m_cpuMemSlider = new SliderPanel(1.0f, 4096.0f, 10.0f, 1, false);

    ((FlowLayout) m_videoMemSlider.getLayout()).setAlignment(FlowLayout.LEFT);
    ((FlowLayout) m_cpuMemSlider.getLayout()).setAlignment(FlowLayout.LEFT);

    m_videoMemSlider.addInfoText("Max GPU memory used (MB)");
    m_cpuMemSlider.addInfoText("Max CPU memory used (MB)");

    m_videoMemSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        if ( m_volumeDataLeft != null )
        {
          m_volumeDataLeft.ldmResourceParameters.getValue().maxTexMemory.setValue((int) value);
          m_volumeDataRight.ldmResourceParameters.getValue().maxTexMemory.setValue((int) value);
        }
      }
    });
    m_cpuMemSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        if ( m_volumeDataLeft != null )
        {
          m_volumeDataLeft.ldmResourceParameters.getValue().maxMainMemory.setValue((int) value);
          m_volumeDataRight.ldmResourceParameters.getValue().maxMainMemory.setValue((int) value);
        }
      }
    });

    JPanel highlightPanel = new JPanel();
    highlightPanel.setLayout(new FlowLayout(FlowLayout.LEFT, 5, 5));
    highlightPanel.add(m_videoMemSlider);
    highlightPanel.add(m_cpuMemSlider);

    return highlightPanel;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Uniform tiles");
  }
}
