package com.openinventor.inventor.viewercomponents.awt;

import java.awt.Component;

import com.openinventor.inventor.SoSceneManager;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.nodes.SoNode;

/**
 * Interface for basic render area.
 *
 */
public interface IRenderArea
{

  /**
   * Gets the parent component of this render area.
   *
   * @return the parent component
   */
  public Component getComponent();

  /**
   * Sets the scene graph. This is the Open Inventor scene which will be
   * traversed for rendering and event processing.
   *
   * @param sceneGraph
   *          the scene graph
   */
  public void setSceneGraph(SoNode sceneGraph);

  /**
   * Gets the GLRenderAction used for rendering.
   *
   * @return the GLRenderAction used for rendering
   */
  public SoGLRenderAction getGLRenderAction();

  /**
   * Sets the GLRenderAction used for rendering.
   *
   * @param ra
   *          the GLRenderAction used for rendering
   */
  public void setGLRenderAction(SoGLRenderAction ra);

  /**
   * Gets the global transparency algorithm to use when rendering.
   *
   * @return the global transparency algorithm
   */
  public SoGLRenderAction.TransparencyTypes getTransparencyType();

  /**
   * Sets the global transparency algorithm to use when rendering.
   *
   * @param type
   *          the global transparency algorithm
   */
  public void setTransparencyType(SoGLRenderAction.TransparencyTypes type);

  /**
   * Gets the antialiasing quality value.
   *
   * @return the antialiasing quality, factor in the range [0.0,1.0].
   *
   */
  public float getAntialiasingQuality();

  /**
   * Sets the antialiasing quality value.
   * <p>
   * Default value is 0.0. The value 0.0 turns off antialiasing.
   *
   * @param quality
   *          The quality is a factor in the range [0.0,1.0].
   *
   */
  public void setAntialiasingQuality(float quality);

  /**
   * Gets the antialiasing mode.
   *
   * @return the antialiasing algorithm
   */
  public SoSceneManager.AntialiasingModes getAntialiasingMode();

  /**
   * Sets the antialiasing mode.
   * <p>
   * Default value is NO_ANTIALIASING which turns off antialiasing.
   *
   * @param mode
   *          the antialiasing algorithm.
   */
  public void setAntialiasingMode(SoSceneManager.AntialiasingModes mode);

  /**
   * Gets quality for supersampling when "still" (not interacting).
   *
   * @return the quality for supersampling, factor in the range [0.0,1.0].
   */
  public float getStillSuperSamplingQuality();

  /**
   * Sets quality for supersampling when "still" (not interacting).
   * <p>
   * When quality is greater than 0, still images will be automatically
   * supersampled. Default value is 0.0. Use the value 0.0 to turn off still
   * supersampling. 0.5 is a typical value.
   *
   * @param quality
   *          The quality is a factor in the range [0.0,1.0].
   */
  public void setStillSuperSamplingQuality(float quality);

  /**
   * Gets delay for supersampling when "still" (not interacting).
   *
   * @return the delay for supersampling in milliseconds.
   */
  public int getStillSuperSamplingDelay();

  /**
   * Sets delay for supersampling when "still" (not interacting).
   * <p>
   * If greater than 0, images will be supersampled after the specified delay.
   * Default value is 0. The delay is in milliseconds.
   *
   * @param delay
   *          The delay for supersampling in milliseconds.
   */
  public void setStillSuperSamplingDelay(int delay);

  /**
   * Schedules a redraw for some time in the near future.
   */
  public void scheduleRedraw();

  /**
   * Activates or deactivates raw stereo.
   *
   * @param activated
   *          true to activate raw stereo
   */
  public void activateStereo(boolean activated);

  /**
   * Gets availability of raw stereo.
   *
   * @return true if the raw stereo is available
   */
  public boolean isRawStereoAvailable();

  /**
   * Dispose area resources.
   */
  public void dispose();

}
