package com.openinventor.inventor.viewercomponents.awt.glcanvas.renderareas;

import java.awt.Cursor;
import java.awt.Point;
import java.awt.Toolkit;

import javax.swing.ImageIcon;

import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionModeListener;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.NavigationMode;

/**
 * Class to render an OpenInventor scene graph in an AWT GLCanvas.
 * This class extends RenderAreaInteractive to add examiner viewer behaviors.
 * <p>
 * Note: When using this component, the automatic interactive mode is disabled,
 * see
 * {@link com.openinventor.inventor.SoSceneManager#setAutoInteractiveMode(boolean)}
 * for more details. Interactive mode is managed by the SceneExaminer node.
 */
public class RenderAreaExaminer extends RenderAreaInteractive implements IRenderAreaExaminer
{
  private static final Cursor SEEK_CURSOR =
      createCursor("/com/openinventor/inventor/viewercomponents/icons/CursorSeek.png", "seek cursor");
  private static final Cursor VIEWING_CURSOR =
      createCursor("/com/openinventor/inventor/viewercomponents/icons/CursorCurvedHand.png", "hand cursor");

  private SceneExaminer m_examinerRootSceneGraph;

  public RenderAreaExaminer()
  {
    super(false);

    init();
  }

  public RenderAreaExaminer(GLCapabilities caps)
  {
    super(false, caps);

    init();
  }

  private void init()
  {
    // Disable SceneExaminer automatic adjustment of clipping planes,
    // adjustment is managed by the render area.
    SoPreferences.setValue("OIV_SCENE_EXAMINER_AUTO_CLIPPING_PLANES", "false");
    m_rootSceneGraph = m_examinerRootSceneGraph = new SceneExaminer();

    buildSceneGraph();

    updateInteractionCursor();
    m_examinerRootSceneGraph.addInteractionModeListener(new InteractionModeListener()
    {
      @Override
      public void seekModeChanged(boolean onOrOff)
      {
        updateSeekCursor(onOrOff);
      }

      @Override
      public void interactionModeChanged(InteractionMode newMode)
      {
        updateInteractionCursor();
      }
    });
  }

  @Override
  public void init(GLAutoDrawable drawable)
  {
    super.init(drawable);

    // Deactivate auto interactive mode,
    // interactive mode is managed by the SceneExaminer.
    m_renderAreaCore.getSceneManager().setAutoInteractiveMode(false);
  }

  @Override
  public void setSeekMode(boolean onOrOff)
  {
    m_examinerRootSceneGraph.setSeekMode(onOrOff);
  }

  @Override
  public void setNavigationMode(NavigationMode mode)
  {
    m_examinerRootSceneGraph.setNavigationMode(mode);
  }

  @Override
  public NavigationMode getNavigationMode()
  {
    return m_examinerRootSceneGraph.getNavigationMode();
  }

  @Override
  public void setInteractionMode(InteractionMode mode)
  {
    m_examinerRootSceneGraph.setInteractionMode(mode);
  }

  @Override
  public InteractionMode getInteractionMode()
  {
    return m_examinerRootSceneGraph.getInteractionMode();
  }

  @Override
  public void addInteractionModeListener(InteractionModeListener listener)
  {
    m_examinerRootSceneGraph.addInteractionModeListener(listener);
  }

  @Override
  public void removeInteractionModeListener(InteractionModeListener listener)
  {
    m_examinerRootSceneGraph.removeInteractionModeListener(listener);
  }

  /**
   * @deprecated As of Open Inventor 10.5.0, use {@link #getSceneInteractor} instead.
   */
  @Deprecated
  @Override
  public SceneExaminer getRootSceneGraph()
  {
    return m_examinerRootSceneGraph;
  }

  @Override
  public SceneExaminer getSceneInteractor()
  {
    return m_examinerRootSceneGraph;
  }

  private void updateSeekCursor(boolean seek)
  {
    if ( seek )
      setCursor(SEEK_CURSOR);
    else
      updateInteractionCursor();
  }

  private void updateInteractionCursor()
  {
    if ( getInteractionMode() == InteractionMode.NAVIGATION )
      setCursor(VIEWING_CURSOR);
    else
      setCursor(Cursor.getDefaultCursor());
  }

  private static Cursor createCursor(String resourceName, String cursorName)
  {
    return Toolkit.getDefaultToolkit().createCustomCursor(
        new ImageIcon(RenderAreaExaminer.class.getResource(resourceName)).getImage(), new Point(16, 16), cursorName);
  }

}
