package com.openinventor.inventor.viewercomponents.awt.tools;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Locale;

import javax.swing.JCheckBox;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class SliderPanel extends JPanel
{

  public static class Listener implements ChangeListener
  {

    @Override
    public void stateChanged(ChangeEvent e)
    {}

    public void stateChanged(float value)
    {}

  }

  private int m_precision;
  private boolean m_realValues;

  private JSlider m_slider;
  private JFormattedTextField m_textField;
  private JCheckBox m_sliderBox;
  private JLabel m_infoText;
  private ArrayList<Listener> m_splList;

  private SliderPanel()
  {
    m_infoText = null;
    m_sliderBox = new JCheckBox();
    m_sliderBox.setSelected(true);

    m_splList = new ArrayList<Listener>();
  }

  public SliderPanel(int sliderMin, int sliderMax, int sliderInit)
  {
    this(sliderMin, sliderMax, sliderInit, true);
  }

  // Integer slider
  public SliderPanel(int sliderMin, int sliderMax, int sliderInit, boolean showCheckbox)
  {
    this();

    m_slider = new JSlider(sliderMin, sliderMax, sliderInit);
    // format field for int value
    m_textField = new JFormattedTextField(NumberFormat.getIntegerInstance(Locale.FRENCH));
    m_textField.setColumns(3);
    m_textField.setValue(Integer.valueOf(sliderInit));
    m_realValues = false;
    m_precision = 0;

    addListeners();

    if ( showCheckbox )
    {
      add(m_sliderBox);
    }
    add(m_slider);
    add(m_textField);
  }

  // Real slider
  public SliderPanel(float sliderMin, float sliderMax, float sliderInit, int precision)
  {
    this(sliderMin, sliderMax, sliderInit, precision, true);
  }

  public SliderPanel(float sliderMin, float sliderMax, float sliderInit, int precision, boolean showCheckbox)
  {
    this();

    m_precision = precision < 1 ? 1 : precision;
    m_slider = new JSlider((int) (sliderMin * Math.pow(10, m_precision)), (int) (sliderMax * Math.pow(10, m_precision)),
        (int) (sliderInit * Math.pow(10, m_precision)));
    // format field for float value
    NumberFormat float_format = NumberFormat.getNumberInstance();
    float_format.setMinimumFractionDigits(1);
    float_format.setMaximumFractionDigits(m_precision);
    m_textField = new JFormattedTextField(float_format);
    m_textField.setColumns(2 + m_precision);
    m_textField.setValue(Float.valueOf(sliderInit));
    m_realValues = true;

    addListeners();

    if ( showCheckbox )
      add(m_sliderBox);
    add(m_slider);
    add(m_textField);
  }

  @Override
  public void setEnabled(boolean enabled)
  {
    m_slider.setEnabled(enabled);
    m_textField.setEnabled(enabled);
    m_sliderBox.setEnabled(enabled);
    if ( m_infoText != null )
      m_infoText.setEnabled(enabled);
  }

  @Override
  public boolean isEnabled()
  {
    return m_slider.isEnabled();
  }

  public void setBoxSelected(boolean selected)
  {
    m_sliderBox.setSelected(selected);
  }

  public void addInfoText(String text)
  {
    m_infoText = new JLabel(text);
    add(m_infoText, 0);
  }

  public void setInfoTextSize(Dimension d)
  {
    if ( m_infoText != null )
      m_infoText.setPreferredSize(d);
  }

  public void setSliderBackground(Color background)
  {
    m_slider.setBackground(background);
  }

  public Color getSliderBackground()
  {
    return m_slider.getBackground();
  }

  public void setSliderSize(Dimension slider_dim)
  {
    m_slider.setPreferredSize(slider_dim);
  }

  public float getRelativeSliderValue()
  {
    if ( m_realValues )
      return (m_slider.getValue() - m_slider.getMinimum()) * (float) Math.pow(10, -m_precision);
    else
      return m_slider.getValue() - m_slider.getMinimum();
  }

  public float getSliderValue()
  {
    if ( m_realValues )
      return m_slider.getValue() * (float) Math.pow(10, -m_precision);
    else
      return m_slider.getValue();
  }

  public boolean isSliderBoxSelected()
  {
    return m_sliderBox.isSelected();
  }

  public void setSliderValue(float value)
  {
    boolean wasNotSelected = m_sliderBox.isSelected() == false;
    if ( wasNotSelected )
      m_sliderBox.setSelected(true);
    if ( m_realValues )
      m_slider.setValue((int) (value * Math.pow(10, m_precision)));
    else
      m_slider.setValue((int) value);
    if ( wasNotSelected )
      m_sliderBox.setSelected(false);
  }

  public void setSliderMax(float value)
  {
    if ( m_realValues )
      m_slider.setMaximum((int) (value * Math.pow(10, m_precision)));
    else
      m_slider.setMaximum((int) value);
  }

  public void setSliderMin(float value)
  {
    if ( m_realValues )
      m_slider.setMinimum((int) (value * Math.pow(10, m_precision)));
    else
      m_slider.setMinimum((int) value);
  }

  public void setTextFieldColumns(int nb_columns)
  {
    m_textField.setColumns(nb_columns);
  }

  public int getTextFieldColumns()
  {
    return m_textField.getColumns();
  }

  public void setMinorTickSpacing(double space)
  {
    if ( m_realValues )
      m_slider.setMajorTickSpacing((int) (space * Math.pow(10, m_precision)));
    else
      m_slider.setMajorTickSpacing((int) space);
    m_slider.setSnapToTicks(true);
  }

  public void addSliderPanelListener(Listener listener)
  {
    m_splList.add(listener);
  }

  public void removeSliderPanelListener(Listener listener)
  {
    m_splList.remove(listener);
  }

  public void addSliderMouseMotionListener(MouseMotionListener listener)
  {
    m_slider.addMouseMotionListener(listener);
  }

  public void addSliderMouseListener(MouseListener listener)
  {
    m_slider.addMouseListener(listener);
  }

  private void addListeners()
  {
    m_textField.addKeyListener(new FieldKeyListener());

    m_slider.addChangeListener(new ChangeListener()
    {
      @Override
      public void stateChanged(ChangeEvent e)
      {
        float value;
        if ( m_realValues )
        {
          value = m_slider.getValue() / ((float) Math.pow(10, m_precision));
          m_textField.setValue(Float.valueOf(value));
        }
        else
        {
          value = m_slider.getValue();
          m_textField.setValue(Integer.valueOf((int) value));
        }
        if ( m_sliderBox.isSelected() )
          for ( Listener listener : m_splList )
            listener.stateChanged(value);
      }
    });

    m_slider.addMouseListener(new MouseAdapter()
    {
      @Override
      public void mouseReleased(MouseEvent e)
      {
        if ( !m_slider.isEnabled() )
          return;

        float value;
        if ( m_realValues )
          value = m_slider.getValue() / ((float) Math.pow(10, m_precision));
        else
          value = m_slider.getValue();

        for ( Listener listener : m_splList )
          listener.stateChanged(value);
      }
    });
  }

  class FieldKeyListener extends KeyAdapter
  {
    @Override
    public void keyPressed(KeyEvent ke)
    {
      if ( ke.getKeyCode() == KeyEvent.VK_ENTER )
      {
        try
        {
          // check input value
          m_textField.commitEdit();

          Object new_obj = m_textField.getValue();
          int new_value;
          if ( m_realValues )
            new_value = (int) (((Number) new_obj).floatValue() * Math.pow(10, m_precision));
          else
            new_value = ((Number) new_obj).intValue();

          if ( new_value > m_slider.getMaximum() )
            m_slider.setMaximum(new_value);

          if ( !m_sliderBox.isSelected() )
          {
            m_sliderBox.setSelected(true);
            m_slider.setValue(new_value);
            m_sliderBox.setSelected(false);
          }
          else
            m_slider.setValue(new_value);
        }
        catch (ParseException e)
        {
          // edit old value
          m_textField.setValue(m_textField.getValue());
        }
      }
    }
  }
}
