package com.openinventor.inventor.viewercomponents.swt.glcanvas.viewers;

import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLProfile;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionModeListener;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.NavigationMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.inventor.viewercomponents.swt.glcanvas.renderareas.RenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.swt.glcanvas.renderareas.RenderAreaInteractive.ClippingPlanesAdjustMode;
import com.openinventor.inventor.viewercomponents.swt.tools.PreferencesDialog;
import com.openinventor.inventor.viewercomponents.swt.tools.PreferencesDialog.GeneralPreferencesListener;
import com.openinventor.inventor.viewercomponents.swt.tools.SWTResourceManager;

public class ViewerExaminer extends Composite
{
  private static Cursor seekCursor;
  private static Cursor viewingCursor;
  private static ToolItem seekItem;

  private RenderAreaExaminer m_renderArea;
  private SoSeparator m_sceneGraph;
  private SoGroup m_objectGroup;

  public ViewerExaminer(Shell shell)
  {
    super(shell, SWT.NONE);
    setLayout(new FillLayout(SWT.VERTICAL));
    setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 2));

    seekCursor = new Cursor(shell.getDisplay(),
        SWTResourceManager
            .getImage(ViewerExaminer.class, "/com/openinventor/inventor/viewercomponents/icons/CursorSeek.png")
            .getImageData(),
        16, 16);
    viewingCursor = new Cursor(shell.getDisplay(),
        SWTResourceManager
            .getImage(ViewerExaminer.class, "/com/openinventor/inventor/viewercomponents/icons/CursorCurvedHand.png")
            .getImageData(),
        16, 16);

    GLCapabilities glCapabilities = new GLCapabilities(GLProfile.getDefault());
    glCapabilities.setStereo(true);
    m_renderArea = new RenderAreaExaminer(this, SWT.NONE, glCapabilities);

    m_objectGroup = new SoGroup();
    m_sceneGraph = new SoSeparator();
    {
      m_sceneGraph.addChild(new SoGradientBackground());
      m_sceneGraph.addChild(m_objectGroup);
    }
    m_renderArea.setSceneGraph(m_sceneGraph);

    addInteractionModeListener(new InteractionModeListener()
    {
      @Override
      public void seekModeChanged(boolean onOrOff)
      {
        seekItem.setSelection(onOrOff);
        updateUI(getInteractionMode());
      }

      @Override
      public void interactionModeChanged(InteractionMode newMode)
      {
        updateUI(newMode);
      }
    });

    buildToolbar(shell);

    // Preferences
    ToolBar toolBar = new ToolBar(shell, SWT.FLAT | SWT.RIGHT);
    toolBar.setLayoutData(new GridData(SWT.CENTER, SWT.BOTTOM, false, false, 1, 1));

    ToolItem preferencesItem = new ToolItem(toolBar, SWT.NONE);
    preferencesItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        openPreferences(shell);
      }
    });
    preferencesItem.setToolTipText("Preferences...");
    preferencesItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/preferences.png"));

    // open chair.iv by default
    loadSceneGraph("$OIVJHOME/data/models/chair.iv");

    updateUI(getInteractionMode());
  }

  public void loadSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      return;
    }
    m_objectGroup.removeAllChildren();
    m_objectGroup.addChild(node);
    viewAll();

    input.closeFile();
  }

  private void updateUI(InteractionMode activeMode)
  {
    switch ( activeMode )
    {
    case NAVIGATION :
      seekItem.setEnabled(true);
      if ( seekItem.getSelection() )
        setCursor(seekCursor);
      else
        setCursor(viewingCursor);
      break;
    case SELECTION :
      seekItem.setSelection(false);
      seekItem.setEnabled(false);
      setCursor(Display.getDefault().getSystemCursor(SWT.CURSOR_ARROW));
      break;
    default:
      break;
    }
  }

  private void buildToolbar(Shell shell)
  {

    ToolBar toolBar = new ToolBar(shell, SWT.FLAT | SWT.RIGHT | SWT.VERTICAL);
    toolBar.setLayoutData(new GridData(SWT.CENTER, SWT.TOP, false, false, 1, 1));

    ToolItem viewAllItem = new ToolItem(toolBar, SWT.NONE);
    viewAllItem.setToolTipText("View All");
    viewAllItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        viewAll();
      }
    });
    viewAllItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerViewAllTech.png"));

    ToolItem xViewItem = new ToolItem(toolBar, SWT.NONE);
    xViewItem.setToolTipText("YZ view");
    xViewItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        viewAxis(new SbVec3f(-1, 0, 0), new SbVec3f(0, 0, 1));
        viewAll();
      }
    });
    xViewItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerViewYZTech.png"));

    ToolItem yViewItem = new ToolItem(toolBar, SWT.NONE);
    yViewItem.setToolTipText("XZ view");
    yViewItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        viewAxis(new SbVec3f(0, -1, 0), new SbVec3f(1, 0, 0));
        viewAll();
      }
    });
    yViewItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerViewXZTech.png"));

    ToolItem zViewItem = new ToolItem(toolBar, SWT.NONE);
    zViewItem.setToolTipText("XY view");
    zViewItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        viewAxis(new SbVec3f(0, 0, -1), new SbVec3f(0, 1, 0));
        viewAll();
      }
    });
    zViewItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerViewXYTech.png"));

    ToolItem restoreItem = new ToolItem(toolBar, SWT.NONE);
    restoreItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerHome.png"));
    restoreItem.setToolTipText("Restore Camera");
    restoreItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        restoreCamera();
      }
    });

    ToolItem saveItem = new ToolItem(toolBar, SWT.NONE);
    saveItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerSetHome.png"));
    saveItem.setToolTipText("Save Camera");
    saveItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        saveCamera();
      }
    });

    ToolItem orthoItem = new ToolItem(toolBar, SWT.NONE);
    orthoItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerOrtho.png"));
    orthoItem.setToolTipText("Switch to orthographic camera");
    orthoItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent arg0)
      {
        setCameraType(CameraMode.ORTHOGRAPHIC);
        // disable stereo
        PreferencesDialog.getInstance(shell).enableStereoPreferences(false);
      }
    });

    ToolItem perspItem = new ToolItem(toolBar, SWT.NONE);
    perspItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerPerspective.png"));
    perspItem.setToolTipText("Switch to perspective camera");
    perspItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent arg0)
      {
        setCameraType(CameraMode.PERSPECTIVE);
        if ( isStereoSupported() && isRawStereoAvailable() )
        {
          // enable stereo
          PreferencesDialog.getInstance(shell).enableStereoPreferences(true);
        }
      }
    });

    seekItem = new ToolItem(toolBar, SWT.CHECK);
    seekItem.setImage(SWTResourceManager.getImage(ViewerExaminer.class,
        "/com/openinventor/inventor/viewercomponents/icons/ViewerSeek.png"));
    seekItem.setToolTipText("Seek");
    seekItem.addSelectionListener(new SelectionAdapter()
    {
      @Override
      public void widgetSelected(SelectionEvent e)
      {
        setSeekMode(seekItem.getSelection());
      }
    });
  }

  private void openPreferences(final Shell shell)
  {
    PreferencesDialog dialog = PreferencesDialog.getInstance(shell);

    if ( !isStereoSupported() || !isRawStereoAvailable() )
    {
      // disable stereo
      PreferencesDialog.getInstance(shell).enableStereoPreferences(false);
    }
    else if ( dialog.getStereoListener() == null )
    {
      dialog.setStereoListener(new PreferencesDialog.StereoListener()
      {

        @Override
        public void offsetValueChanged(float value)
        {
          setStereoCameraOffset(value);
        }

        @Override
        public void balanceValueChanged(float value)
        {
          setStereoCameraBalance(value);
        }

        @Override
        public void activateRawStereo(boolean activated)
        {
          activateStereo(activated);
        }
      });
    }

    if ( dialog.getGeneralPreferencesListener() == null )
    {
      dialog.setNavigationMode(getNavigationMode());
      dialog.setClippingPlanesAdjustMode(getClippingPlanesAdjustMode());
      dialog.setGeneralPreferencesListener(new GeneralPreferencesListener()
      {
        @Override
        public void navigationModeChanged(NavigationMode newMode)
        {
          setNavigationMode(newMode);
        }

        @Override
        public void clippingModeChanged(ClippingPlanesAdjustMode newMode)
        {
          setClippingPlanesAdjustMode(newMode);
        }
      });
    }
    dialog.open();
  }

  public void viewAll()
  {
    Point size = m_renderArea.getSize();
    m_renderArea.viewAll(new SbViewportRegion((short) size.x, (short) size.y));
  }

  public RenderAreaExaminer getRenderArea()
  {
    return m_renderArea;
  }

  public void saveCamera()
  {
    m_renderArea.saveCamera();
  }

  public void restoreCamera()
  {
    m_renderArea.restoreCamera();
  }

  public void setClippingPlanesAdjustMode(ClippingPlanesAdjustMode mode)
  {
    m_renderArea.setClippingPlanesAdjustMode(mode);
  }

  public ClippingPlanesAdjustMode getClippingPlanesAdjustMode()
  {
    return m_renderArea.getClippingPlanesAdjustMode();
  }

  public void setSeekMode(boolean onOrOff)
  {
    m_renderArea.setSeekMode(onOrOff);
  }

  public void setNavigationMode(NavigationMode mode)
  {
    m_renderArea.setNavigationMode(mode);
  }

  public NavigationMode getNavigationMode()
  {
    return m_renderArea.getNavigationMode();
  }

  public InteractionMode getInteractionMode()
  {
    return m_renderArea.getInteractionMode();
  }

  public void viewAxis(SbVec3f direction, SbVec3f up)
  {
    m_renderArea.viewAxis(direction, up);
  }

  public void addInteractionModeListener(InteractionModeListener listener)
  {
    m_renderArea.addInteractionModeListener(listener);
  }

  public void setCameraType(CameraMode mode)
  {
    m_renderArea.setCameraType(mode);
  }

  public boolean isRawStereoAvailable()
  {
    return m_renderArea.isRawStereoAvailable();
  }

  public boolean isStereoSupported()
  {
    return m_renderArea.isStereoSupported();
  }

  public void setCursor(Cursor cursor)
  {
    m_renderArea.setCursor(cursor);
  }

  public void activateStereo(boolean activated)
  {
    try
    {
      m_renderArea.activateStereo(activated);
    }
    catch (UnsupportedOperationException e)
    {
      e.printStackTrace();
    }
  }

  public void setStereoCameraBalance(float value)
  {
    m_renderArea.setStereoCameraBalance(value);
  }

  public void setStereoCameraOffset(float value)
  {
    m_renderArea.setStereoCameraOffset(value);
  }
}
