///////////////////////////////////////////////////////////////////////////////
//
// This code is part of the Open Inventor Medical Edition utility library.
//
// Open Inventor customers may use this source code to create or enhance Open
// Inventor-based applications.
//
// The medical utility classes are provided as a jar named com.openinventor.medical.jar,
// that can be used directly in an Open Inventor application. The classes in this jar
// are documented and supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
///////////////////////////////////////////////////////////////////////////////
package com.openinventor.medical.nodes;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.draggers.SoTranslate2Dragger;
import com.openinventor.inventor.fields.SoField;
import com.openinventor.inventor.fields.SoSFColor;
import com.openinventor.inventor.fields.SoSFFloat;
import com.openinventor.inventor.fields.SoSFNode;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoFieldSensor;

//@formatter:off
/**
 * <strong>(Preview Feature) </strong>Shape node to display a magnified view of a region of the screen.
 *
 * <p style="background-color: #c7c7c7;">Preview Feature means this class is fully supported and can be used in Open Inventor applications.
 * Being tagged as a Preview Feature just means that the implementation is still
 * subject to API changes and adjustments based on feedback from early adopters.
 * Please be also aware that source compatibility might be broken regardless of
 * the Open Inventor compatibility changes policy due to our commitment to bring
 * needed changes to be sure the specifications of this Preview Feature match
 * the expectations of our customers.
 * </p>
 *
 * This node displays a circular magnified (zoomed in) view of the
 * screen region under the center of the circle. The magnifier is
 * an SoTranslate2Dragger so the user can click and drag to move it
 * around the screen.
 * <p>
 * <b>Limitations</b>:
 * <ul>
 * <li>Can only be used in a "2D" viewing environment using an
 *     SoOrthographicCamera.<br>
 *     For example viewing a single slice of a medical volume.
 *     </li>
 * <p>
 * <li>Can only be used with the default camera orientation
 *     (looking toward -Z). </li>
 * </ul>
 * <p>
 * <b>File format/default:</b>
 * <p>
 * Magnifier {
 * <ul><table>
 * <tr><td> sceneToMagnify    </td><td> NULL       </td></tr>
 * <tr><td> magnifierColor    </td><td> 0 0.6 0    </td></tr>
 * <tr><td> magnifierFactor   </td><td> 50         </td></tr>
 * </table></ul> }
 *
 * @see DicomInfo
 * @see Gnomon
 * @see Ruler
 *
 */
//@formatter:on
public class Magnifier extends SoTranslate2Dragger
{
  /**
   * Specifies the root node of the scene that will be magnified. Default is
   * null.
   */
  public SoSFNode sceneToMagnify;

  /**
   * Specifies the color of the border around the magnifier. Default is 0,0.6,0
   * (green).
   */
  public SoSFColor magnifierColor;

  /**
   * Specifies the magnification factor. Default is 50. Technically speaking
   * this field specifies the value to set in the height field of the
   * magnifier's camera.
   */
  public SoSFFloat magnifierFactor;

  // Internal state
  private SoSeparator _magnifierSGSep;
  private SoOrthographicCamera _magnifierCam;
  private SoMaterial _magnifierBorderMaterial;
  private SoSeparator _sceneToMagnifyParent;
  private SoFieldSensor _sceneFieldSensor;

  /**
   * Constructor
   */
  public Magnifier()
  {
    sceneToMagnify = new SoSFNode(this, "sceneToMagnify", SoField.FieldTypes.EXPOSED_FIELD, null);
    magnifierColor =
        new SoSFColor(this, "magnifierColor", SoField.FieldTypes.EXPOSED_FIELD, new SbColor(0.0f, 0.6f, 0.0f));
    magnifierFactor = new SoSFFloat(this, "magnifierFactor", SoField.FieldTypes.EXPOSED_FIELD, 50.0f);

    buildSceneGraph();
  }

  @Override
  public RenderModes getRenderEngineMode()
  {
    return RenderModes.OIV_OPENINVENTOR_RENDERING;
  }

  /**
   * Builds the scene graph that represents a magnifier object.
   */
  protected void buildSceneGraph()
  {
    // Name main node
    setName("Magnifier Dragger");

    // --------------- Sub-scenegraph used as dragger geometry ----------------
    SoSeparator magnifierSep = new SoSeparator();

    SoLightModel lightModel = new SoLightModel();
    lightModel.model.setValue(SoLightModel.Models.BASE_COLOR);
    magnifierSep.addChild(lightModel);

    _magnifierBorderMaterial = new SoMaterial();
    magnifierSep.addChild(_magnifierBorderMaterial);

    SoComplexity cylinderComplexity = new SoComplexity();
    cylinderComplexity.value.setValue(1.0f);
    magnifierSep.addChild(cylinderComplexity);

    SoRotationXYZ cylinderRotation = new SoRotationXYZ();
    cylinderRotation.angle.setValue(1.5f);
    magnifierSep.addChild(cylinderRotation);

    // Border for the magnifier
    SoCylinder magnifierBorderSupport = new SoCylinder();
    magnifierBorderSupport.height.setValue(0);
    magnifierBorderSupport.radius.setValue(40.5f);
    magnifierBorderSupport.parts.setValue(SoCylinder.PartType.TOP);
    magnifierSep.addChild(magnifierBorderSupport);

    // Texture mapped on the cylinder issue from magnifierRenderToTexture.
    SoTexture2 magnifierTexture = new SoTexture2();
    magnifierTexture.model.setValue(SoTexture2.Models.REPLACE);
    magnifierSep.addChild(magnifierTexture);

    // The center of this cylinder (circle) is the center of the magnifier
    // camera (magnifierCam).
    SoCylinder magnifier = new SoCylinder();
    magnifier.height.setValue(1);
    magnifier.radius.setValue(40);
    magnifier.parts.setValue(SoCylinder.PartType.TOP);
    magnifierSep.addChild(magnifier);

    // Replace default dragger parts with our geometry
    setPart("translatorActive", magnifierSep);
    setPart("translator", magnifierSep);
    setPart("xAxisFeedback", new SoSeparator());
    setPart("yAxisFeedback", new SoSeparator());

    // ---------------- Sub-scenegraph used by "magnifierRenderToTexture"
    // ---------------
    // This scenegraph is rendered into a texture when dragger position changes.
    _magnifierSGSep = new SoSeparator();
    _magnifierSGSep.addChild(new SoDirectionalLight());

    _magnifierCam = new SoOrthographicCamera();
    _magnifierCam.position.connectFrom(translation);
    // _magnifierCam.farDistance.Value = 2000;
    _magnifierCam.height.setValue(magnifierFactor.getValue());
    _magnifierCam.viewAll(_magnifierSGSep, new SbViewportRegion((short) 100, (short) 100));
    _magnifierSGSep.addChild(_magnifierCam);

    // We will replace the children of this node with the scene to magnify.
    _sceneToMagnifyParent = new SoSeparator();
    if ( sceneToMagnify.getValue() != null )
      _sceneToMagnifyParent.addChild(sceneToMagnify.getValue());
    _magnifierSGSep.addChild(_sceneToMagnifyParent);

    // Set scene to be rendered into texture and attach to texture node.
    SoRenderToTextureProperty magnifierRenderToTexture = new SoRenderToTextureProperty();
    magnifierRenderToTexture.node.set1Value(0, _magnifierSGSep);
    magnifierRenderToTexture.size.setValue(new SbVec2s((short) 512, (short) 512));
    magnifierTexture.renderToTextureProperty.setValue(magnifierRenderToTexture);

    // ---------- Manage field modification ----------
    // These internal fields will be updated automatically when the external
    // fields are changed.
    _magnifierBorderMaterial.diffuseColor.connectFrom(magnifierColor);
    _magnifierCam.height.connectFrom(magnifierFactor);

    // We need to do a little more work when the sceneToMagnify field changes.
    _sceneFieldSensor = new SoFieldSensor();
    _sceneFieldSensor.setTask(new SceneFieldChangedCB());
    _sceneFieldSensor.setPriority(0);
    _sceneFieldSensor.attach(sceneToMagnify);
  }

  private class SceneFieldChangedCB implements Runnable
  {
    @Override
    public void run()
    {
      _sceneToMagnifyParent.removeAllChildren();
      _sceneToMagnifyParent.addChild(sceneToMagnify.getValue());

      // Update internal camera to view sceneToMagnify.
      // Don't forget to reset camera position and height after viewAll changes
      // them...
      _magnifierCam.viewAll(_magnifierSGSep, new SbViewportRegion((short) 100, (short) 100));
      _magnifierCam.position.setValue(translation.getValue());
      _magnifierCam.height.setValue(magnifierFactor.getValue());
    }
  }
}
