//Reference: www2.ati.com/misc/samples/dx9/FixedFuncShader.pdf

#define NUM_LIGHTS               8

#define LIGHT_TYPE_NONE          0
#define LIGHT_TYPE_POINT         1
#define LIGHT_TYPE_SPOT          2
#define LIGHT_TYPE_DIRECTIONAL   3
#define LIGHT_NUM_TYPES          4

#define FOG_MODE_NONE            0
#define FOG_MODE_EXP             1
#define FOG_MODE_EXP2            2
#define FOG_MODE_LINEAR          3

#define TEXGEN_TYPE_NONE             0
#define TEXGEN_TYPE_OBJECT_PLANE     1
#define TEXGEN_TYPE_REFLECTION_MAP   2
#define TEXGEN_TYPE_SPHERE_MAP       3
#define TEXGEN_TYPE_NORMAL_MAP       4
#define TEXGEN_TYPE_EYE_PLANE        5

#define PI 3.14159265358979312

void OivSetupShadowVertex();

uniform bool OivLighting = true;
uniform bool OivFogFragmentDepth = false;

//fog
uniform int OivFogMode = FOG_MODE_NONE;

uniform int OivTexGenType = TEXGEN_TYPE_NONE;

uniform int OivLights[NUM_LIGHTS];

struct ColorPair
{
  vec4 color;
  vec4 colorSpec;
};

/*****************************************************************************/
ColorPair computeDirLight(vec3 N, vec3 V, int i, gl_LightProducts lightProd,
                          gl_MaterialParameters matParam)
{
  ColorPair res;
  vec3 L = normalize(gl_LightSource[i].position.xyz);
  float NdotL = dot(N, L);
  res.color = lightProd.ambient;
  res.colorSpec = vec4(0.);
  if(NdotL > 0.)
  {
    //compute diffuse color
    res.color += NdotL * lightProd.diffuse;

    //add specular component
    vec3 H = normalize(L + V);
    res.colorSpec = pow(max(0., dot(H, N)), matParam.shininess) * lightProd.specular;
  }
  return res;
}

/*****************************************************************************/
ColorPair computePointLight(vec3 N, vec3 V, vec3 P, int i, gl_LightProducts lightProd,
                            gl_MaterialParameters matParam)
{
  ColorPair res;

  vec3 L = gl_LightSource[i].position.xyz-P;
  float vpLen = length(L);
  L /= vpLen;

  float NdotL = dot(N, L);
  res.color = lightProd.ambient;
  res.colorSpec = vec4(0.);
  float att = 1.;
  if(NdotL >= 0.)
  {
    //compute diffuse color
    res.color += NdotL * lightProd.diffuse;

    //add specular component
    vec3 H = normalize(L + V);   //half vector
    res.colorSpec = pow(max(0., dot(H, N)), matParam.shininess) * lightProd.specular;
  }
  return res;
}

/*****************************************************************************/
ColorPair computeSpotLight(vec3 N, vec3 V, vec3 P, int i, gl_LightProducts lightProd,
                      gl_MaterialParameters matParam)
{
   vec3 L = normalize(gl_LightSource[i].position.xyz-P);
   float vpLen = length(L);
   L /= vpLen;

   ColorPair res;
   float NdotL = dot(N, L);
   res.color = lightProd.ambient;
   res.colorSpec = vec4(0.);
   if(NdotL >= 0.)
   {
     //compute diffuse color
     res.color += NdotL *  lightProd.diffuse;

     //add specular component
     vec3 H = normalize(L + V);   //half vector
     res.colorSpec = pow(max(0., dot(H, N)),  matParam.shininess) *  lightProd.specular;

     //spot cone computation
     float spot = 1.;
     vec3 L2 = normalize(gl_LightSource[i].spotDirection);
     float rho = dot(-L, L2);
     if ( gl_LightSource[i].spotCutoff != 180. )
      {
        if ( rho >= cos(gl_LightSource[i].spotCutoff*PI/180.) )
          spot *= pow(rho, gl_LightSource[i].spotExponent);
        else
          spot = 0.;
      }

     res.color *= spot;
     res.colorSpec *= spot;
   }
   return res;
}

/*****************************************************************************/
ColorPair computeLightColor(vec3 N, vec3 V, vec3 P, gl_LightModelProducts lightModelProd,
                            gl_MaterialParameters frontMat)
{
  ColorPair p;
  p.color = lightModelProd.sceneColor;
  p.colorSpec = vec4(0.);

  //lights
  for(int i = 0; i < NUM_LIGHTS; i++)
  {
    ColorPair res;
    if(OivLights[i]==LIGHT_TYPE_DIRECTIONAL)
      res = computeDirLight(N, V, i, gl_FrontLightProduct[i], frontMat);
    else if(OivLights[i]==LIGHT_TYPE_POINT)
      res = computePointLight(N, V, P, i, gl_FrontLightProduct[i], frontMat);
    else if(OivLights[i]==LIGHT_TYPE_SPOT)
      res = computeSpotLight(N, V, P, i, gl_FrontLightProduct[i], frontMat);
    else
      continue;
    p.color += res.color;
    p.colorSpec += res.colorSpec;
  }
  //saturate
  p.color = min(vec4(1.), p.color);
  p.colorSpec = min(vec4(1.), p.colorSpec);

  return p;
}

/*****************************************************************************/
float computeFog(vec3 P)
{
  float d;
  float fog = 0.;

  if(!OivFogFragmentDepth)
    d = length(P);
  else
    d = P.z;

  if (OivFogMode == FOG_MODE_NONE)
    fog = 1.;
  else if(OivFogMode == FOG_MODE_EXP)
    fog = 1./exp(d * gl_Fog.density);
  else if(OivFogMode == FOG_MODE_EXP2)
    fog = 1./exp(pow(d * gl_Fog.density, 2.));
  else if ( OivFogMode == FOG_MODE_LINEAR)
    fog = clamp((gl_Fog.end - d)*gl_Fog.scale, 0., 1.);

  return fog;
}

/*****************************************************************************/
vec4 computeTexCoords(int i, vec4 V, vec3 N, vec4 P, vec4 inCoords)
{
  vec4 tcoords = vec4(0.);
  if ( OivTexGenType == TEXGEN_TYPE_NONE )
    tcoords = inCoords;
  else
  {
    //Inventor only allows this kind of texgen
    tcoords.x = dot(gl_ObjectPlaneS[i], gl_Vertex);
    tcoords.y = dot(gl_ObjectPlaneT[i], gl_Vertex);
    tcoords.z = dot(gl_ObjectPlaneR[i], gl_Vertex);
    tcoords.w = dot(gl_ObjectPlaneQ[i], gl_Vertex);
  }

  return tcoords;
}

/*****************************************************************************/
void main ()
{
  OivSetupShadowVertex();
  vec4 P = gl_ModelViewMatrix*gl_Vertex;
  vec3 N = normalize(gl_NormalMatrix*gl_Normal);
  vec4 V = -normalize(P);

  ColorPair frontColor = ColorPair(gl_FrontMaterial.diffuse*gl_Color, vec4(0.));
  //light computational
  if ( OivLighting )
    frontColor = computeLightColor(N, V.xyz, P.xyz, gl_FrontLightModelProduct, gl_FrontMaterial);

  vec4 finalColor;
  finalColor.xyz = frontColor.color.xyz+frontColor.colorSpec.xyz;
  finalColor.w = gl_FrontMaterial.diffuse.w*gl_Color.w;
  finalColor = min(vec4(1.), finalColor);

  gl_Position = gl_ModelViewProjectionMatrix*gl_Vertex;
  gl_FrontColor = finalColor;

  gl_TexCoord[0] = computeTexCoords(0, V, N, P, gl_MultiTexCoord0);
//   gl_TexCoord[1] = computeTexCoords(1, V, N, P, gl_MultiTexCoord1);
//   gl_TexCoord[2] = computeTexCoords(2, V, N, P, gl_MultiTexCoord2);
//   gl_TexCoord[3] = computeTexCoords(3, V, N, P, gl_MultiTexCoord3);

  if ( OivFogMode != FOG_MODE_NONE )
    gl_FogFragCoord = computeFog(P.xyz);
}
