package imageviz.gettingstarted.createImage3D;

import java.awt.BorderLayout;
import java.awt.Component;
import java.nio.ByteBuffer;

import com.openinventor.imageviz.SbImageDataType;
import com.openinventor.imageviz.nodes.images.SoMemoryDataAdapter;
import com.openinventor.imageviz.nodes.images.SoVRImageDataReader;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbVec4i32;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private IViewerExaminer myRenderArea;

  @Override
  public void start()
  {
    myRenderArea = ViewerComponentsFactory.createViewerExaminer();

    SoSeparator root = new SoSeparator();

    // 1- Create a Memory Data adapter containing a 256x256x32 image
    // 1.1- Instantiate a CPU buffer object
    int width = 256, height = 256, depth = 32;
    SbVec4i32 imageSize = new SbVec4i32(width, height, depth, 1);
    SoCpuBufferObject imageBuffer = new SoCpuBufferObject();
    imageBuffer.setSize(width * height * depth);

    // 1.2- Define a synthetic square in this buffer object
    byte[] imageDataArray = new byte[(int) imageBuffer.getSize()];
    int borderX = width / 4; // Vertical border around the square
    int borderY = height / 4; // Horizontal border around the square
    // Loop on image rows
    for ( int k = 0; k < depth; ++k )
    {
      // Loop on image columns
      for ( int i = 0; i < width; ++i )
      {
        // Loop on image columns
        for ( int j = 0; j < height; ++j )
        {
          if ( (i >= borderX) && (i <= width - borderX) && (j >= borderY) && (j <= height - borderY) )
            imageDataArray[k * height * width + i * width + j] = (byte) 200; // Value
                                                                             // inside
                                                                             // the
                                                                             // stack
                                                                             // of
                                                                             // squares
          else
            imageDataArray[k * height * width + i * width + j] = (byte) 128; // Background
                                                                             // value
        }
      }
    }

    ByteBuffer imageData = imageBuffer.map(SoCpuBufferObject.AccessModes.SET);
    imageData.put(imageDataArray);
    imageBuffer.unmap();

    // 1.3- Instantiate a memory image adapter containing the data of the buffer
    // object
    SoMemoryDataAdapter memoryImage = SoMemoryDataAdapter.createMemoryDataAdapter(imageSize,
        new SbImageDataType(1, new SbDataType(SbDataType.DataTypes.UNSIGNED_BYTE)),
        SoMemoryDataAdapter.LayerOrganizations.CONTIGUOUS_PER_PLANE, imageBuffer);
    // Now we can use this Image Adapter as input of an ImageViz engine.

    // 2- Display the created image
    // 2- We just show here how to display it in a volume data.
    // 2.1- We need a reader to access data so get it and give it the
    // 2.1- pointer to our imageAdapter;
    SoVRImageDataReader imageReader = new SoVRImageDataReader();
    imageReader.imageData.setValue(memoryImage);

    // 2.2- Get a new SoVolumeData so we can load image data to do some
    // rendering.
    SoVolumeData volumeData = new SoVolumeData();
    volumeData.setReader(imageReader, true);

    // 2.3- Build a simple scenegraph
    // 2.3.1- Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction transFunc = new SoTransferFunction();
    transFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.SEISMIC);

    // 2.3.2- Node in charge of drawing the volume
    SoVolumeRender volRender = new SoVolumeRender();

    // 2.3.3- Assemble the scene graph
    // 2.3.3- Note: SoOrthoSlice must appear after the SoVolumeData node.
    root.addChild(volumeData);
    root.addChild(transFunc);
    root.addChild(volRender);

    myRenderArea.setSceneGraph(root);
    myRenderArea.viewAll();

    final Component component = myRenderArea.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myRenderArea.dispose();
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("ImageViz - Create 3D Image");
  }
}
