package imageviz.workflow.registration;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.imageviz.SbImageDataAdapterHelper;
import com.openinventor.imageviz.engines.SoImageVizEngine;
import com.openinventor.imageviz.engines.geometryandmatching.registration.SoImageRegistrationTransform;
import com.openinventor.imageviz.nodes.images.SoImageDataAdapter;
import com.openinventor.imageviz.nodes.images.SoVRImageDataReader;
import com.openinventor.inventor.SbEventListener;
import com.openinventor.inventor.SbMatrix;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoKeyboardEvent.Keys;
import com.openinventor.inventor.manips.SoTransformerManip;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoOneShotSensor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.draggers.SoOrthoSliceDragger;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  SoImageRegistrationTransform m_registration;
  String m_infoMessage;
  SoText2 m_infoText;
  SoImageDataAdapter m_inputModel;
  SoImageDataAdapter m_inputReference;
  SoSeparator m_root;
  SoTransformerManip m_manipulator;
  SoSwitch m_swithDraggerModel;
  SoSwitch m_swithDraggerRef;
  SoOrthoSlice m_modelImageOrthoSlice;
  SoVolumeRender m_modelVolumeRender;
  SoVolumeRender m_refVolumeRender;
  SoOrthoSlice m_referenceImageOrthoSlice;
  SoTransform m_currTransform;
  long[] m_referenceMinMax;
  long[] m_modelMinMax;
  ASyncText m_asyncText;
  ASyncTransformation m_asyncTransformation;
  int m_priority;

  static IViewerExaminer myRenderArea;

  private final String m_filenameModel = "$OIVJHOME/data/imageviz/CT-data.am";
  private final String m_filenameReference = "$OIVJHOME/data/imageviz/MRT-data.am";
  private final int m_maxMemInMemory = 10000; // in Mb

  @Override
  public void start()
  {

    myRenderArea = ViewerComponentsFactory.createViewerExaminer();

    m_registration = new SoImageRegistrationTransform();
    m_priority = 0;

    // init
    SoPreferences.setValue("LDM_USE_IN_MEM_COMPRESSION", "0");
    SoPreferences.setValue("LDM_USE_BUILDTILE_BY_SLICE", "1");

    // to be notified of events
    m_registration.onProgressRegistration.addEventListener(new RegistrationEventListener());
    m_registration.onBegin.addEventListener(new RegistrationStartListener());
    m_registration.onEnd.addEventListener(new RegistrationEndListener());

    m_infoMessage = "Press r to start the registration ; p to edit the transformation";

    // create the root separator
    m_root = new SoSeparator();

    // create a handle box manipulator
    m_manipulator = new SoTransformerManip();

    // add an event callback
    SoEventCallback eventNode = new SoEventCallback();
    eventNode.addEventCallback(SoKeyboardEvent.class, new ProcessKeyEvents());
    m_root.addChild(eventNode);

    // create a gradient background node
    SoGradientBackground gradientBackgroundNode = new SoGradientBackground();
    gradientBackgroundNode.color0.setValue(0.376f, 0.392f, 0.438f);
    gradientBackgroundNode.color1.setValue(0.204f, 0.223f, 0.278f);
    m_root.addChild(gradientBackgroundNode);

    // loads the images in-memory
    m_inputModel = SbImageDataAdapterHelper.getAppropriateAdapter(m_filenameModel, m_maxMemInMemory);

    m_inputReference = SbImageDataAdapterHelper.getAppropriateAdapter(m_filenameReference, m_maxMemInMemory);
    SoSeparator rootVolumeRendering = new SoSeparator();

    m_root.addChild(rootVolumeRendering);

    // separator for the reference image
    SoSeparator referenceImageSeparator = new SoSeparator();
    rootVolumeRendering.addChild(referenceImageSeparator);

    // to take into consideration the orientation field
    referenceImageSeparator.addChild(createRotation(m_inputReference));

    // create an in-memory reader
    SoVRImageDataReader referenceImageReader = new SoVRImageDataReader();
    referenceImageReader.imageData.setValue(m_inputReference);

    // Node to hold the volume data
    SoVolumeData referenceImageVolumeData = new SoVolumeData();
    referenceImageVolumeData.setName("OrthoSliceInput");
    referenceImageVolumeData.ldmResourceParameters.getValue().tileDimension.setValue(128, 128, 128);
    referenceImageVolumeData.ldmResourceParameters.getValue().fixedResolution.setValue(true);
    referenceImageVolumeData.ldmResourceParameters.getValue().resolution.setValue(0);
    referenceImageVolumeData.texturePrecision.setValue((short) 16);
    referenceImageVolumeData.setReader(referenceImageReader, true);
    referenceImageSeparator.addChild(referenceImageVolumeData);

    // compute the data min - max
    m_referenceMinMax = referenceImageVolumeData.getMinMax();

    // Node to remap data range to full range of data
    SoDataRange referenceImageDataRange = new SoDataRange();
    referenceImageDataRange.min.setValue(m_referenceMinMax[0]);
    referenceImageDataRange.max.setValue(m_referenceMinMax[1]);

    referenceImageSeparator.addChild(referenceImageDataRange);
    // Node for colormap
    SoTransferFunction referenceImageTransferFunction = new SoTransferFunction();
    referenceImageTransferFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);
    referenceImageTransferFunction.minValue.setValue(0);
    referenceImageTransferFunction.maxValue.setValue(255);
    referenceImageSeparator.addChild(referenceImageTransferFunction);

    SoVolumeRenderingQuality pVRVolQuality = new SoVolumeRenderingQuality();
    pVRVolQuality.forVolumeOnly.setValue(true);
    pVRVolQuality.lighting.setValue(false);
    pVRVolQuality.boundaryOpacity.setValue(true);
    pVRVolQuality.boundaryOpacityIntensity.setValue(3.9f);
    pVRVolQuality.boundaryOpacityThreshold.setValue(2.1f);
    pVRVolQuality.edgeDetect2DMethod.setValue(SoVolumeRenderingQuality.EdgeDetect2DMethods.DEPTH);
    pVRVolQuality.surfaceScalarExponent.setValue(8);
    referenceImageSeparator.addChild(pVRVolQuality);

    // Node in charge of drawing the volume

    SoMaterial refMat = new SoMaterial();
    refMat.transparency.setValue(0.5f);
    referenceImageSeparator.addChild(refMat);

    // create an orthoslice
    m_referenceImageOrthoSlice = new SoOrthoSlice();
    m_referenceImageOrthoSlice.sliceNumber.setValue(m_inputReference.getSize().getZ() / 2);
    m_referenceImageOrthoSlice.axis.setValue(2);
    m_referenceImageOrthoSlice.clipping.setValue(true);
    referenceImageSeparator.addChild(m_referenceImageOrthoSlice);

    // Create path to slice node
    SoPath referencePath = new SoPath(m_referenceImageOrthoSlice);

    // switch for enable/disable the dragger
    m_swithDraggerRef = new SoSwitch();
    m_swithDraggerRef.whichChild.setValue(0);

    // Create and initialize dragger
    SoOrthoSliceDragger referenceDragger = new SoOrthoSliceDragger();
    referenceDragger.orthoSlicePath.setValue(referencePath);
    referenceDragger.volumeDimension.setValue(referenceImageVolumeData.data.getSize());
    referenceDragger.volumeExtent.setValue(referenceImageVolumeData.extent.getValue());
    m_swithDraggerRef.addChild(referenceDragger);
    referenceImageSeparator.addChild(m_swithDraggerRef);

    // separator for the model image
    SoSeparator modelImageSeparator = new SoSeparator();
    rootVolumeRendering.addChild(modelImageSeparator);

    modelImageSeparator.addChild(createRotation(m_inputModel));

    // create an in-memory reader
    SoVRImageDataReader modelImageReader = new SoVRImageDataReader();
    modelImageReader.imageData.setValue(m_inputModel);

    // init the initial transform
    m_currTransform = new SoTransform();
    m_currTransform.setName("currTransformation");
    modelImageSeparator.addChild(m_currTransform);
    m_asyncTransformation = new ASyncTransformation(m_currTransform);

    // Node to hold the volume data
    SoVolumeData modelImageVolumeData = new SoVolumeData();
    modelImageVolumeData.setName("OrthoSliceInput");
    modelImageVolumeData.ldmResourceParameters.getValue().tileDimension.setValue(128, 128, 128);
    // Request LDM to render using tiles at specified resolution
    // level (Here resolution 0 == full res).
    modelImageVolumeData.ldmResourceParameters.getValue().fixedResolution.setValue(true);
    modelImageVolumeData.ldmResourceParameters.getValue().resolution.setValue(0);
    modelImageVolumeData.texturePrecision.setValue((short) 16); // Store
                                                                // actual
    // values on GPU.
    modelImageVolumeData.setReader(modelImageReader, true);
    modelImageSeparator.addChild(modelImageVolumeData);

    // compute the data min - max
    m_modelMinMax = modelImageVolumeData.getMinMax();

    // Node to remap data range to full range of data
    SoDataRange inputImageDataRange = new SoDataRange();
    inputImageDataRange.min.setValue(m_modelMinMax[0]);
    inputImageDataRange.max.setValue(m_modelMinMax[1]);
    modelImageSeparator.addChild(inputImageDataRange);

    // Node for colormap
    SoTransferFunction inputImageTransferFunction = new SoTransferFunction();
    inputImageTransferFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);
    inputImageTransferFunction.minValue.setValue((int) m_modelMinMax[0]);
    inputImageTransferFunction.maxValue.setValue((int) m_modelMinMax[1]);
    modelImageSeparator.addChild(inputImageTransferFunction);

    // Node in charge of drawing the volume
    m_modelImageOrthoSlice = new SoOrthoSlice();
    m_modelImageOrthoSlice.sliceNumber.setValue(m_inputModel.getSize().getZ() / 2);
    m_modelImageOrthoSlice.axis.setValue(SoOrthoSlice.AxisType.Z);
    modelImageSeparator.addChild(m_modelImageOrthoSlice);

    // Create path to slice node
    SoPath modelPath = new SoPath(m_modelImageOrthoSlice);

    // Create and initialize dragger
    m_swithDraggerModel = new SoSwitch();
    SoOrthoSliceDragger modelDragger = new SoOrthoSliceDragger();
    modelDragger.orthoSlicePath.setValue(modelPath);
    modelDragger.volumeDimension.setValue(modelImageVolumeData.data.getSize());
    modelDragger.volumeExtent.setValue(modelImageVolumeData.extent.getValue());
    m_swithDraggerModel.addChild(modelDragger);

    m_swithDraggerModel.whichChild.setValue(0);
    modelImageSeparator.addChild(m_swithDraggerModel);

    // separator for text
    SoSeparator textDisplaySeparator = new SoSeparator();
    m_root.addChild(textDisplaySeparator);

    SoOrthographicCamera cam = new SoOrthographicCamera();
    cam.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);
    cam.position.setValue(0.005f, 0.005f, 1);
    cam.height.setValue(0.01f);
    textDisplaySeparator.addChild(cam);

    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    textDisplaySeparator.addChild(pickStyle);

    SoLightModel lModel = new SoLightModel();
    lModel.model.setValue(SoLightModel.Models.BASE_COLOR);
    textDisplaySeparator.addChild(lModel);

    SoFontStyle fontInfo = new SoFontStyle();
    fontInfo.name.setValue("Courier New");
    fontInfo.size.setValue(20);
    fontInfo.style.setValue(SoFontStyle.Styles.BOLD);
    fontInfo.renderStyle.setValue(SoFont.RenderStyles.TEXTURE);
    textDisplaySeparator.addChild(fontInfo);

    SoBaseColor infoColor = new SoBaseColor();
    infoColor.rgb.setValue(1, 1, 1.f);
    textDisplaySeparator.addChild(infoColor);

    SoTranslation transInfo = new SoTranslation();
    transInfo.translation.setValue(0.00025f, 0.00975f, 0.0f);
    textDisplaySeparator.addChild(transInfo);

    m_infoText = new SoText2();
    m_infoText.string.set1Value(0, m_infoMessage);
    textDisplaySeparator.addChild(m_infoText);
    m_asyncText = new ASyncText(m_infoText);

    m_infoText.string.set1Value(3, "E:Euclidean | C:Correlation: M:Mutual Information | SPACE BAR:Cancel execution");
    m_infoText.string.set1Value(6, "0:Translation | 1:Rigid | 2:Rigid + Iso-Scale | 3:Rigid + Aniso-Scale | 4:Affine");

    displayMetric();

    // abort registration
    if ( m_registration.isEvaluating() )
    {
      m_registration.abortEvaluate();
      m_registration.waitEvaluate();
    }

    myRenderArea.setSceneGraph(m_root);
    myRenderArea.viewAll();

    final Component component = myRenderArea.getComponent();
    component.setPreferredSize(new java.awt.Dimension(800, 800));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myRenderArea.dispose();
  }

  // event callback
  // R: starts the intensity registration
  // P: starts a pre-alignement
  // C, M, E: change the metric to respectively: CORRELATION,
  // NORMALIZED_MUTUAL_INFORMATION, EUCLIDIAN
  // SPACE: abort the computation
  // PAD_0 to PAD_3: tunes the transformations
  // K: reset the transformation
  public class ProcessKeyEvents extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback cb)
    {
      SoKeyboardEvent keyEvent = (SoKeyboardEvent) cb.getEvent();
      Keys key = keyEvent.getKey();

      if ( SoKeyboardEvent.isKeyPressEvent(cb.getEvent(), key) )
        return;

      switch ( key )
      {
      case R : // start the registration
        if ( !m_registration.isEvaluating() )
        {
          m_registration.inMovingImage.setValue(m_inputModel);
          m_registration.inFixedImage.setValue(m_inputReference);
          m_registration.initialTransform.setValue(getCurrMatrix());
          m_registration.ignoreFinestLevel.setValue(true);
          m_registration.startEvaluate();
        }
        break;
      case P : // IsoParametic
               // switch from transform to manipulator
               // NB: if there is a manipulator there is no dragging of the
               // orthoslice
               // the switch is used to disabled the dragger
        if ( m_swithDraggerRef.whichChild.getValue() == 0 )
        {
          SoSearchAction mySearchAction = new SoSearchAction();
          mySearchAction.setNodeClass(SoTransform.class);
          mySearchAction.setInterest(SoSearchAction.Interests.FIRST);
          mySearchAction.apply(m_root);

          m_manipulator.replaceNode(mySearchAction.getPath());
          m_swithDraggerModel.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
          m_swithDraggerRef.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
        }
        else
        {
          SoSearchAction mySearchAction = new SoSearchAction();
          mySearchAction.setNodeClass(SoTransformerManip.class);
          mySearchAction.setInterest(SoSearchAction.Interests.FIRST);
          mySearchAction.apply(m_root);
          m_swithDraggerModel.whichChild.setValue(0);
          m_swithDraggerRef.whichChild.setValue(0);
          m_manipulator.replaceManip(mySearchAction.getPath(), m_currTransform);
        }
        break;
      case C : // correlation type
        m_registration.metricType.setValue(SoImageRegistrationTransform.MetricTypes.CORRELATION);
        break;
      case M : // mutual information type
        m_registration.metricType.setValue(SoImageRegistrationTransform.MetricTypes.NORMALIZED_MUTUAL_INFORMATION);
        break;
      case E : // euclidean type
        m_registration.metricType.setValue(SoImageRegistrationTransform.MetricTypes.EUCLIDIAN);
        break;
      case SPACE : // abort
        m_registration.abortEvaluate();
        break;
      case PAD_0 : // translation
      case NUMBER_0 :
        m_registration.transformType.setValue(SoImageRegistrationTransform.TransformationTypes.TRANSLATION);
        break;
      case PAD_1 : // rigid
      case NUMBER_1 :
        m_registration.transformType.setValue(SoImageRegistrationTransform.TransformationTypes.RIGID);
        break;
      case PAD_2 : // rigid iso scaling
      case NUMBER_2 :
        m_registration.transformType.setValue(SoImageRegistrationTransform.TransformationTypes.RIGID_ISOTROPIC_SCALING);
        break;
      case PAD_3 : // rigid aniso scaling
      case NUMBER_3 :
        m_registration.transformType
            .setValue(SoImageRegistrationTransform.TransformationTypes.RIGID_ANISOTROPIC_SCALING);
        break;
      case PAD_4 : // affine
      case NUMBER_4 :
        m_registration.transformType.setValue(SoImageRegistrationTransform.TransformationTypes.AFFINE);
        break;
      default:
        // do nothing
        break;
      }
      cb.setHandled();
      displayMetric();
    }
  }

  public SbMatrix getCurrMatrix()
  {
    SoTransform transform = getTransform();
    SbMatrix tr = SbMatrix.identity();
    tr.setTransform(transform.translation.getValue(), transform.rotation.getValue(), transform.scaleFactor.getValue(),
        transform.scaleOrientation.getValue(), transform.center.getValue());
    return tr;
  }

  public SoTransform getTransform()
  {
    SoTransform transform = null;

    SoSearchAction mySearchAction = new SoSearchAction();
    mySearchAction.setNodeClass(SoTransform.class);
    mySearchAction.setInterest(SoSearchAction.Interests.FIRST);
    mySearchAction.apply(m_root);

    if ( mySearchAction.getPath() != null )
      transform = (SoTransform) mySearchAction.getPath().regular.getTail();

    return transform;
  }

  // progress notification for the registration
  // the text and transformation are updated only once every 10 times to avoid
  // a latency due to the interface update
  class RegistrationEventListener implements SbEventListener<SoImageRegistrationTransform.RegistrationEvent>
  {
    @Override
    public void onEvent(SoImageRegistrationTransform.RegistrationEvent event)
    {
      if ( m_priority % 10 == 0 )
      {
        setInfoValueAsync("Progression " + event.getProgress() + " %", 0);
        setInfoValueAsync("Metric " + event.getSimilarity() + " %", 1);
        updateTransformation(event.getTransform());
      }
      else
      {
        m_priority++;
      }
    }
  }

  // registration start notification
  class RegistrationStartListener implements SbEventListener<SoImageVizEngine.EventArg>
  {
    @Override
    public void onEvent(SoImageVizEngine.EventArg event)
    {
      setInfoValueAsync("Registration starts", 0);
    }
  }

  // registration end notification
  class RegistrationEndListener implements SbEventListener<SoImageVizEngine.EventArg>
  {
    @Override
    public void onEvent(SoImageVizEngine.EventArg event)
    {
      updateTransformation(m_registration.getOutputTransformation());
      setInfoValueAsync("", 1);
      setInfoValueAsync(m_infoMessage, 0);
      m_priority = 0;
    }
  }

  // add update messages in a priority queue
  public void setInfoValueAsync(String value, int index)
  {
    m_asyncText.setValues(value, index);
    SoOneShotSensor sensor = new SoOneShotSensor();
    sensor.setTask(m_asyncText);
    sensor.setPriority(m_priority);
    sensor.schedule();
    m_priority++;
  }

  // add output transformation in a priority queue
  public void setTransformationValueAsync(SbVec3f translation, SbRotation rotation, SbVec3f scaleFactor,
      SbRotation scaleOrientation, SbVec3f center)
  {
    m_asyncTransformation.setValues(translation, rotation, scaleFactor, scaleOrientation, center);
    SoOneShotSensor sensor = new SoOneShotSensor();
    sensor.setTask(m_asyncTransformation);
    sensor.setPriority(m_priority);
    sensor.schedule();
    m_priority++;
  }

  // Create the rotation matrix needed to take in consideration the direction
  // cosines (orientation field)
  public SoMatrixTransform createRotation(SoImageDataAdapter image)
  {
    SbMatrix TrInv = SbMatrix.identity();
    TrInv.setTranslate(image.getOrigin());

    SbMatrix Rot = SbMatrix.identity();
    Rot.setRotate(image.orientation.getValue());

    SbMatrix Tr = SbMatrix.identity();
    SbVec3f invOrigin = image.getOrigin();
    invOrigin.negate();
    Tr.setTranslate(invOrigin);

    SoMatrixTransform transformation = new SoMatrixTransform();
    SbMatrix trm = Tr;
    trm.multiply(Rot);
    trm.multiply(TrInv);
    transformation.matrix.setValue(trm);

    return transformation;
  }

  public void updateTransformation(SbMatrix matrix)
  {
    SbVec3f center = new SbVec3f(0.0f, 0.0f, 0.0f);
    SbMatrix.Decomposition decomp = matrix.decompose();
    setTransformationValueAsync(decomp.translation, decomp.rotation, decomp.scaleFactor, decomp.scaleOrientation,
        center);
  }

  // activate the transformation in parameter
  public String active(SoImageRegistrationTransform.TransformationTypes type)
  {
    return ((m_registration.transformType.getValue() == type.getValue()) ? "[.]" : "[ ]");
  }

  // display information about the metric and the transformation
  public void displayMetric()
  {
    String metricStr;
    if ( m_registration.metricType.getValue() == SoImageRegistrationTransform.MetricTypes.EUCLIDIAN.getValue() )
      metricStr = "Euclidean";
    else if ( m_registration.metricType.getValue() == SoImageRegistrationTransform.MetricTypes.CORRELATION.getValue() )
      metricStr = "Correlation";
    else
      metricStr = "Normalized Mutual Information";
    String msg = "metric is : " + metricStr;
    m_infoText.string.set1Value(4, msg);
    m_infoText.string.set1Value(7,
        active(SoImageRegistrationTransform.TransformationTypes.TRANSLATION) + "Translation Only");
    m_infoText.string.set1Value(8, active(SoImageRegistrationTransform.TransformationTypes.RIGID) + "Rigid");
    m_infoText.string.set1Value(9,
        active(SoImageRegistrationTransform.TransformationTypes.RIGID_ISOTROPIC_SCALING) + "Rigid + Iso-Scale");
    m_infoText.string.set1Value(10,
        active(SoImageRegistrationTransform.TransformationTypes.RIGID_ANISOTROPIC_SCALING) + "Rigid + Aniso-Scale");
    m_infoText.string.set1Value(11, active(SoImageRegistrationTransform.TransformationTypes.AFFINE) + "Affine");
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("ImageViz - Registration");
  }

}
