package inventor.advanced.MultipleRenderToTexture3D;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.File;

import javax.swing.BoxLayout;
import javax.swing.JPanel;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SbVec3s;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.fields.SoSFImage3;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * This demo exposes how to use SoRenderToTarget with a unique SoTexture3
 * attached to multiple targets. Each target is rendered with a specific layer
 * in the SoTexture3. The result is a textured cube: each face of the cube is
 * textured with a layer (i.e. 6 layers, one for each face) of a SoTexture3
 * generated with the SoRenderToTarget.
 */
public class Main extends Example
{
  private IViewerExaminer myViewer;
  private SoRenderToTarget renderToTarget;

  // default size for render to target viewport
  static short defaultSize = 256;

  // specify texture internal format: 0 for RGBA8, 1 for RGBA_FLOAT32
  static boolean useFloatColorBuffer = false;

  /*---------------------------------------------------------------------------*/
  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("MultipleRenderToTexture3D");
  }

  /*---------------------------------------------------------------------------*/
  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Add path to find our data
    SoInput.addDirectoryFirst(getDataPath() + "shaders");

    SoSeparator root = buildSceneGraph();

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    // SWING part
    JPanel sliderPanel = createSliderPanel();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(sliderPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  /*---------------------------------------------------------------------------*/
  private SoSeparator buildSceneGraph()
  {
    // initialize texture target
    SoTexture3 rgbaTexture = new SoTexture3();
    // specify the layout to use in SoTexture3: SoTexture3.Layouts.VOLUME for
    // Volume, SoTexture3.Layouts.ARRAY for Array
    SoTexture3.Layouts textureLayout = SoTexture3.Layouts.ARRAY;
    rgbaTexture.layout.setValue(textureLayout);

    if ( useFloatColorBuffer )
    {
      rgbaTexture.internalFormat.setValue(SoTexture.InternalFormats.RGBA_FLOAT32);
    }
    else
    {
      rgbaTexture.internalFormat.setValue(SoTexture.InternalFormats.RGBA8);
    }

    // allocate texture memory
    SbVec3s size = new SbVec3s(defaultSize, defaultSize, (short) 6);
    SoCpuBufferObject cpuBuffer = new SoCpuBufferObject();
    cpuBuffer.setSize(defaultSize * defaultSize * 6 * 4);
    rgbaTexture.images.setValue(size, 4, cpuBuffer, SoSFImage3.CopyPolicies.NO_COPY);

    // initialize render to target node
    renderToTarget = new SoRenderToTarget();
    renderToTarget.size.setValue(new SbVec2i32(defaultSize, defaultSize));

    // set targets, i.e. textures in which we want to render each target
    // here, the same texture, a SoTexture3, is attached to all targets
    renderToTarget.targets.set1Value(SoRenderToTarget.Attachments.COLOR0.getValue(), rgbaTexture);
    renderToTarget.targets.set1Value(SoRenderToTarget.Attachments.COLOR1.getValue(), rgbaTexture);
    renderToTarget.targets.set1Value(SoRenderToTarget.Attachments.COLOR2.getValue(), rgbaTexture);
    renderToTarget.targets.set1Value(SoRenderToTarget.Attachments.COLOR3.getValue(), rgbaTexture);
    renderToTarget.targets.set1Value(SoRenderToTarget.Attachments.COLOR4.getValue(), rgbaTexture);
    renderToTarget.targets.set1Value(SoRenderToTarget.Attachments.COLOR5.getValue(), rgbaTexture);

    // since the same texture is attached for all render target, we specify
    // the layer in which each target must write into
    renderToTarget.layers.set1Value(SoRenderToTarget.Attachments.COLOR0.getValue(), 0);
    renderToTarget.layers.set1Value(SoRenderToTarget.Attachments.COLOR1.getValue(), 1);
    renderToTarget.layers.set1Value(SoRenderToTarget.Attachments.COLOR2.getValue(), 2);
    renderToTarget.layers.set1Value(SoRenderToTarget.Attachments.COLOR3.getValue(), 3);
    renderToTarget.layers.set1Value(SoRenderToTarget.Attachments.COLOR4.getValue(), 4);
    renderToTarget.layers.set1Value(SoRenderToTarget.Attachments.COLOR5.getValue(), 5);

    // build the scene for offline rendering
    SoGroup rttScene = new SoGroup();
    {
      SoShaderProgram prog = new SoShaderProgram();
      prog.setVertexShader(0, "RTT_vert.glsl");
      prog.setFragmentShader(1, "RTT_frag.glsl");
      rttScene.addChild(prog);

      // just a quad in this scene to draw on it
      SoGradientBackground bgrd = new SoGradientBackground();
      bgrd.color0.setValue(new SbColor(0.0f, 0.0f, 0.0f));
      bgrd.color1.setValue(new SbColor(0.0f, 0.0f, 0.0f));
      rttScene.addChild(bgrd);
    }
    renderToTarget.addChild(rttScene);

    // build the scene for online rendering
    SoSeparator scene = new SoSeparator();
    {
      SoTextureUnit texUnit = new SoTextureUnit();
      texUnit.unit.setValue(0);

      scene.addChild(texUnit);
      scene.addChild(rgbaTexture);

      SoShaderProgram prog = new SoShaderProgram();

      prog.setVertexShader(0, "Scene_vert.glsl");
      SoFragmentShader frag = prog.setFragmentShader(1, "Scene_frag.glsl");

      SoShaderParameter1i rgbaTexture3d = new SoShaderParameter1i();
      rgbaTexture3d.name.setValue("rgbaTexture3d");
      rgbaTexture3d.value.setValue(0);
      SoShaderParameter1i rgbaTexture2dArray = new SoShaderParameter1i();
      rgbaTexture2dArray.name.setValue("rgbaTexture2dArray");
      rgbaTexture2dArray.value.setValue(0);
      SoShaderParameter1i texLayout = new SoShaderParameter1i();
      texLayout.name.setValue("tex_layout_type");
      texLayout.value.setValue(textureLayout.getValue());

      frag.parameter.addShaderParameter(rgbaTexture3d);
      frag.parameter.addShaderParameter(rgbaTexture2dArray);
      frag.parameter.addShaderParameter(texLayout);
      texLayout.value.connectFrom(rgbaTexture.layout);

      scene.addChild(prog);
      scene.addChild(new SoCube());
    }

    // build the whole scene graph
    SoSeparator root = new SoSeparator();
    root.addChild(new SoGradientBackground());
    root.addChild(renderToTarget);
    root.addChild(scene);

    return root;
  }

  private String getDataPath()
  {
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    return SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;
  }

  /*---------------------------------------------------------------------------*/
  private JPanel createSliderPanel()
  {
    SliderPanel targetSizeSlider = new SliderPanel(256, 2048, 256);
    targetSizeSlider.addInfoText("Target size : ");
    targetSizeSlider.setSliderSize(new Dimension(200, 20));
    targetSizeSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        renderToTarget.size.setValue(new SbVec2i32((int) value, (int) value));
      }
    });

    JPanel sliderPanel = new JPanel();
    sliderPanel.setLayout(new BoxLayout(sliderPanel, BoxLayout.Y_AXIS));
    sliderPanel.add(targetSizeSlider);

    return sliderPanel;
  }

}
