package inventor.advanced.multiInstancing.bufferedshape;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.File;
import java.nio.FloatBuffer;

import javax.swing.BoxLayout;
import javax.swing.JPanel;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbDataType.DataTypes;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoBoxHighlightRenderAction;
import com.openinventor.inventor.devices.SoBufferObject.AccessModes;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.nodes.SoExtSelection;
import com.openinventor.inventor.nodes.SoExtSelection.LassoModes;
import com.openinventor.inventor.nodes.SoExtSelection.LassoPolicies;
import com.openinventor.inventor.nodes.SoExtSelection.LassoTypes;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoInstanceParameter;
import com.openinventor.inventor.nodes.SoInstanceParameter.PredefinedParameters;
import com.openinventor.inventor.nodes.SoMultipleInstance;
import com.openinventor.inventor.nodes.SoSelection.Policies;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private final static int SBVEC3F_SIZE = Float.SIZE / 8 * 3;
  private final static int SBVEC4F_SIZE = Float.SIZE / 8 * 4;

  private IViewerExaminer myViewer;
  private SoMultipleInstance m_multiInstance;
  private int m_numInstX = 10;
  private int m_numInstY = 10;
  private int m_numInstZ = 10;

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Add path to find our data
    SoInput.addDirectoryFirst(getDataPath());
    SoSeparator sceneGraph = createSceneGraph();

    // set auto selection feedback
    SoBoxHighlightRenderAction boxHighlightRenderAction = new SoBoxHighlightRenderAction();
    boxHighlightRenderAction.setColor(new SbColor(1.f, 0.f, 0.f));
    myViewer.getRenderArea().setGLRenderAction(boxHighlightRenderAction);
    myViewer.setSceneGraph(sceneGraph);
    myViewer.viewAll();

    // SWING part
    JPanel sliderPanel = createSliderPanel();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(sliderPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  // create SoBufferObject containing per instance translations
  private SoCpuBufferObject createTranslationInstanceParameterBuffer(int numInstanceX, int numInstanceY,
      int numInstanceZ)
  {
    SoCpuBufferObject bufferValues = new SoCpuBufferObject();
    bufferValues.setSize(numInstanceX * numInstanceY * numInstanceZ * SBVEC3F_SIZE);

    FloatBuffer data = bufferValues.map(AccessModes.SET).asFloatBuffer();
    for ( int k = 0; k < numInstanceZ; k++ )
      for ( int j = 0; j < numInstanceY; j++ )
        for ( int i = 0; i < numInstanceX; i++ )
        {
          int index = (i + j * numInstanceX + k * numInstanceX * numInstanceY) * 3;
          data.put(index, i * 100.f);
          data.put(index + 1, j * 100.f);
          data.put(index + 2, k * 100.f);
        }
    bufferValues.unmap();

    return bufferValues;
  }

  // This method is provided as an explicit example of setting a generic
  // SoInstanceParameter.
  // As a matter of fact the parameter set is one of the predefined ones.
  // The same setting could be done in a less verbose way calling
  // the SoInstanceParameter.createPredefinedParameter method directly passing
  // the SoCpuBufferObject returned by
  // createTranslationInstanceParameterBuffer()
  private SoInstanceParameter createTranslationInstanceParameter(int numInstanceX, int numInstanceY, int numInstanceZ)
  {
    SoInstanceParameter translation = new SoInstanceParameter();
    translation.name.setValue(SoInstanceParameter.getPredefinedParameterName(PredefinedParameters.TRANSLATION));
    translation.components.setValue(3);
    translation.type.setValue(DataTypes.FLOAT);

    SoCpuBufferObject bufferValues = createTranslationInstanceParameterBuffer(numInstanceX, numInstanceY, numInstanceZ);
    translation.value.setValue(bufferValues);

    return translation;
  }

  // create SoBufferObject containing per instance scaling factors
  private SoCpuBufferObject createScaleInstanceParameterBuffer(int numInstanceX, int numInstanceY, int numInstanceZ)
  {
    SoCpuBufferObject bufferValues = new SoCpuBufferObject();
    bufferValues.setSize(numInstanceX * numInstanceY * numInstanceZ * SBVEC3F_SIZE);

    FloatBuffer data = bufferValues.map(AccessModes.SET).asFloatBuffer();
    for ( int k = 0; k < numInstanceZ; k++ )
    {
      for ( int j = 0; j < numInstanceY; j++ )
      {
        for ( int i = 0; i < numInstanceX; i++ )
        {
          float s =
              (float) (1.f + 0.5f * Math.cos(10. * Math.PI * (i / (float) numInstanceZ))
                  * Math.cos(10. * Math.PI * (j / (float) numInstanceY))
                  * Math.cos(10. * Math.PI * (k / (float) numInstanceX)));
          int index = (i + j * numInstanceX + k * numInstanceX * numInstanceY) * 3;
          data.put(index, s);
          data.put(index + 1, s);
          data.put(index + 2, s);
        }
      }
    }
    bufferValues.unmap();

    return bufferValues;
  }

  // create SoBufferObject containing per instance rotations factors
  private SoCpuBufferObject createRotationInstanceParameterBuffer()
  {
    int numRotations = 4;

    SoCpuBufferObject bufferValues = new SoCpuBufferObject();
    bufferValues.setSize(numRotations * SBVEC4F_SIZE);

    // Will hold a set of quaternions
    FloatBuffer data = bufferValues.map(AccessModes.SET).asFloatBuffer();
    // Pi/2 rotation on x
    data.put(0, 1.0f);
    data.put(1, 0.0f);
    data.put(2, 0.0f);
    data.put(3, (float) (Math.PI / 4.f));

    // Pi/2 rotation on y
    data.put(4, 0.0f);
    data.put(5, 1.0f);
    data.put(6, 0.0f);
    data.put(7, (float) (Math.PI / 4.f));

    // Pi/2 rotation on z
    data.put(8, 0.0f);
    data.put(9, 0.0f);
    data.put(10, 1.0f);
    data.put(11, (float) (Math.PI / 4.f));

    // Pi/2 rotation on x and y
    data.put(12, (float) (Math.sin(Math.PI / 4) / Math.sqrt(2)));
    data.put(13, (float) (Math.sin(Math.PI / 4) / Math.sqrt(2)));
    data.put(14, 0.0f);
    data.put(15, (float) (Math.PI / 4.f));
    bufferValues.unmap();

    return bufferValues;
  }

  // Read an Inventor file and return the top separator
  private SoSeparator readFile(String filename)
  {
    // Open the input file
    SoInput mySceneInput = new SoInput();
    if ( !mySceneInput.openFile(filename) )
      return null;

    // Read the whole file into the database
    SoSeparator myGraph = SoDB.readAll(mySceneInput);
    if ( myGraph == null )
      return null;

    mySceneInput.closeFile();

    return myGraph;
  }

  private void setVariableInstanceData()
  {
    int numInstances = m_numInstX * m_numInstY * m_numInstZ;

    // set number of instances
    m_multiInstance.numInstances.setValue(numInstances);

    // Set per instance translations
    // The following code provided just as an explicit example of setting
    // a generic SoInstanceParameter see createTranslationInstanceParameter.
    // The same settings could be done in a less verbose way by calling
    // SoInstanceParameter.createPredefinedParameter as it is done
    // afterwards for scale factors and rotations parameters
    SoInstanceParameter translationParameter =
        m_multiInstance.parameters.findParameterByName(SoInstanceParameter
            .getPredefinedParameterName(PredefinedParameters.TRANSLATION));
    if ( translationParameter == null )
    {
      translationParameter = createTranslationInstanceParameter(m_numInstX, m_numInstY, m_numInstZ);
      m_multiInstance.parameters.set1Value(m_multiInstance.parameters.getNum(), translationParameter);
    }
    else
    {
      SoCpuBufferObject translationsBuffer =
          createTranslationInstanceParameterBuffer(m_numInstX, m_numInstY, m_numInstZ);
      translationParameter.value.setValue(translationsBuffer);
    }

    // set per instance scale factors
    SoCpuBufferObject scaleBuffer = createScaleInstanceParameterBuffer(m_numInstX, m_numInstY, m_numInstZ);
    SoInstanceParameter scaleParameter =
        m_multiInstance.parameters.findParameterByName(SoInstanceParameter
            .getPredefinedParameterName(PredefinedParameters.SCALE));
    if ( scaleParameter == null )
    {
      scaleParameter = SoInstanceParameter.createPredefinedParameter(PredefinedParameters.SCALE, scaleBuffer);
      m_multiInstance.parameters.set1Value(m_multiInstance.parameters.getNum(), scaleParameter);
    }
    else
      scaleParameter.value.setValue(scaleBuffer);

    // set per instance rotations
    int numRotations = 4;
    int rotationDivisor = 1 + numInstances / numRotations;

    SoCpuBufferObject rotationBuffer = createRotationInstanceParameterBuffer();
    SoInstanceParameter rotationParameter =
        m_multiInstance.parameters.findParameterByName(SoInstanceParameter
            .getPredefinedParameterName(PredefinedParameters.ROTATION));
    if ( rotationParameter == null )
    {
      rotationParameter =
          SoInstanceParameter.createPredefinedParameter(PredefinedParameters.ROTATION, rotationBuffer, rotationDivisor);
      m_multiInstance.parameters.set1Value(m_multiInstance.parameters.getNum(), rotationParameter);
    }
    else
    {
      rotationParameter.value.setValue(rotationBuffer);
      rotationParameter.divisor.setValue(rotationDivisor);
    }
  }

  private SoSeparator createSceneGraph()
  {
    SoSeparator root = new SoSeparator();
    root.addChild(new SoGradientBackground());

    // create selection
    SoExtSelection sel = new SoExtSelection();
    sel.useFastEditing(true);
    sel.setLassoColor(new SbColor(0, 1, 0));
    sel.setOverlayLassoPattern((short) 0xE0E0);
    sel.animateOverlayLasso(true);
    sel.policy.setValue(Policies.SINGLE);
    sel.lassoType.setValue(LassoTypes.RECTANGLE);
    sel.lassoPolicy.setValue(LassoPolicies.FULL_BBOX);
    sel.lassoMode.setValue(LassoModes.ALL_SHAPES);

    // add selection to scene graph
    root.addChild(sel);

    // create multiple instance group
    m_multiInstance = new SoMultipleInstance();
    sel.addChild(m_multiInstance);

    // add BufferShape representing a screw as single node of the
    // SoMultipleInstance group
    SoSeparator sg = readFile("screw_buffered_shape.iv");
    m_multiInstance.addChild(sg);

    setVariableInstanceData();

    return root;
  }

  private String getDataPath()
  {
    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    return SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;
  }

  private JPanel createSliderPanel()
  {
    SliderPanel numXSlider = new SliderPanel(1, 25, m_numInstX);
    numXSlider.addInfoText("NumX : ");
    numXSlider.setSliderSize(new Dimension(200, 20));
    numXSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_numInstX = (int) value;
        setVariableInstanceData();
      }
    });

    SliderPanel numYSlider = new SliderPanel(1, 25, m_numInstY);
    numYSlider.addInfoText("NumY : ");
    numYSlider.setSliderSize(new Dimension(200, 20));
    numYSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_numInstY = (int) value;
        setVariableInstanceData();
      }
    });

    SliderPanel numZSlider = new SliderPanel(1, 25, m_numInstZ);
    numZSlider.addInfoText("NumZ : ");
    numZSlider.setSliderSize(new Dimension(200, 20));
    numZSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_numInstZ = (int) value;
        setVariableInstanceData();
      }
    });

    JPanel sliderPanel = new JPanel();
    sliderPanel.setLayout(new BoxLayout(sliderPanel, BoxLayout.Y_AXIS));
    sliderPanel.add(numXSlider);
    sliderPanel.add(numYSlider);
    sliderPanel.add(numZSlider);

    return sliderPanel;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Instanced Buffered Shapes");
  }
}
