///////////////////////////////////////////////////////////////////////
//
// Open Inventor example for Shaders usage.
// This example shows you how to code within a vertex shader program
// an animation in order to speed up the rendering.
//
// Source code involved with shaders is framed with
// BEGIN SHADER and END SHADER.
//
// A dialog window allows to select which shading language to use when
// this example is executed.
//
///////////////////////////////////////////////////////////////////////

package inventor.advanced.shaders.AnimatedFlag;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.IntegerValuedEnumHelper;
import com.openinventor.inventor.SbTime;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoTimerSensor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;


public class Main extends Example {

  private static final int MAX_ANIM_STEP = 10000;
  private static final int NumPolyPerRowFlag = 300; // Initialization of its
                                                    // complexity

  private IViewerExaminer myViewer;
  private SoShaderObject.SourceTypes shadingLanguageUsed;
  private SbVec2f dimensionFlag = new SbVec2f(9.0f, 6.9f); // Dimensions
  private SoVertexProperty flagVertexProperty;
  private SoQuadMesh flagGeom;
  private SoShaderParameter1f animStep; // Shader's Parameter
  private SoTexture2 logoTexture;
  private SoShaderProgram shaderProgram;
  private SoVertexShader vertexShader;
  private boolean doAnim = true;

  /*---------------------------------------------------------------------------*/
  public static void main(String[] argv) {
    Main example = new Main();
    example.demoMain("Animated flag with shaders");
  }

  /*---------------------------------------------------------------------------*/
  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    shadingLanguageUsed = SoShaderObject.SourceTypes.GLSL_PROGRAM;

    SoSeparator scene_graph = makeSceneGraph();

    SoTimerSensor timer = new SoTimerSensor();
    timer.setInterval(new SbTime(0.01));
    timer.setTask(new Runnable() {
      @Override
      public void run() {
        if (doAnim) {
          float v = animStep.value.getValue();
          v = (v < MAX_ANIM_STEP) ? v + 0.3f : 0.0f;
          animStep.value.setValue(v);
        }
      }
    }
    );
    timer.schedule();


    myViewer.setSceneGraph(scene_graph);
    myViewer.viewAll();


    UserInterface ui = new UserInterface(this);

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(ui.getMainPanel(), BorderLayout.WEST);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  void configureShader(int shadingLanguageUsed) {
    /*********************** BEGIN SHADER ****************************/

    // Initialize and set the vertex shader

    // path to shader language file and texture files
    String texture_path = "$OIVJHOME/data/textures/";
    String shaders_path = "$OIVJHOME/data/shaders/";

    logoTexture.filename.setValue(texture_path + "png/VSG_logo.png");

    // We set the shader source file, depending on which shading language has been chosen
    switch(IntegerValuedEnumHelper.fromIntValue(SoShaderObject.SourceTypes.class, shadingLanguageUsed)) {
    case CG_PROGRAM:
      vertexShader.sourceProgram.setValue(shaders_path + "AnimatedFlagVtx.cg");
      animStep.name.setValue("AnimStep");
      break;
    case ARB_PROGRAM:
      vertexShader.sourceProgram.setValue(shaders_path + "AnimatedFlagVtx.vp");
      animStep.identifier.setValue(0);
      break;
    case GLSL_PROGRAM:
      vertexShader.sourceProgram.setValue(shaders_path + "AnimatedFlagVtx.glsl");
      animStep.name.setValue("AnimStep");
      break;
    default:
      throw new java.lang.UnsupportedOperationException("shader not supported");
    }

    // Set the parameter
    vertexShader.parameter.set1Value(0, animStep);
    /************************ END SHADER *****************************/
  }

  private SoSeparator makeSceneGraph() {
    SoSeparator flagSep;
    SoSeparator noClippingSep;
    SoSeparator flagGeomSep;
    SoSeparator poleSep;

    // Create the flag
    SoShapeHints hints = new SoShapeHints();
    hints.vertexOrdering.setValue(SoShapeHints.VertexOrderings.COUNTERCLOCKWISE);

    // Fill the flag's vertexProperty
    flagVertexProperty = new SoVertexProperty();
    flagVertexProperty.normalBinding.setValue(SoVertexProperty.Bindings.OVERALL);
    flagVertexProperty.materialBinding.setValue(SoVertexProperty.Bindings.OVERALL);
    fillCoords(dimensionFlag, NumPolyPerRowFlag, 1.0f, flagVertexProperty);


    flagGeom = new SoQuadMesh();
    flagGeom.verticesPerRow.setValue(NumPolyPerRowFlag+1);
    flagGeom.verticesPerColumn.setValue(NumPolyPerRowFlag+1);
    flagGeom.vertexProperty.setValue(flagVertexProperty);

    logoTexture = new SoTexture2();
    logoTexture.model.setValue(SoTexture2.Models.DECAL);

    // Create the pole
    SoMaterial poleMaterial = new SoMaterial();
    poleMaterial.diffuseColor.setValue(1.0f, 1.0f, 0.0f);

    SoTransform poleTransf = new SoTransform();
    poleTransf.scaleFactor.setValue(0.20f, 9.0f, 0.20f);
    poleTransf.translation.setValue(-0.20f, 6.2f, 0.0f);

    /*********************** BEGIN SHADER ****************************/
    // Create a separator for the shader
    SoSeparator vertexShaderSep = new SoSeparator();
    animStep = new SoShaderParameter1f();
    animStep.value.setValue(0.0f);

    // Initialize and set the shader program
    shaderProgram = new SoShaderProgram();
    vertexShader = new SoVertexShader();

    setShaderLanguage(shadingLanguageUsed.getValue());

    /************************ END SHADER *****************************/

    SoCube eliminateClipping = new SoCube();
    eliminateClipping.height.setValue(20);
    eliminateClipping.width.setValue(20);
    eliminateClipping.depth.setValue(20);

    SoTranslation noClippingTrans = new SoTranslation();
    noClippingTrans.translation.setValue(0.0f, 6.0f, 0.0f);

    SoDrawStyle invisible =  new SoDrawStyle();
    invisible.style.setValue(SoDrawStyle.Styles.INVISIBLE);

    // Assemble the scene graph
    SoSeparator root = new SoSeparator();
    {
      root.addChild(hints);
      // Create a separator for the different parts of the flag
      root.addChild(flagSep = new SoSeparator());
      {
        // Add the flag's geometry animated by the shader
        // and add the pole
        flagSep.addChild(vertexShaderSep);
        {
          vertexShaderSep.addChild(shaderProgram);
          // Add the flag's geometry to the shader's separator to apply the shader to it
          vertexShaderSep.addChild(flagGeomSep = new SoSeparator());
          {
            // Put the flag's geometry in a Separator
            flagGeomSep.addChild(logoTexture);
            flagGeomSep.addChild(flagGeom);
          }
        }
        flagSep.addChild(poleSep = new SoSeparator());
        {
          poleSep.addChild(poleMaterial);
          poleSep.addChild(poleTransf);
          poleSep.addChild(new SoCylinder());
        }
      }
      // Create an invisible cube englobing the entire flag to avoid clipping effect.
      root.addChild(noClippingSep = new SoSeparator());
      {
        noClippingSep.addChild(invisible);
        noClippingSep.addChild(noClippingTrans);
        noClippingSep.addChild(eliminateClipping);
      }
    }

    return root;
  }

   private void fillCoords( SbVec2f dimension, int numPolyPerRow, float repeatTex,
                   SoVertexProperty vertexProperty) {
    // Dimension of the SoQuadMesh using the SoVertexProperty
    float lenght = dimension.getX();
    float width  = dimension.getY();

    float stepLength = lenght/numPolyPerRow;
    float stepWidth  = width/numPolyPerRow;

    int numVertexPerRow = numPolyPerRow+1;
    int size = numVertexPerRow*numVertexPerRow;

    // Set vertices' positions
    SbVec3f[] vertices = new SbVec3f[size];

    int f = 0;
    for (int i=0; i<numVertexPerRow;i++)
      for (int j=0; j<numVertexPerRow;j++)
        vertices[f++] = new SbVec3f(j*stepLength, 8.1f + i*stepWidth, 0);

    vertexProperty.vertex.setValues(0, vertices);

    // Set vertices' texture coordinates
    SbVec2f[] texCoord = new SbVec2f[size];

    for (int j=0; j<size;j++) {
      texCoord[j] = new SbVec2f(
        (vertices[j].getX()/lenght)*repeatTex,
        -0.175f+(vertices[j].getY()/width)*repeatTex
      );
    }

    vertexProperty.texCoord.setValues(0, texCoord);
  }

  public void setShaderLanguage(int shadingLanguage) {
    configureShader(shadingLanguage);
    shaderProgram.shaderObject.setValue(vertexShader);
  }

  public void setAnim(boolean doAnim) {
    this.doAnim = doAnim;
  }

}
