
package inventor.mentor.binding;

import java.awt.BorderLayout;
import java.awt.Checkbox;
import java.awt.CheckboxGroup;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JPanel;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.nodes.SoIndexedFaceSet;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoVertexProperty;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

/**
  * This is an example from the Inventor Mentor,
  * chapter 5, example 5.
  *
  * This example illustrates a variety of ways to bind
  * materials to a polygon object.
  * Three cases of a switch statement show various ways of
  * binding materials to objects.
  * The object used for all three examples is the stellated
  * dodecahedron from an earlier example in this chapter.
  *
  * @author Loic Vigneras
  * @author Patrick Vigneras
  * @author Jean Kinda
  */
public class Main extends Example implements ActionListener, ItemListener {
  static CheckboxGroup bindingGroup = new CheckboxGroup();
  static SoSeparator root;

  // Positions of all of the vertices:
  static float vertexPositions[][] = {
    { 0.0000F,  1.2142F,  0.7453F},  // top

    { 0.0000F,  1.2142F, -0.7453F},  // points surrounding top
    {-1.2142F,  0.7453F,  0.0000F},
    {-0.7453F,  0.0000F,  1.2142F},
    { 0.7453F,  0.0000F,  1.2142F},
    { 1.2142F,  0.7453F,  0.0000F},

    { 0.0000F, -1.2142F,  0.7453F},  // points surrounding bottom
    {-1.2142F, -0.7453F,  0.0000F},
    {-0.7453F,  0.0000F, -1.2142F},
    { 0.7453F,  0.0000F, -1.2142F},
    { 1.2142F, -0.7453F,  0.0000F},

    { 0.0000F, -1.2142F, -0.7453F}, // bottom
  };

  // Connectivity, information; 12 faces with 5 vertices each ,
  // (plus the end-of-face indicator for each face):

  static int indices[] = {
    1,  2,  3,  4, 5, SoIndexedFaceSet.SO_END_FACE_INDEX, // top face

    0,  1,  8,  7, 3, SoIndexedFaceSet.SO_END_FACE_INDEX, // 5 faces about top
    0,  2,  7,  6, 4, SoIndexedFaceSet.SO_END_FACE_INDEX,
    0,  3,  6, 10, 5, SoIndexedFaceSet.SO_END_FACE_INDEX,
    0,  4, 10,  9, 1, SoIndexedFaceSet.SO_END_FACE_INDEX,
    0,  5,  9,  8, 2, SoIndexedFaceSet.SO_END_FACE_INDEX,

    9,  5, 4, 6, 11, SoIndexedFaceSet.SO_END_FACE_INDEX, // 5 faces about bottom
    10,  4, 3, 7, 11, SoIndexedFaceSet.SO_END_FACE_INDEX,
    6,  3, 2, 8, 11, SoIndexedFaceSet.SO_END_FACE_INDEX,
    7,  2, 1, 9, 11, SoIndexedFaceSet.SO_END_FACE_INDEX,
    8,  1, 5,10, 11, SoIndexedFaceSet.SO_END_FACE_INDEX,

    6,  7, 8, 9, 10, SoIndexedFaceSet.SO_END_FACE_INDEX, // bottom face
  };

  // Colors for the 12 faces
  static float colors[][] = {
    {1.0F, .0F, 0F}, {.0F,  .0F, 1.0F}, {0F, .7F,  .7F}, { .0F, 1.0F,  0F},
    { .7F, .7F, 0F}, {.7F,  .0F,  .7F}, {0F, .0F, 1.0F}, { .7F,  .0F, .7F},
    { .7F, .7F, 0F}, {.0F, 1.0F,  .0F}, {0F, .7F,  .7F}, {1.0F,  .0F,  0F}
  };

  // Which material to use to color the faces
  // half red & half blue
  static int materialIndices[] = {
    0, 0, 0, 0, 0, 0,
    1, 1, 1, 1, 1, 1,
  };

  private IViewerExaminer myViewer;

  // Routine to create a scene graph representing a dodecahedron
  SoSeparator makeStellatedDodecahedron() {
    SoSeparator result = new SoSeparator();

    // This is the preferred code for Inventor 2.1
    // otherwise, see below the commented block

    // Using the new SoVertexProperty node is more efficient
    SoVertexProperty myVertexProperty = new SoVertexProperty();

    // The material binding.
    myVertexProperty.materialBinding.setValue(SoVertexProperty.Bindings.PER_FACE);

    // Define colors for the faces
    for (int i=0 ; i<12 ; i++)
      myVertexProperty.orderedRGBA.set1Value(i, new SbColor(colors[i]).getPackedValue());

    // Define coordinates for vertices
    myVertexProperty.vertex.setValues(0, vertexPositions);

    // Define the IndexedFaceSet, with indices into
    // the vertices:
    SoIndexedFaceSet myFaceSet = new SoIndexedFaceSet();
    myFaceSet.coordIndex.setValues(0, indices);

    myFaceSet.vertexProperty.setValue(myVertexProperty);
    result.addChild(myFaceSet);

    return result;
  }

  void setBindingType(int whichBinding) {
    // prefered way of programming
    // otherwise see below the commented part

    // Get the indexed face set for editing
    //    SoIndexedFaceSet myIndexedFaceSet = (SoIndexedFaceSet)
    //      Inventor.recreate("com.openinventor.inventor.nodes.SoIndexedFaceSet", root.getChild(0)) ;
    SoIndexedFaceSet myIndexedFaceSet = (SoIndexedFaceSet)root.getChild(0);

    // Get the SoVertexProperty node for editing the material binding
    //    SoVertexProperty myVertexProperty  = (SoVertexProperty) Inventor.recreate
    //      ("com.openinventor.inventor.nodes.SoVertexProperty", myIndexedFaceSet.vertexProperty.getValue()) ;
    SoVertexProperty myVertexProperty  = (SoVertexProperty)myIndexedFaceSet.vertexProperty.getValue();

    //////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE (Inventor 2.1)

    switch(whichBinding) {
    case 0:
      // Set up binding to use a different color for each face
      myVertexProperty.materialBinding.setValue(SoVertexProperty.Bindings.PER_FACE) ;
      break;
    case 1:
      // Set up binding to use a different color at each
      // vertex, BUT, vertices shared between faces will
      // have the same color.
      myVertexProperty.materialBinding.setValue(SoVertexProperty.Bindings.PER_VERTEX_INDEXED) ;
      myIndexedFaceSet.materialIndex.setValues(0, indices);
      break;
    case 2:
      myVertexProperty.materialBinding.setValue(SoVertexProperty.Bindings.PER_FACE_INDEXED) ;
      myIndexedFaceSet.materialIndex.setValues(0, materialIndices);
      break;
    }
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    //  Binding type checkbox
    Checkbox face, vertex_indexed, per_face_indexed;
    JPanel bindingPanel = new JPanel();

    bindingPanel.add(face = new Checkbox("PER_FACE"));
    face.setCheckboxGroup ( bindingGroup );

    bindingPanel.add(vertex_indexed = new Checkbox("PER_VERTEX_INDEXED"));
    vertex_indexed.setCheckboxGroup ( bindingGroup );

    bindingPanel.add(per_face_indexed = new Checkbox("PER_FACE_INDEXED"));
    per_face_indexed.setCheckboxGroup ( bindingGroup );

    bindingGroup.setSelectedCheckbox(face);

    face.addItemListener(this);
    vertex_indexed.addItemListener(this);
    per_face_indexed.addItemListener(this);
    int whichBinding = 0;

    root = makeStellatedDodecahedron();

    setBindingType(whichBinding);

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setName("Material Bindings");
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(bindingPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  @Override
  public void itemStateChanged(ItemEvent e) {

    if (e.getStateChange() == ItemEvent.SELECTED) {
      String checkName = (String)e.getItem();

      if (checkName.equals("PER_FACE")) setBindingType(0);
      if (checkName.equals("PER_VERTEX_INDEXED")) setBindingType(1);
      if (checkName.equals("PER_FACE_INDEXED")) setBindingType(2);
    }

  }

  @Override
  public void actionPerformed(ActionEvent e) {
    /* try {
       int newBase = Integer.valueOf(e.getActionCommand(), 16).intValue();
       symbols.setBase(newBase);
       } catch (NumberFormatException nfe) {
       Toolkit.getDefaultToolkit().beep();
       baseText.select(0, Integer.MAX_VALUE);
       } */

    System.out.println(" actionPerformed ==========");
  }

  public static void main(String argv []) {
    Main example = new Main();
    example.demoMain("Binding");
  }
}
