/*--------------------------------------------------------------
 *  This is an example from the Inventor Mentor
 *  chapter 13, example 5.
 *
 *  Gate engine.
 *  Mouse button presses enable and disable a gate engine.
 *  The gate engine controls an elapsed time engine that
 *  controls the motion of the duck.
 *------------------------------------------------------------*/

package inventor.mentor.gate;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Font;

import javax.swing.JLabel;
import javax.swing.SwingConstants;

import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.engines.SoElapsedTime;
import com.openinventor.inventor.engines.SoGate;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.fields.SoMFFloat;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private IRenderAreaInteractive myRenderArea;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Duck Pond");
  }

  @Override
  public void start() {
    myRenderArea = ViewerComponentsFactory.createRenderAreaInteractive();

    // Setup camera
    SoPerspectiveCamera myCamera = (SoPerspectiveCamera) myRenderArea.getRootSceneGraph().getCamera();
    myCamera.position.setValue(0.0f, -4.0f, 8.0f);
    myCamera.heightAngle.setValue((float)(Math.PI/2.5f));
    myCamera.nearDistance.setValue(1.0f);
    myCamera.farDistance.setValue(15.0f);

    // Rotate scene slightly to get better view
    SoRotationXYZ globalRotXYZ = new SoRotationXYZ();
    globalRotXYZ.axis.setValue(SoRotationXYZ.AxisType.X);
    globalRotXYZ.angle.setValue((float)(Math.PI/9.0f));

    // Pond group
    SoSeparator pond = new SoSeparator();
    SoMaterial cylMaterial = new SoMaterial();
    cylMaterial.diffuseColor.setValue(0.0f, 0.3f, 0.8f);
    SoTranslation cylTranslation = new SoTranslation();
    cylTranslation.translation.setValue(0.0f, -6.725f, 0.0f);
    SoCylinder myCylinder = new SoCylinder();
    myCylinder.radius.setValue(4.0f);
    myCylinder.height.setValue(0.5f);
    {
      pond.addChild(cylMaterial);
      pond.addChild(cylTranslation);
      pond.addChild(myCylinder);
    }

    /////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE  (part 1)

    // Duck group
    SoSeparator duck = new SoSeparator();

    // Read the duck object from a file and add to the group
    SoInput myInput = new SoInput();
    String filename = "$OIVJHOME/data/models/duck.iv";
    if (!myInput.openFile(filename)) {
      System.err.println("Cannot open file " + filename);
      System.exit(1);
    }
    SoSeparator duckObject = SoDB.readAll(myInput);
    if (duckObject == null) {
      System.err.println("Cannot read file " + filename);
      System.exit(1);
    }

    // Set up the duck transformations
    SoRotationXYZ duckRotXYZ = new SoRotationXYZ();
    SoTransform initialTransform = new SoTransform();
    initialTransform.translation.setValue(0.0f, 0.0f, 3.0f);
    initialTransform.scaleFactor.setValue(6.0f, 6.0f, 6.0f);
    {
      duck.addChild(duckRotXYZ);
      duck.addChild(initialTransform);
      duck.addChild(duckObject);
    }

    // Update the rotation value if the gate is enabled.
    SoGate myGate = new SoGate(SoMFFloat.class);
    SoElapsedTime myCounter = new SoElapsedTime();
    myGate.input.connectFrom(myCounter.timeOut);
    duckRotXYZ.axis.setValue(SoRotationXYZ.AxisType.Y);  // rotate about Y axis
    duckRotXYZ.angle.connectFrom(myGate.output);

    // Add an event callback to catch mouse button presses.
    // Each button press will enable or disable the duck motion.
    SoEventCallback myEventCB = new SoEventCallback();
    myEventCB.addEventCallback(SoMouseButtonEvent.class,
                               new MousePressCB(myGate));

    // CODE FOR The Inventor Mentor ENDS HERE
    /////////////////////////////////////////////////////////////

    SoSeparator root = new SoSeparator();
    { // Assemble scene graph
      root.addChild(globalRotXYZ);
      root.addChild(pond);
      root.addChild(duck);
      root.addChild(myEventCB);
    }

    myRenderArea.setSceneGraph(root);

    JLabel label = new JLabel("Click the left mouse button to enable/disable the duck motion",
                              SwingConstants.CENTER);
    label.setFont(new Font(null, 1, 15));

    final Component canvas = myRenderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(canvas);
    add(label, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myRenderArea.dispose();
  }

  /////////////////////////////////////////////////////////////
  // CODE FOR The Inventor Mentor STARTS HERE  (part 2)

  // This routine is called for every mouse button event.
  class MousePressCB extends SoEventCallbackCB {

    private SoGate m_gate;

    public MousePressCB(SoGate gate) {
      m_gate = gate;
    }

    @Override
    public void invoke(SoEventCallback eventCB) {
      SoEvent event = eventCB.getEvent();

      // Check for mouse button being pressed
      if (SoMouseButtonEvent.isButtonPressEvent(event, SoMouseButtonEvent.Buttons.ANY)) {
        // Toggle the gate that controls the duck motion
        if (m_gate.enable.getValue())
          m_gate.enable.setValue(false);
        else
          m_gate.enable.setValue(true);

        eventCB.setHandled();
      }
    }
  }

  // CODE FOR The Inventor Mentor ENDS HERE
  /////////////////////////////////////////////////////////////
}
