/*-----------------------------------------------------------
 *  This is an example from the Inventor Mentor,
 *  chapter 9, example 1.
 *
 *  Printing example.
 *  Read in an Inventor file and display it in ExaminerViewer.  Press
 *  the "p" key and the scene renders into a PostScript
 *  file for printing.
 *----------------------------------------------------------*/

package inventor.mentor.print;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Font;

import javax.swing.JLabel;
import javax.swing.SwingConstants;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoOffscreenRenderer;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example {

  private static final int PRINTER_DPI = 75;
  private String m_filename;
  private String m_exportFilename;
  private String m_exportLabel;
  private int m_type;

  private IViewerExaminer m_myViewer;

  public Main() {
    // set default files name
    m_filename = "$OIVJHOME/data/models/toys/jackInTheBox.iv";

    if (System.getProperty("os.name").indexOf("Windows") != -1) {
      m_exportFilename = "offscreen.bmp";
      m_exportLabel = "Bitmap";
      m_type = 0;
    }
    else {
      m_exportFilename = "offscreen.ps";
      m_exportLabel = "PostScript";
      m_type = 1;
    }
  }

  public static void main(String[] args) {
    Main example = new Main();

    if (args.length >= 1) {
      example.m_filename = args[0];
      if (args.length >= 2)
        example.m_exportFilename = args[1];
    }

    example.demoMain("Print to " + example.m_exportLabel);
  }

  @Override
  public void start() {
    m_myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Make a scene containing an event callback node
    SoSeparator root = new SoSeparator();
    SoEventCallback eventCB = new SoEventCallback();

    // Read the geometry from a file and add to the scene
    SoInput myInput = new SoInput();
    if (!myInput.openFile(m_filename)) {
      System.err.println("Could not open file " + m_filename);
      System.exit(1);
    }

    SoSeparator geomObject = SoDB.readAll(myInput);
    if (geomObject == null) {
      System.err.println("Could not generate scene graph from file " + m_filename) ;
      System.exit(1);
    }

    SbColor bkgColor = new SbColor(.6f, .7f, .9f);
    SoGradientBackground background = new SoGradientBackground();
    background.color0.setValue(bkgColor);
    background.color1.setValue(bkgColor);

    { // Assemble scene graph
      root.addChild(background);
      root.addChild(eventCB);
      root.addChild(geomObject);
    }

    m_myViewer.setSceneGraph(root);
    m_myViewer.viewAll();

    eventCB.addEventCallback(SoKeyboardEvent.class,
                             new ProcessKeyEvents(m_myViewer),
                             null);

    JLabel jl = new JLabel("To print the scene : press the 'p' key",
                           SwingConstants.CENTER);
    jl.setFont(new Font(null, 1 ,15));

    final Component component = m_myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(jl, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_myViewer.dispose();
  }

  //////////////////////////////////////////////////////////////
  // CODE FOR The Inventor Mentor STARTS HERE

  private static boolean printToFile(String file, IViewerExaminer viewer, int printerDPI, int type) {
    // Calculate size of the images in inches which is equal to
    // the size of the viewport in pixels divided by the number
    // of pixels per inch of the screen device.  This size in
    // inches will be the size of the Postscript image that will
    // be generated.
    SbViewportRegion vp = new SbViewportRegion(viewer.getRenderArea().getComponent().getSize());
    SbVec2s imagePixSize = vp.getViewportSizePixels();
    SbVec2f imageInches = new SbVec2f();

    float pixPerInch = SoOffscreenRenderer.getScreenPixelsPerInch();
    imageInches.setValue(imagePixSize.getX() / pixPerInch,
                         imagePixSize.getY() / pixPerInch);

    // The resolution to render the scene for the printer
    // is equal to the size of the image in inches times
    // the printer DPI;
    SbVec2s postScriptRes = new SbVec2s();
    postScriptRes.setValue((short)(imageInches.getX()*printerDPI),
                           (short)(imageInches.getY()*printerDPI));

    // Create a viewport to render the scene into.
    SbViewportRegion myViewport = new SbViewportRegion();
    myViewport.setWindowSize(postScriptRes);
    myViewport.setPixelsPerInch(printerDPI);

    // Render the scene
    SoOffscreenRenderer myRenderer = new SoOffscreenRenderer(myViewport);
    myRenderer.setBackgroundColor(new SbColor(.6f, .7f, .9f));

    if (!myRenderer.render(viewer.getRenderArea().getRootSceneGraph()))
      return false;

    // Generate PostScript or BMP and write it to the given file
    if (type == 0) {
      try {
        myRenderer.writeToBMP(file);
      }
      catch (UnsupportedOperationException e) {
        System.err.println();
        e.printStackTrace();
        return false ;
      }
    } else
      myRenderer.writeToPostScript(file);

    System.gc();
    return true;
  }

  // CODE FOR The Inventor Mentor ENDS HERE
  //////////////////////////////////////////////////////////////


  // inner classes
  class ProcessKeyEvents extends SoEventCallbackCB {
    private IViewerExaminer m_viewer;

    public ProcessKeyEvents(IViewerExaminer viewer) {
      m_viewer = viewer;
    }

    @Override
    public void invoke(SoEventCallback event) {
      if (SoKeyboardEvent.isKeyPressEvent(event.getEvent(), SoKeyboardEvent.Keys.P)) {
        System.out.println("Printing scene...");

        System.out.flush();

        if (!printToFile(m_exportFilename, m_viewer, PRINTER_DPI, m_type)) {
          System.err.println("Cannot print image");
          System.out.flush();
          event.setHandled();
        }
        else {
          System.out.println("  ...done printing.");
          System.out.flush();
          event.setHandled();
        }
      }
    }
  }
}
