/*-------------------------------------------------------------
 *  This is an example from The Inventor Mentor,
 *  chapter 10, example 5.
 *
 *  The scene graph has a sphere and a text 3D object.
 *  A selection node is placed at the top of the scene graph.
 *  When an object is selected, a selection callback is called
 *  to change the material color of that object.
 *------------------------------------------------------------*/

package inventor.mentor.selectionCB;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.misc.callbacks.SoSelectionPathCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

// global data
  static final float REDDISH[] = {1.0f, 0.2f, 0.2f};  // Color when selected
  static final float WHITE[] = {0.8f, 0.8f, 0.8f};    // Color when not selected

  private IRenderAreaInteractive myRenderArea;
  SoMaterial textMaterial;
  SoMaterial sphereMaterial;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("My Selection Callback");
  }

  @Override
  public void start() {
    myRenderArea = ViewerComponentsFactory.createRenderAreaInteractive();

    // Create and set up the selection node
    SoSelection selectionRoot = new SoSelection();
    selectionRoot.policy.setValue(SoSelection.Policies.SINGLE);
    selectionRoot.addSelectionCallback(new MySelectionCB());
    selectionRoot.addDeselectionCallback(new MyDeselectionCB());

    // Create the scene graph
    SoSeparator root = new SoSeparator();

    // Add a sphere node
    SoSeparator sphereRoot = new SoSeparator();
    SoTransform sphereTransform = new SoTransform();
    sphereTransform.translation.setValue(17.0f, 17.0f, 0.0f);
    sphereTransform.scaleFactor.setValue(8.0f, 8.0f, 8.0f);
    sphereMaterial = new SoMaterial();
    sphereMaterial.diffuseColor.setValue(0.8f, 0.8f, 0.8f);
    {
      sphereRoot.addChild(sphereTransform);
      sphereRoot.addChild(sphereMaterial);
      sphereRoot.addChild(new SoSphere());
    }

    // Add a text node
    SoSeparator textRoot = new SoSeparator();
    SoTransform textTransform = new SoTransform();
    textTransform.translation.setValue(0.0f, -1.0f, 0.0f);
    textMaterial = new SoMaterial();
    textMaterial.diffuseColor.setValue(0.8f, 0.8f, 0.8f);
    SoPickStyle textPickStyle = new SoPickStyle();
    textPickStyle.style.setValue(SoPickStyle.Styles.BOUNDING_BOX);
    SoText3 myText = new SoText3();
    myText.string.setValue("rhubarb");
    {
      textRoot.addChild(textTransform);
      textRoot.addChild(textMaterial);
      textRoot.addChild(textPickStyle);
      textRoot.addChild(myText);
    }

    { // Assemble scene graph
      root.addChild(sphereRoot);
      root.addChild(textRoot);
      selectionRoot.addChild(root);
    }

    myRenderArea.setSceneGraph(selectionRoot);

    final Component canvas = myRenderArea.getComponent();

    // Make the camera see the whole scene
    SbViewportRegion myViewport =  new SbViewportRegion(canvas.getSize());
    myRenderArea.viewAll(myViewport);

    canvas.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    myRenderArea.dispose();
  }

  class MySelectionCB extends SoSelectionPathCB {
    @Override
    public void invoke(SoPath p) {
      if (p.regular.getTail() instanceof SoText3) {
        textMaterial.diffuseColor.setValue(REDDISH);
      } else if (p.regular.getTail() instanceof SoSphere) {
        sphereMaterial.diffuseColor.setValue(REDDISH);
      }
    }
  }

  class MyDeselectionCB extends SoSelectionPathCB {
    @Override
    public void invoke(SoPath p) {
      if (p.regular.getTail() instanceof SoText3) {
        textMaterial.diffuseColor.setValue(WHITE);
      } else if (p.regular.getTail() instanceof SoSphere) {
        sphereMaterial.diffuseColor.setValue(WHITE);
      }
    }
  }
}
