/*--------------------------------------------------------------
 *  This is an example from the Inventor Mentor
 *  chapter 13, example 4.
 *
 *  Time counter engine.
 *  The output from an time counter engine is used to control
 *  horizontal and vertical motion of a figure object.
 *  The resulting effect is that the figure jumps across
 *  the screen.
 *------------------------------------------------------------*/

package inventor.mentor.timeCounter;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.engines.SoComposeVec3f;
import com.openinventor.inventor.engines.SoTimeCounter;
import com.openinventor.inventor.nodes.SoPerspectiveCamera;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.nodes.SoTranslation;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  private IRenderAreaInteractive myRenderArea;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Jumping Man");
  }

  @Override
  public void start() {
    myRenderArea = ViewerComponentsFactory.createRenderAreaInteractive();

    // Setup camera
    SoPerspectiveCamera myCamera = (SoPerspectiveCamera) myRenderArea.getRootSceneGraph().getCamera();
    myCamera.position.setValue(-8.0f, -7.0f, 20.0f);
    myCamera.heightAngle.setValue((float)(Math.PI/2.5f));
    myCamera.nearDistance.setValue(15.0f);
    myCamera.farDistance.setValue(25.0f);

    //////////////////////////////////////////////////////////////
    // CODE FOR The Inventor Mentor STARTS HERE

    // Set up transformations
    SoTranslation jumpTranslation = new SoTranslation();
    SoTransform initialTransform = new SoTransform();
    initialTransform.translation.setValue(-20.0f, 0.0f, 0.0f);
    initialTransform.scaleFactor.setValue(40.0f, 40.0f, 40.0f);
    initialTransform.rotation.setValue(new SbVec3f(1.0f, 0.0f, 0.0f),
                                       (float)(Math.PI/2.0f));

    // Read the man object from a file and add to the scene
    SoInput myInput = new SoInput();
    String filename = "$OIVJHOME/data/models/jumpyMan.iv";
    if (!myInput.openFile(filename)) {
      System.err.println("Cannot open file " + filename);
      System.exit(1);
    }

    SoSeparator manObject = SoDB.readAll(myInput);
    if (manObject == null) {
      System.err.println("Cannot read file " + filename);
      System.exit(1);
    }

    // Create two counters, and connect to X and Y translations.
    // The Y counter is small and high frequency.
    // The X counter is large and low frequency.
    // This results in small jumps across the screen,
    // left to right, again and again and again and ....
    SoTimeCounter jumpHeightCounter = new SoTimeCounter();
    SoTimeCounter jumpWidthCounter = new SoTimeCounter();
    SoComposeVec3f jump = new SoComposeVec3f();

    jumpHeightCounter.max.setValue((short)4);
    jumpHeightCounter.frequency.setValue(1.5f);
    jumpWidthCounter.max.setValue((short)40);
    jumpWidthCounter.frequency.setValue(0.15f);

    jump.x.connectFrom(jumpWidthCounter.output);
    jump.y.connectFrom(jumpHeightCounter.output);
    jumpTranslation.translation.connectFrom(jump.vector);

    // CODE FOR The Inventor Mentor ENDS HERE
    //////////////////////////////////////////////////////////////

    SoSeparator root = new SoSeparator();
    { // Assemble scene graph
      root.addChild(jumpTranslation);
      root.addChild(initialTransform);
      root.addChild(manObject);
    }

    myRenderArea.setSceneGraph(root);

    final Component canvas = myRenderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    myRenderArea.dispose();
  }
}
