package inventor.mentor.trimSurf;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.nodes.* ;
import com.openinventor.inventor.viewercomponents.SoCameraInteractor;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

/**
  *  This is an example from the Inventor Mentor Programming Guide,
  *  chapter 8, example 4.
  *
  *  This example creates and displays a Bezier Surface
  *  with trim curves.  The surface is identical to the
  *  surface in example 08.3.BezSurf.  The SoNurbsProfile
  *  class is used to define the trims.
  *
  *  One trim curve makes a counter-clockwise square around
  *  the whole surface.  Two other trim curves are combined
  *  end to end to cut a hole from the surface.  The outside
  *  trim curve and the first inside trim curve are both
  *  order 2 curves and are therefore sets of straight lines
  *  in parameter space.  The second inside curve is a
  *  Bezier curve.
  *
  * @author Loic Vigneras
  * @author Patrick Vigneras
  * @author Jean Kinda
  */
public class Main extends Example
{

  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("TrimSurf");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Create the scene graph for the floor texture
    SoTexture2 floor_texture = new SoTexture2();
    floor_texture.filename.setValue("$OIVJHOME/data/textures/rgb/oak.rgb");

    // Create the scene graph for the floor
    URL url_floor = getURL(SoPreferences.getValue("OIVJHOME") + "/data/models/Floor.iv");
    if (url_floor == null) return;

    SoSeparator floor = new SoSeparator();
    SoTranslation xlate = new SoTranslation();
    SoScale scale = new SoScale();

    SoInput in = new SoInput();
    in.openURL(url_floor);
    SoNode result = SoDB.readNode(in);
    xlate.translation.setValue(new SbVec3f(-12, -5, -5));
    scale.scaleFactor.setValue(new SbVec3f(2, 1, 2));
    {
      floor.addChild(xlate);
      floor.addChild(scale);
      floor.addChild(result);
    }

    // Create the scene graph for the carpet;
    SoSeparator carpet = new SoSeparator();
    SoSeparator surf = makeSurface();
    SoTexture2 tex = new SoTexture2();

    tex.filename.setValue("$OIVJHOME/data/textures/rgb/diamondRug.rgb");
    {
      carpet.addChild(tex);
      carpet.addChild(surf);
    }

    // Create the scene graph for the carpet's shadow
    SoSeparator shadow = new SoSeparator();
    SoLightModel shmdl = new SoLightModel();
    SoMaterial shmtl = new SoMaterial();
    SoBaseColor shclr = new SoBaseColor();
    SoTranslation shxl = new SoTranslation();
    SoScale shscl = new SoScale();

    shmdl.model.setValue(SoLightModel.Models.BASE_COLOR);
    shclr.rgb.setValue(new SbColor(0.21F, 0.15F, 0.09F));
    shmtl.transparency.setValue(0.3F);
    shxl.translation.setValue(new SbVec3f(0, -4.9F, 0));
    shscl.scaleFactor.setValue(new SbVec3f(1, 0.01F, 1));
    {
      shadow.addChild(shmtl);
      shadow.addChild(shmdl);
      shadow.addChild(shclr);
      shadow.addChild(shxl);
      shadow.addChild(shscl);
      shadow.addChild(surf);
    }

    SoRotation rot = new SoRotation();
    rot.rotation.setValue(new SbRotation(new SbVec3f(0, 1, 0), (float)(Math.PI/2)));

    SoSeparator root = new SoSeparator();
    { // assemble scene graph
      root.addChild(rot);
      root.addChild(carpet);
      root.addChild(floor_texture);
      root.addChild(floor);
      root.addChild(shadow);
    }

    myViewer.setSceneGraph(root);

    SoCameraInteractor interactor = myViewer.getRenderArea().getRootSceneGraph().getCameraInteractor();
    interactor.setPosition(new SbVec3f(-8.5F, 13, 23));
    interactor.pointAt(new SbVec3f(-2, -2, -4));

    final Component component = myViewer.getComponent();
    component.setName("Trimmed Nurbs Surface");
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  // Create the nodes needed for the Bezier patch
  // and its trim curves.
  SoSeparator makeSurface() {
    // The array of trim coordinates
    SbVec2f tpts[] = {
      new SbVec2f(0, 0),
      new SbVec2f(1, 0),
      new SbVec2f(1, 1),
      new SbVec2f(0, 1),
      new SbVec2f(0.2F, 0.2F),
      new SbVec2f(0.2F, 0.7F),
      new SbVec2f(0.9F, 0.7F),
      new SbVec2f(0.9F, 0.2F),
      new SbVec2f(0.7F, 0),
      new SbVec2f(0.4F, 0.8F)};

    // The 16 coordinates defining the Bezier surface.
    float pts[][] = {
      {-4.5F, -2,  8},
      {-2,  1,  8},
      { 2, -3,  6},
      { 5, -1,  8},
      {-3,  3,  4},
      { 0, -1,  4},
      { 1, -1,  4},
      { 3,  2,  4},
      {-5, -2, -2},
      {-2, -4, -2},
      { 2, -1, -2},
      { 5,  0, -2},
      {-4.5F,  2, -6},
      {-2, -4, -5},
      { 2,  3, -5},
      { 4.5F, -2, -6}};

    // The 3 knot vectors for the 3 trim curves.
    float tknots1[] = {0, 0, 1, 2, 3, 4, 4};
    float tknots2[] = {0, 0, 1, 2, 3, 3};
    float tknots3[] = {0, 0, 0, 0, 1, 1, 1, 1};

    // The Bezier knot vector for the surface.
    // This knot vector is used in both the U and
    // V directions.
    float knots[] = {0, 0, 0, 0, 1, 1, 1, 1};

    SoSeparator surfSep = new SoSeparator();

    // Define the Bezier surface including the control
    // points, trim curve, and a complexity.
    SoComplexity complexity = new SoComplexity();
    SoCoordinate3 controlPts = new SoCoordinate3();
    SoNurbsSurface surface = new SoNurbsSurface();

    complexity.value.setValue(0.7F);
    controlPts.point.setValues(0, pts);
    surface.numUControlPoints.setValue(4);
    surface.numVControlPoints.setValue(4);
    surface.uKnotVector.setValues(0, knots);
    surface.vKnotVector.setValues(0, knots);
    surfSep.addChild(complexity);
    surfSep.addChild(controlPts);

    SoProfileCoordinate2 trimPts = new SoProfileCoordinate2();
    SoNurbsProfile nTrim1 = new SoNurbsProfile();
    SoNurbsProfile nTrim2 = new SoNurbsProfile();
    SoNurbsProfile nTrim3 = new SoNurbsProfile();

    trimPts.point.setValues(0, tpts);
    int[] trimInds1 = new int[5];
    trimInds1[0] = 0;
    trimInds1[1] = 1;
    trimInds1[2] = 2;
    trimInds1[3] = 3;
    trimInds1[4] = 0;
    nTrim1.index.setValues(0, trimInds1);
    nTrim1.knotVector.setValues(0, tknots1);

    int[] trimInds2 = new int[4];
    trimInds2[0] = 4;
    trimInds2[1] = 5;
    trimInds2[2] = 6;
    trimInds2[3] = 7;
    nTrim2.linkage.setValue(SoProfile.Profiles.START_NEW);
    nTrim2.index.setValues(0, trimInds2);
    nTrim2.knotVector.setValues(0, tknots2);

    int[] trimInds3 = new int[4];
    trimInds3[0] = 7;
    trimInds3[1] = 8;
    trimInds3[2] = 9;
    trimInds3[3] = 4;
    nTrim3.linkage.setValue(SoProfile.Profiles.ADD_TO_CURRENT);
    nTrim3.index.setValues(0, trimInds3);
    nTrim3.knotVector.setValues(0, tknots3);

    surfSep.addChild(trimPts);
    surfSep.addChild(nTrim1);
    surfSep.addChild(nTrim2);
    surfSep.addChild(nTrim3);
    surfSep.addChild(surface);

    return surfSep;
  }

  private URL getURL(String fileName) {
    URL url = null;
    try {
      url = new URL("file:"+(new File(fileName)).getAbsolutePath()) ;
    }
    catch (MalformedURLException exc) {
      System.err.println(exc + " : (" + url + ") " + fileName + " invalid URL");
      url = null;
    }
    catch (Exception e) {
      System.err.println(e + " : cant load File "+ fileName);
      url = null;
    }
    return url;
  }
}

