package inventor.sample.annotation2D_2;

import java.awt.BorderLayout;
import java.awt.Component;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2i32;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.actions.SoAction;
import com.openinventor.inventor.elements.SoViewportRegionElement;
import com.openinventor.inventor.misc.SoState;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.nodes.SoCallback.CB;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private IViewerExaminer myViewer;

  public class PixelCameraCB implements CB
  {
    // ----------------------------------------------------------------------
    // Callback callback that modifies the camera
    //
    // Gets the current viewport from the traversal state and, if the
    // viewport has changed, modifies the associated orthographic camera
    // so we have a "pixel space" view volume in X and Y.
    //
    // - action : The action currently traversing the scene graph.
    // Could be a render action, bounding box action, etc.
    // In this case we do the same thing for all actions.
    //

    private final PixelCameraInfo m_info;

    public PixelCameraCB(PixelCameraInfo info)
    {
      m_info = info;
    }

    // ----------------------------------------------------------------------
    // Convenience function to set up a simulated "pixel space" camera
    //
    // - parent : Group node to add the camera under
    // - invertX : X will be 0..N-1 if False, else -(N-1)..0
    // - invertY : Y will be 0..N-1 if False, else -(N-1)..0
    //
    @Override
    public void invoke(SoAction action)
    {
      // Get the current traversal state
      SoState state = action.getState();
      if ( state == null )
        return;

      // Get the current viewport from the traversal state
      SbViewportRegion vpRegion = SoViewportRegionElement.get(state);
      SbVec2i32 viewport = vpRegion.getViewportSizePixelsi32();

      // Return if viewport same as last time (no need to change camera)
      if ( viewport.equals(m_info.m_lastViewport) )
        return;

      // Save viewport and compute aspect ratio
      m_info.m_lastViewport.setValue(viewport);
      int[] vpValue = viewport.getValue();
      int vpWidth = vpValue[0];
      int vpHeight = vpValue[1];
      float vpAspect = (float) vpWidth / vpHeight;

      // First disable notification on the camera node
      // (we're already in the middle of a traversal)
      //
      // Set the view volume height to number of pixels in Y.
      // Set the aspect ratio the same as the viewport
      // (effectively setting view volume width to pixels in X).
      SoOrthographicCamera camera = m_info.m_camera;
      camera.enableNotify(false);
      camera.height.setValue(vpHeight);
      camera.aspectRatio.setValue(vpAspect);

      // Compute an offset that will make the view volume 0..N-1
      // (or the reverse if inverted).
      //
      // Remember that Inventor first computes a view volume centered around
      // 0,0f I.e. -N/2 to N/2, where N is the width or the height.
      // Then it translates the view volume by camera.position. So...
      // - If the viewport width is even, for example 400, we need the left
      // edge at zero, the center at 199.5 and the right edge at 399.
      // - If the viewport width is odd, for example 399, we need the left
      // edge at zero, the center at 199 and the right edge at 398.
      //
      // If the invert flag is set, that axis will be -(N-1)..0 instead.
      // This allows positioning geometry relative to the right/top edge.
      float xRadius = 0.5f * (vpWidth - 1);
      float yRadius = 0.5f * (vpHeight - 1);
      if ( m_info.m_invertX )
        xRadius = -xRadius;
      if ( m_info.m_invertY )
        yRadius = -yRadius;
      camera.position.setValue(new SbVec3f(xRadius, yRadius, 2));
      camera.enableNotify(true);

      //System.out.println("--- Camera " + camera + " updated, vp = " + vpWidth + " " + vpHeight);
    }

  }

  // //////////////////////////////////////////////////////////////////////
  //
  // Declarations and code to simulate a "pixel space camera"
  //
  // It's simulated because we're really using a standard orthographic
  // camera plus a callback node that modifies the view volume to match
  // the current pixel viewport dimensions. In other words, each pixel
  // space camera is
  //
  // Actually creating a new kind of camera would be more elegant, but
  // also much more complicated and involving a lot of stuff that's not
  // directly relevant to this example.

  // ----------------------------------------------------------------------
  // Information an instance of pixelSpaceCamera needs
  // - The ortho camera that will be modified
  // - Invert flags for X and Y
  // If false (default), the range of values is 0..N-1 pixels.
  // Else the range is -(N-1)..0, which allows geometry to be
  // positioned relative to the right and/or top edge of the window.
  // - Last viewport (so we don't modify camera unless there is a change).
  class PixelCameraInfo
  {
    public PixelCameraInfo()
    {
      m_camera = new SoOrthographicCamera();
      m_invertX = false;
      m_invertY = false;
      m_lastViewport = new SbVec2i32(0, 0);
    }

    public SoOrthographicCamera m_camera; // Ortho camera we will modify
    public boolean m_invertX; // 0..N-1 if false, else -(N-1)..0
    public boolean m_invertY; // ditto
    public SbVec2i32 m_lastViewport; //
  }

  void addPixelSpaceCamera(SoGroup parent, boolean invertX, boolean invertY)
  {
    // Create an orthographic camera
    SoOrthographicCamera cam2d = new SoOrthographicCamera();

    // Set the viewportMapping to LEAVE_ALONE because we're going to
    // make the view volume aspect match the viewport ourselves.
    cam2d.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);

    // We're going to put the camera position at +2 in Z, so these
    // near and far distances make the view volume 2 units deep in
    // Z centered around zero. Obviously we're assuming all the
    // 2D annotation geometry really is flat!
    cam2d.nearDistance.setValue(1);
    cam2d.farDistance.setValue(3);

    // Setup info for pixel space camera callback
    PixelCameraInfo info = new PixelCameraInfo();
    info.m_camera = cam2d;
    info.m_invertX = invertX;
    info.m_invertY = invertY;

    // Create callback node to modify the camera.
    // Set it to call pixelCameraCB with info
    SoCallback cam2dCB = new SoCallback();
    cam2dCB.setCallback(new PixelCameraCB(info));

    // Add callback and camera to scene graph (callback before camera)
    parent.addChild(cam2dCB);
    parent.addChild(cam2d);
  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    // Root node of scene graph
    // 3D (application) and 2D (annotation) scene graphs below this.
    SoSeparator root = new SoSeparator();

    // Try to read the file in 3D Root separator
    SoSeparator sep3d = readSceneGraph("$OIVJHOME/data/models/spongetri4.iv");

    // Add the 3D scene to the scene graph.
    root.addChild(sep3d);

    // ----- 2D Annotation Setup ---

    // Font settings and material for 2D annotations.
    SoFont font1 = new SoFont();
    font1.name.setValue("Arial");
    font1.size.setValue(20);
    font1.renderStyle.setValue(SoFont.RenderStyles.TEXTURE);
    SoMaterial matl1 = new SoMaterial();
    matl1.diffuseColor.setValue(new SbColor(1, 1, 1));

    // Create 2D scene root and build basic 2D scene
    SoSeparator sep2d = new SoSeparator();
    root.addChild(sep2d);
    sep2d.addChild(font1);
    sep2d.addChild(matl1);

    // Create 2D text annotation in upper left corner of window.
    //
    // We'll put this under its own separator so the camera and
    // translation don't affect anything else.
    //
    // First we'll create a "pixel space" camera with the Y axis inverted
    // so we can position geometry relative to the top edge of the window.
    //
    // Next we need a translation node to position the text 5 pixels in
    // from the left and 20 pixels down from the top. (20 pixels down
    // because font height is 20 meaning text is slightly less.)
    SoSeparator textSep1 = new SoSeparator();
    addPixelSpaceCamera(textSep1, false, true); // Y is inverted
    SoTranslation tran1 = new SoTranslation();
    tran1.translation.setValue(new SbVec3f(5, -20, 0)); // Upper left corner
    textSep1.addChild(tran1);
    SoText2 text1 = new SoText2();
    text1.string.setValue("Upper Left");
    textSep1.addChild(text1);
    sep2d.addChild(textSep1);

    // Create 2D text annotation in lower left corner of window.
    // Same structure as previous, except neither axis is inverted.
    SoSeparator textSep2 = new SoSeparator();
    addPixelSpaceCamera(textSep2, false, false); // Neither X nor Y is
    // inverted
    SoTranslation tran2 = new SoTranslation();
    tran2.translation.setValue(new SbVec3f(5, 5, 0)); // Lower left corner
    textSep2.addChild(tran2);
    SoText2 text2 = new SoText2();
    text2.string.setValue("Lower Left");
    textSep2.addChild(text2);
    sep2d.addChild(textSep2);

    // Create 2D text annotation in lower right corner of window.
    // Same structure but X axis is inverted to allow positioning
    // relative to the right edge of the window, and also the
    // text justification is set to RIGHT so we can position the
    // text without knowing exactly how wide it is.
    SoSeparator textSep3 = new SoSeparator();
    addPixelSpaceCamera(textSep3, true, false); // X is inverted
    SoTranslation tran3 = new SoTranslation();
    tran3.translation.setValue(new SbVec3f(-5, 5, 0)); // Lower right corner
    textSep3.addChild(tran3);
    SoText2 text3 = new SoText2();
    text3.string.setValue("Lower Right");
    text3.justification.setValue(SoText2.Justifications.RIGHT);
    textSep3.addChild(text3);
    sep2d.addChild(textSep3);

    // Create 2D text annotation in upper right corner of window.
    // Same structure as previous except both axes are inverted.
    SoSeparator textSep4 = new SoSeparator();
    addPixelSpaceCamera(textSep4, true, true); // Both axes inverted
    SoTranslation tran4 = new SoTranslation();
    tran4.translation.setValue(new SbVec3f(-5, -20, 0)); // Upper right corner
    textSep4.addChild(tran4);
    SoText2 text4 = new SoText2();
    text4.string.setValue("Upper Right");
    text4.justification.setValue(SoText2.Justifications.RIGHT);
    textSep4.addChild(text4);
    sep2d.addChild(textSep4);

    // Finally... We don't want the extent of the annotations to affect
    // a "viewAll" of the application geometry, so we'll reset the
    // bounding box to empty after traversing the 2D annotation stuff.
    SoResetTransform resetBbox = new SoResetTransform();
    resetBbox.whatToReset.setValue(SoResetTransform.ResetTypes.BBOX);
    sep2d.addChild(resetBbox);

    // Load the scene graph and adjust the 3D camera to view entire scene.
    // Have to do this "manually" because we have our own camera.
    // Don't forget to save this as the Home position!
    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private static SoSeparator readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("2D Annotation");
  }
}
