package inventor.sample.indexedTexture;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.Panel;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.ShortBuffer;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JTextField;

import com.openinventor.inventor.SbVec2s;
import com.openinventor.inventor.SoOutput;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoWriteAction;
import com.openinventor.inventor.fields.SoSFArray.DataTypes;
import com.openinventor.inventor.nodes.SoColorMap;
import com.openinventor.inventor.nodes.SoCube;
import com.openinventor.inventor.nodes.SoIndexedTexture2;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoTexture;
import com.openinventor.inventor.nodes.SoTexture2;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;

import util.Example;
import util.ViewerComponentsFactory;


public class Main extends Example
{

  private IViewerExaminer m_myViewer;

	private SoSeparator m_root;
	private SoColorMap m_colorMap;
	private SoIndexedTexture2 m_indexedTexture;

	private JTextField m_dataDimText;
	private JLabel m_dataRangeLabel;
	private JTextField m_dataMinText;
	private JTextField m_dataMaxText;
	private JComboBox<String> m_colormapTypeCombo;
	private JTextField m_colormapDimText;

	private JTextField m_colormapMinText;
	private JTextField m_colormapMaxText;

	private int m_dataType;
	private short m_dataDim;
	private ByteBuffer m_data;
	private int m_colormapType;
	private short m_colormapDim;

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Indexed Texture");
  }

	public Main() {}

  @Override
  public void start() {
	    m_myViewer = ViewerComponentsFactory.createViewerExaminer();

		// build scenegraph

		m_root = new SoSeparator();

		m_colorMap = new SoColorMap();
  		m_root.addChild(m_colorMap);

		m_indexedTexture = new SoIndexedTexture2();
		m_indexedTexture.wrapS.setValue(SoTexture.WrapType.CLAMP_TO_EDGE);
		m_indexedTexture.wrapT.setValue(SoTexture.WrapType.CLAMP_TO_EDGE);
		m_indexedTexture.minFilter.setValue(SoTexture.Filters.LINEAR);
		m_indexedTexture.magFilter.setValue(SoTexture2.Filters.LINEAR);
		m_root.addChild(m_indexedTexture);

		m_root.addChild(new SoCube());


		// build interface

		m_myViewer.setSceneGraph(m_root);
		m_myViewer.viewAll();
		m_myViewer.getRenderArea().setTransparencyType(SoGLRenderAction.TransparencyTypes.SORTED_TRIANGLES_BLEND);


		// options panel
		Panel optionsPanel = new Panel(new GridLayout(0,2));


		optionsPanel.add( new JLabel("Data type:") );
		JComboBox<String> dataTypeCombo = new JComboBox<>();
		dataTypeCombo.addItem("Byte");
		dataTypeCombo.addItem("Short");
		dataTypeCombo.addItem("Integer");
		dataTypeCombo.addItem("Float");
		dataTypeCombo.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_dataType = ((JComboBox<?>)e.getSource()).getSelectedIndex();
				generateData();
			}
		});
		optionsPanel.add( dataTypeCombo );

		optionsPanel.add( new JLabel("Data dim:") );
		m_dataDimText = new JTextField();
		m_dataDimText.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_dataDim = Short.parseShort(m_dataDimText.getText());
				generateData();
			}
		});
		optionsPanel.add( m_dataDimText );

		optionsPanel.add( new JLabel("Real data range:") );
		m_dataRangeLabel = new JLabel("");
		optionsPanel.add( m_dataRangeLabel );

		optionsPanel.add( new JLabel("Data Min:") );
		m_dataMinText = new JTextField();
		m_dataMinText.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_indexedTexture.minValue.setValue(Float.parseFloat(m_dataMinText.getText()));
			}
		});
		optionsPanel.add( m_dataMinText );

		optionsPanel.add( new JLabel("Data Max:") );
		m_dataMaxText = new JTextField();
		m_dataMaxText.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_indexedTexture.maxValue.setValue(Float.parseFloat(m_dataMaxText.getText()));
			}
		});
		optionsPanel.add( m_dataMaxText );


		optionsPanel.add( new JLabel("Rescale Tex Coords:") );
		JCheckBox rescaleButton = new JCheckBox("");
		rescaleButton.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_indexedTexture.rescaleTexCoord.setValue(
						((JCheckBox)e.getSource()).isSelected());
			}
		});
		optionsPanel.add( rescaleButton );

		optionsPanel.add( new JLabel("Predefined Colormap:") );
		JComboBox<String> colormapCombo = new JComboBox<>();
		colormapCombo.addItem("NONE");
		colormapCombo.addItem("GRAY");
		colormapCombo.addItem("TEMPERATURE");
		colormapCombo.addItem("PHYSICS");
		colormapCombo.addItem("STANDARD");
		colormapCombo.addItem("GLOW");
		colormapCombo.addItem("BLUE_RED");
		colormapCombo.addItem("SEISMIC");
		colormapCombo.addItem("BLUE_WHITE_RED");
		colormapCombo.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				int index = ((JComboBox<?>)e.getSource()).getSelectedIndex();
				m_colorMap.predefinedColorMap.setValue(index);
				m_colormapTypeCombo.setEnabled(index==0);
				m_colormapDimText.setEnabled(index==0);
			}
		});
		optionsPanel.add( colormapCombo );


		optionsPanel.add( new JLabel("Colormap type:") );
		m_colormapTypeCombo = new JComboBox<>();
		m_colormapTypeCombo.addItem("LUMINANCE");
		m_colormapTypeCombo.addItem("ALPHA");
		m_colormapTypeCombo.addItem("LUMINANCE_ALPHA");
		m_colormapTypeCombo.addItem("RGB");
		m_colormapTypeCombo.addItem("RGB_ALPHA");
		m_colormapTypeCombo.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_colormapType = ((JComboBox<?>)e.getSource()).getSelectedIndex();
				generateColormap();
			}
		});
		optionsPanel.add( m_colormapTypeCombo );

		optionsPanel.add( new JLabel("Colormap dim:") );
		m_colormapDimText = new JTextField();
		m_colormapDimText.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_colormapDim = Short.parseShort(m_colormapDimText.getText());
				generateColormap();
			}
		});
		optionsPanel.add( m_colormapDimText );


		optionsPanel.add( new JLabel("Colormap Min:") );
		m_colormapMinText = new JTextField();
		m_colormapMinText.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_colorMap.min.setValue(Float.parseFloat(m_colormapMinText.getText()));
			}
		});
		optionsPanel.add( m_colormapMinText );

		optionsPanel.add( new JLabel("Colormap Max:") );
		m_colormapMaxText = new JTextField();
		m_colormapMaxText.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_colorMap.max.setValue(Float.parseFloat(m_colormapMaxText.getText()));
			}
		});
		optionsPanel.add( m_colormapMaxText );

		optionsPanel.add( new JLabel("Write Data On Save:") );
		JCheckBox saveDataButton = new JCheckBox("");
		saveDataButton.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				m_indexedTexture.imageIndex.setNeverWrite(
						! ((JCheckBox)e.getSource()).isSelected());
			}
		});
		optionsPanel.add( saveDataButton );

		optionsPanel.add( new JLabel("") );
		JButton writeButton = new JButton("Write scenegraph");
		writeButton.addActionListener(new ActionListener() {
			@Override
      public void actionPerformed(ActionEvent e) {
				SoOutput file = new SoOutput();
				file.openFile("output.iv");
				SoWriteAction wa = new SoWriteAction(file);
				wa.apply(m_root);
				file.closeFile();
			}
		});
		optionsPanel.add( writeButton );

    final Component component = m_myViewer.getComponent();
    component.setName("Indexed Texture");
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(optionsPanel, BorderLayout.SOUTH);

		// init

		m_dataType = 0;
		m_dataDim = 200;
		dataTypeCombo.setSelectedIndex(m_dataType);
		m_dataDimText.setText(""+m_dataDim);

		m_dataMinText.setText("-100");
		m_dataMaxText.setText("100");
		m_indexedTexture.minValue.setValue(-100);
		m_indexedTexture.maxValue.setValue(100);

		rescaleButton.setSelected(m_indexedTexture.rescaleTexCoord.getValue());

		m_colorMap.predefinedColorMap.setValue(2); // TEMPERATURE
		colormapCombo.setSelectedIndex(2);
		m_colormapMinText.setText("-100");
		m_colormapMaxText.setText("100");
		m_colorMap.min.setValue(-100);
		m_colorMap.max.setValue(100);

		m_colormapType = SoColorMap.ColorMapFormats.LUMINANCE.getValue();
		m_colormapDim = 256;
		m_colormapTypeCombo.setSelectedIndex(0);
		m_colormapDimText.setText(""+m_colormapDim);

		saveDataButton.setSelected(! m_indexedTexture.imageIndex.isNeverWrite());
	}

  @Override
  public void stop()
  {
    m_myViewer.dispose();
  }

	private void generateData() {
	    try {
			int capacity = m_dataDim * m_dataDim;

			float min=0;
			float max=1;
			if (m_dataType == 0) {
			    m_data = ByteBuffer.allocateDirect(capacity);
			    m_data.order(ByteOrder.nativeOrder());
			    for (int i = 0; i < capacity; i++) {
			    	m_data.put( (byte)(i % 200 - 100) );
			    }
				m_indexedTexture.imageIndex.setValue(new SbVec2s(m_dataDim, m_dataDim), DataTypes.SIGNED_BYTE, m_data);
				min = -100;
				max = 100;
			} else if (m_dataType == 1) {
			    m_data = ByteBuffer.allocateDirect(capacity*2);
			    m_data.order(ByteOrder.nativeOrder());
			    ShortBuffer sdata = m_data.asShortBuffer();
			    for (int i = 0; i < capacity; i++) {
			    	sdata.put( (short)(i % 200 - 100) );
			    }
				m_indexedTexture.imageIndex.setValue(new SbVec2s(m_dataDim, m_dataDim), DataTypes.SIGNED_SHORT, m_data);
				min = -100;
				max = 100;
			} else if (m_dataType == 2) {
			    m_data = ByteBuffer.allocateDirect(capacity*4);
			    m_data.order(ByteOrder.nativeOrder());
			    IntBuffer idata = m_data.asIntBuffer();
			    for (int i = 0; i < capacity; i++) {
			    	idata.put( i % 200 - 100 );
			    }
				m_indexedTexture.imageIndex.setValue(new SbVec2s(m_dataDim, m_dataDim), DataTypes.SIGNED_INT32, m_data);
				min = -100;
				max = 100;
			} else if (m_dataType == 3) {
			    m_data = ByteBuffer.allocateDirect(capacity*4);
			    m_data.order(ByteOrder.nativeOrder());
			    FloatBuffer fdata = m_data.asFloatBuffer();
			    for (int i = 0; i < capacity; i++) {
			    	fdata.put( ((float)i) / capacity / 2 + 0.25f );
			    }
				m_indexedTexture.imageIndex.setValue(new SbVec2s(m_dataDim, m_dataDim), DataTypes.FLOAT, m_data);
				min = 0.25f;
				max = 0.75f;
			} else System.err.println("Unexpected data type");
			m_dataMinText.setText(""+min);
			m_dataMaxText.setText(""+max);
			m_dataRangeLabel.setText(min + " ; " + max);
		}
	    catch (OutOfMemoryError e) {
			System.err.println("Not enough memory");
		}
	}

	private void generateColormap() {
    	float[] values = null;
	    if (m_colormapType == SoColorMap.ColorMapFormats.LUMINANCE.getValue() || m_colormapType == SoColorMap.ColorMapFormats.ALPHA.getValue()) {
	    	values = new float[m_colormapDim];
	    } else if (m_colormapType == SoColorMap.ColorMapFormats.LUMINANCE_ALPHA.getValue()) {
	    	values = new float[2*m_colormapDim];
	    } else if (m_colormapType == SoColorMap.ColorMapFormats.RGB.getValue()) {
	    	values = new float[3*m_colormapDim];
	    } else if (m_colormapType == SoColorMap.ColorMapFormats.RGB_ALPHA.getValue()) {
	    	values = new float[4*m_colormapDim];
	    } else System.err.println("Unexpected colormap type");

    	for (int i = 0; i < values.length; i++)
    		values[i] = ((float)i) / values.length;
    	m_colorMap.colorMap.setNum(values.length);
    	m_colorMap.colorMap.setValues(0, values);
	}

}
