package medical.analysis.medicalRuler;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.errors.SoError;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoAnnotation;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.NavigationMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Ruler;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.details.SoOrthoSliceDetail;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  public static final String EXAMPLE_NAME = "Medical Ruler";
  public static final String ANNOTATION = "/medical/data/dicomSample/CVH001.dcm";
  public static final String SCENE_GRAPH = "/medical/analysis/medicalRuler/simpleVolumeRender.iv";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private static String _sceneGraphFile;
  private static String _annotation;
  private static Ruler _ruler;
  private static TextBox _infoText;
  private IRenderAreaExaminer _renderArea;
  private SoVolumeData _volumeData;
  private static boolean _mousePressed;
  private static boolean _firstClick;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    // Load example resources
    try
    {
      _sceneGraphFile = (new File(Main.class.getResource(SCENE_GRAPH).toURI())).toString();
      _annotation = (new File(Main.class.getResource(ANNOTATION).toURI())).toString();
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    _renderArea.getRootSceneGraph().setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);
    _renderArea.getRootSceneGraph().setNavigationMode(NavigationMode.PLANE);

    _renderArea.setSceneGraph(buildSceneGraph());
    _renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    _renderArea.getRootSceneGraph().setInteractionMode(SceneExaminer.InteractionMode.SELECTION);

    SoCamera camera = _renderArea.getRootSceneGraph().getCameraInteractor().getCamera();
    MedicalHelper.orientView(MedicalHelper.Axis.AXIAL, camera, _volumeData);

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  /**
   * MouseMove event handler
   */
  private class MyMouseMoveCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      if ( _mousePressed )
      {
        // Retrieve picked point.
        SoPickedPoint p = node.getPickedPoint();

        if ( p != null )
        {
          // Get detail to pass to the ruler
          SoOrthoSliceDetail detail = (SoOrthoSliceDetail) p.getDetail();

          // Let ruler manage the mouse.
          _ruler.manageMouseMove(_firstClick, detail);
          _firstClick = false;

          // Display position and value.
          _infoText.setLine("Position IJK : " + String.valueOf(detail.getValueDataPos().getX()) + " "
              + String.valueOf(detail.getValueDataPos().getY()) + " " + String.valueOf(detail.getValueDataPos().getZ()),
              0);
          _infoText.setLine("Position XYZ: " + String.valueOf(detail.getValueObjectPos().getX()) + " "
              + String.valueOf(detail.getValueObjectPos().getY()) + " "
              + String.valueOf(detail.getValueObjectPos().getZ()), 1);
          _infoText.setLine("Voxel value  : " + String.valueOf(detail.getValue()), 2);

        }
        node.setHandled();
      }
    }
  }

  /**
   * MouseMove event handler.<br>
   * This could be handled with locate2Event.wasButton1Pressed but this function
   * always returns false: #58130
   */
  private class MyMousePressCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoMouseButtonEvent mouseEvent = (SoMouseButtonEvent) node.getEvent();

      if ( SoMouseButtonEvent.isButtonPressEvent(mouseEvent, mouseEvent.getButton()) )
      {
        _mousePressed = true;
        _firstClick = true;
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(mouseEvent, mouseEvent.getButton()) )
        _mousePressed = false;
      node.setHandled();
    }
  }

  /**
   * Build the scene graph to be displayed.
   */
  private SoSeparator buildSceneGraph()
  {
    SoSeparator root = new SoSeparator();

    // Add node to handle events
    SoEventCallback eventCB = new SoEventCallback();
    eventCB.addEventCallback(SoMouseButtonEvent.class, new MyMousePressCB());
    eventCB.addEventCallback(SoLocation2Event.class, new MyMouseMoveCB());
    root.addChild(eventCB);

    // Load rendering scene graph from file
    SoSeparator scene = MedicalHelper.readFile(_sceneGraphFile);
    if ( scene == null )
      return root;

    root.addChild(scene);

    // Query some nodes in the rendering scene graph
    _volumeData = MedicalHelper.find(scene, SoVolumeData.class);
    if ( _volumeData == null )
      SoError.post("SoVolumeData node not found");

    // Compute scale factor based on volume extent.
    float globalFactor = 1.0f;
    SbVec3f xyz = _volumeData.extent.getValue().getSize();
    globalFactor = (xyz.getX() + xyz.getY() + xyz.getZ()) / 3 / 100;

    // Add the interactive ruler
    SoAnnotation rulerSep = new SoAnnotation();
    root.addChild(rulerSep);
    _ruler = new Ruler();
    _ruler.globalFactor.setValue(globalFactor);
    _ruler.getFont().name.setValue("Arial : Bold");
    rulerSep.addChild(_ruler);

    // Add some typical image viewer annotation
    SoOrthoSlice orthoSlice = MedicalHelper.find(scene, SoOrthoSlice.class);
    if ( orthoSlice == null )
      SoError.post("SoOrthoSlice node not found");
    SoCamera camera = _renderArea.getRootSceneGraph().getCameraInteractor().getCamera();
    root.addChild(MedicalHelper.buildSliceAnnotation(camera, orthoSlice, _annotation));

    // Add Open Inventor logo at the left-bottom corner
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    root.addChild(logoBackground);

    // Status display
    _infoText = new TextBox();
    _infoText.fontName.setValue("Arial");
    _infoText.fontSize.setValue(18);
    _infoText.position.setValue(-0.99f, -0.85f, 0); // Normalized device coords
                                                    // -1..1
    _infoText.alignmentH.setValue(TextBox.AlignmentH.LEFT);
    _infoText.alignmentV.setValue(TextBox.AlignmentV.BOTTOM);
    _infoText.addLine("Click and drag on the image to measure.");
    root.addChild(_infoText);

    return root;
  }
}
