package medical.rendering.visualization.medicalOrthoSliceBorder;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.sensors.SoNodeSensor;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Gnomon;
import com.openinventor.medical.nodes.OrthoSliceBorder;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.draggers.SoOrthoSliceDragger;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeShader;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  public static final String EXAMPLE_NAME = "Medical OrthoSlice Border";
  public static final String DATA = "/medical/data/dicomSample/listOfDicomFiles512.dcm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private static String _dataFile;
  private IRenderAreaExaminer _renderArea;
  private SoVolumeData _volData;
  private SbColor _prevBorderColor = new SbColor(1, 1, 1);

  // Variables for updating slice status box
  private TextBox _sliceTextBox; // TextBox is ambiguous with
                                 // System.Windows.Forms
  private OrthoSliceBorder _aSlice;
  private OrthoSliceBorder _cSlice;
  private OrthoSliceBorder _sSlice;
  private SoNodeSensor _aSliceSensor;
  private SoNodeSensor _cSliceSensor;
  private SoNodeSensor _sSliceSensor;

  private static SbColor ActiveBorderColor = new SbColor(0.84f, 0.43f, 0.02f);

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    // Load example resources
    try
    {
      _dataFile = (new File(Main.class.getResource(DATA).toURI())).toString();
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    _renderArea.getRootSceneGraph().setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);

    _renderArea.setSceneGraph(buildSceneGraph());
    _renderArea.getRootSceneGraph().setInteractionMode(SceneExaminer.InteractionMode.SELECTION);

    SoCamera camera = _renderArea.getRootSceneGraph().getCameraInteractor().getCamera();
    camera.position.setValue(-322.032f, -407.812f, 210.785f);
    camera.orientation.setValue(new SbVec3f(0.828883f, -0.303221f, -0.470117f), 1.27283f);
    _renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  /**
   * Create scene graph
   */
  private SoSeparator buildSceneGraph()
  {
    // Create the scene root
    SoSeparator root = new SoSeparator();

    // Volume visualization
    root.addChild(buildVolumeSceneGraph());

    // Show boundaries of volume (must call _after_ creating the volume data
    // node)
    root.addChild(MedicalHelper.createBoundingBox(_volData.extent.getValue(), new SbColor(0.8f, 0.8f, 0.8f)));

    // Add Open Inventor logo at the left-bottom corner
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    root.addChild(logoBackground);

    // Medical Gnomon
    root.addChild(new Gnomon());

    // Status info
    _sliceTextBox = new TextBox();
    _sliceTextBox.position.setValue(-0.98f, 0.98f, 0);
    _sliceTextBox.fontName.setValue("Arial : Bold");
    _sliceTextBox.fontSize.setValue(16);
    _sliceTextBox.addLine("Click and drag to reposition slices");
    _sliceTextBox.addLine("Axial: ");
    _sliceTextBox.addLine("Coronal: ");
    _sliceTextBox.addLine("Sagittal: ");
    root.addChild(_sliceTextBox);

    // Setup to update the status info (detect change to any of the slices).
    _aSliceSensor = new SoNodeSensor();
    _cSliceSensor = new SoNodeSensor();
    _sSliceSensor = new SoNodeSensor();
    _aSliceSensor.setTask(new SliceSensorCB());
    _cSliceSensor.setTask(new SliceSensorCB());
    _sSliceSensor.setTask(new SliceSensorCB());
    _aSliceSensor.attach(_aSlice);
    _cSliceSensor.attach(_cSlice);
    _sSliceSensor.attach(_sSlice);
    updateSliceTextBox();

    return root;
  }

  /**
   * Build VolumeViz part of scene graph
   */
  private SoSeparator buildVolumeSceneGraph()
  {
    SoSeparator volSep = new SoSeparator();

    // Load volume (and make standard adjustments for DICOM data).
    SoVolumeData volData = new SoVolumeData();
    volData.fileName.setValue(_dataFile);
    MedicalHelper.dicomAdjustVolume(volData, true);
    volSep.addChild(volData);
    _volData = volData;

    // Set range of data values to visualize.
    SoDataRange volRange = new SoDataRange();
    MedicalHelper.dicomAdjustDataRange(volRange, volData);
    volSep.addChild(volRange);

    // Standard gray scale transferfunction
    SoTransferFunction volTF = new SoTransferFunction();
    volTF.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);
    volSep.addChild(volTF);

    // Display slices at full intensity
    SoMaterial volMat = new SoMaterial();
    volMat.diffuseColor.setValue(1, 1, 1);
    volSep.addChild(volMat);

    // Remove tile boundary artifacts while moving.
    SoVolumeShader volShader = new SoVolumeShader();
    volShader.interpolateOnMove.setValue(true);
    volSep.addChild(volShader);

    // Create Sagittal slice and dragger (X axis)
    OrthoSliceBorder sagittalPlaneBorder = new OrthoSliceBorder();
    {
      sagittalPlaneBorder.borderColor.setValue(0.37f, 0.37f, 1); // Blue
                                                                 // luminance
                                                                 // 55%;
      sagittalPlaneBorder.setName("sagittalPlane");
      volSep.addChild(sagittalPlaneBorder);

      SoOrthoSlice sagittalPlane = new SoOrthoSlice();
      sagittalPlane.interpolation.setValue(SoOrthoSlice.Interpolations.MULTISAMPLE_12);
      sagittalPlane.axis.setValue(SoOrthoSlice.AxisType.X);
      sagittalPlane.sliceNumber.setValue(volData.data.getSize().getX() / 2);
      volSep.addChild(sagittalPlane);

      sagittalPlaneBorder.axis.connectFrom(sagittalPlane.axis);
      sagittalPlaneBorder.sliceNumber.connectFrom(sagittalPlane.sliceNumber);

      // Attach a dragger to this slice.
      SoOrthoSliceDragger sliceDragger = new SoOrthoSliceDragger();
      SoPath slicePath = new SoPath(sagittalPlane);
      sliceDragger.orthoSlicePath.setValue(slicePath);
      sliceDragger.volumeDimension.setValue(volData.data.getSize());
      sliceDragger.volumeExtent.setValue(volData.extent.getValue());
      // Make the border color change while dragging is active
      sliceDragger.addStartCallback(new DraggerStartCB(sagittalPlaneBorder), null);
      sliceDragger.addFinishCallback(new DraggerFinishCB(sagittalPlaneBorder), null);
      volSep.addChild(sliceDragger);

      _sSlice = sagittalPlaneBorder;
    }
    // Create Coronal slice and dragger (Y axis)
    OrthoSliceBorder coronalPlaneBorder = new OrthoSliceBorder();
    {
      coronalPlaneBorder.borderColor.setValue(0, 0.7f, 0); // Green luminance
                                                           // 55%
      coronalPlaneBorder.setName("coronalPlane");
      volSep.addChild(coronalPlaneBorder);

      SoOrthoSlice coronalPlane = new SoOrthoSlice();
      coronalPlane.axis.setValue(SoOrthoSlice.AxisType.Y);
      coronalPlane.interpolation.setValue(SoOrthoSlice.Interpolations.MULTISAMPLE_12);
      coronalPlane.sliceNumber.setValue(volData.data.getSize().getY() / 2);
      volSep.addChild(coronalPlane);

      coronalPlaneBorder.axis.connectFrom(coronalPlane.axis);
      coronalPlaneBorder.sliceNumber.connectFrom(coronalPlane.sliceNumber);

      // Attach a dragger to this slice.
      SoOrthoSliceDragger sliceDragger = new SoOrthoSliceDragger();
      SoPath slicePath = new SoPath(coronalPlane);
      sliceDragger.orthoSlicePath.setValue(slicePath);
      sliceDragger.volumeDimension.setValue(volData.data.getSize());
      sliceDragger.volumeExtent.setValue(volData.extent.getValue());
      // Make the border color change while dragging is active
      sliceDragger.addStartCallback(new DraggerStartCB(coronalPlaneBorder), null);
      sliceDragger.addFinishCallback(new DraggerFinishCB(coronalPlaneBorder), null);
      volSep.addChild(sliceDragger);

      _cSlice = coronalPlaneBorder;
    }
    // Create Axial slice and dragger (Z axis)
    OrthoSliceBorder axialPlaneBorder = new OrthoSliceBorder();
    {
      axialPlaneBorder.borderColor.setValue(1, 0.11f, 0.11f); // Red luminance
                                                              // 55%
      axialPlaneBorder.setName("axialPlane");
      volSep.addChild(axialPlaneBorder);

      SoOrthoSlice axialPlane = new SoOrthoSlice();
      axialPlane.axis.setValue(SoOrthoSlice.AxisType.Z);
      axialPlane.interpolation.setValue(SoOrthoSlice.Interpolations.MULTISAMPLE_12);
      axialPlane.sliceNumber.setValue(volData.data.getSize().getZ() / 2);
      volSep.addChild(axialPlane);

      axialPlaneBorder.axis.connectFrom(axialPlane.axis);
      axialPlaneBorder.sliceNumber.connectFrom(axialPlane.sliceNumber);

      // Attach a dragger to this slice.
      SoOrthoSliceDragger sliceDragger = new SoOrthoSliceDragger();
      SoPath slicePath = new SoPath(axialPlane);
      sliceDragger.orthoSlicePath.setValue(slicePath);
      sliceDragger.volumeDimension.setValue(volData.data.getSize());
      sliceDragger.volumeExtent.setValue(volData.extent.getValue());
      // Make the border color change while dragging is active
      sliceDragger.addStartCallback(new DraggerStartCB(axialPlaneBorder), null);
      sliceDragger.addFinishCallback(new DraggerFinishCB(axialPlaneBorder), null);
      volSep.addChild(sliceDragger);

      _aSlice = axialPlaneBorder;
    }
    return volSep;
  }

  /**
   * Handle start dragging event.
   */
  class DraggerStartCB extends SoDraggerCB
  {
    OrthoSliceBorder planeBorder;

    public DraggerStartCB(OrthoSliceBorder border)
    {
      planeBorder = border;
    }

    @Override
    public void invoke(SoDragger _dragger)
    {
      _prevBorderColor = planeBorder.borderColor.getValue();
      planeBorder.borderColor.setValue(ActiveBorderColor);
    }
  }

  /**
   * Handle finish dragging event.
   *
   */
  class DraggerFinishCB extends SoDraggerCB
  {
    OrthoSliceBorder planeBorder;

    public DraggerFinishCB(OrthoSliceBorder border)
    {
      planeBorder = border;
    }

    @Override
    public void invoke(SoDragger _dragger)
    {
      planeBorder.borderColor.setValue(_prevBorderColor);
    }
  }

  /**
   * Automatically called whenever one of the slice nodes is modified.
   *
   * @param sensor
   */
  class SliceSensorCB implements Runnable
  {
    @Override
    public void run()
    {
      updateSliceTextBox();
    }
  }

  /**
   * Update display of current / maximum slice number for each axis.
   *
   * Note that line 0 of the text box is a label, so the actual line we update
   * for each axis is the axis enum (0, 1 or 2) plus 1!
   */
  private void updateSliceTextBox()
  {
    {
      int axis = _aSlice.axis.getValue();
      int sliceNum = _aSlice.sliceNumber.getValue();
      String str = String.format("Axial      : " + String.valueOf(sliceNum) + " / "
          + String.valueOf(_volData.data.getSize().getValue()[axis]));
      _sliceTextBox.setLine(str, axis + 1);
    }
    {
      int axis = _cSlice.axis.getValue();
      int sliceNum = _cSlice.sliceNumber.getValue();
      String str = String.format(
          "Coronal: " + String.valueOf(sliceNum) + " / " + String.valueOf(_volData.data.getSize().getValue()[axis]));
      _sliceTextBox.setLine(str, axis + 1);
    }
    {
      int axis = _sSlice.axis.getValue();
      int sliceNum = _sSlice.sliceNumber.getValue();
      String str = String.format(
          "Sagittal : " + String.valueOf(sliceNum) + " / " + String.valueOf(_volData.data.getSize().getValue()[axis]));
      _sliceTextBox.setLine(str, axis + 1);
    }
  }

}
