package medical.rendering.visualization.medicalmip;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.Point;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JFrame;
import javax.swing.JPanel;

import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3d;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.manips.SoTransformerManip;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoNodeSensor;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.volumeviz.nodes.SoVolumeClippingGroup;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  private static final String EXAMPLE_NAME = "Medical MIP";
  private static final String VIEWER_2D = "Viewer 2D";
  private static final String VIEWER_3D = "Viewer 3D";
  private static final String DATA = "/medical/data/files/3DHEAD.ldm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private static boolean m_isStandAlone = false;

  private int m_renderMode;
  private SoVolumeRender m_volumeRender;
  private SoTransferFunction m_transfertFunction;
  private SoTransformerManip m_manip;

  private List<IRenderAreaInteractive> m_viewerList;
  private List<JFrame> m_frameList;

  public static void printHelp()
  {
    System.out.println("Launching example: " + EXAMPLE_NAME);
    System.out.println("LeftClick a face of the bounding box to drag along two axis");
    System.out.println("LeftClick a 'white cube' to scale the dragger along all axis");
    System.out.println("LeftClick a 'green ball' to rotate the dragger around one axis");
    System.out.println("Shift+LeftClick a 'white cube' to scale the dragger along one axis");
    System.out.println("Shift+LeftClick a 'green ball' to rotate the dragger around all axis");
    System.out.println("Press R on the 3D viewer to reset the dragger fields");
  }

  public static void main(String[] args)
  {
    m_isStandAlone = true;

    Main example = new Main();
    example.start();
  }

  /**
   * Keyboard callback.
   */
  private class CustomKeyboardEventCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();
      if ( evt != null )
      {
        if ( SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.PAD_ADD) )
        {
          m_renderMode = (m_renderMode + 1) % 3;
          switch ( m_renderMode )
          {
          case 0 : // Min
            m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.MIN_INTENSITY_PROJECTION);
            m_transfertFunction.maxValue.setValue(93);
            break;
          case 1 : // Max
            m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.MAX_INTENSITY_PROJECTION);
            m_transfertFunction.maxValue.setValue(255);
            break;
          case 2 : // Average
            m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.AVERAGE_INTENSITY_PROJECTION);
            m_transfertFunction.maxValue.setValue(155);
            break;
          }
        }
        else if ( SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.R) )
        {
          // Reset manip fields
          m_manip.translation.setValue(0, 0, 0);
          m_manip.rotation.setValue(SbRotation.identity());
          m_manip.scaleFactor.setValue(1, 1, 1);
        }
      }
    }
  }

  /**
   * Watch the node manip, and update the camera of the 2D viewer accordingly
   */
  private class ManipWatcher implements Runnable
  {
    @Override
    public void run()
    {
      SoOrthographicCamera camera = (SoOrthographicCamera) (SoNode.getByName("viewer2D_camera"));
      camera.position.setValue(m_manip.translation.getValue());
      camera.orientation.setValue(m_manip.rotation.getValue());
    }
  }

  @Override
  public void start()
  {
    printHelp();

    // Load example resources
    File dataFile;
    try
    {
      dataFile = new File(Main.class.getResource(DATA).toURI());
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    // Launch examples
    m_viewerList = new ArrayList<IRenderAreaInteractive>();
    m_frameList = new ArrayList<JFrame>();

    String dataPath = dataFile.toString();

    //////////////////////////////////////////////////////
    // 3D Viewer
    //////////////////////////////////////////////////////
    final IRenderAreaInteractive viewer3D = ViewerComponentsFactory.createRenderAreaInteractive();
    viewer3D.setCameraType(CameraMode.ORTHOGRAPHIC);

    SoSeparator viewer3DRoot = new SoSeparator();
    viewer3DRoot.setName(VIEWER_3D);

    // Define keyboard callback on the 3D viewer
    SoEventCallback eventCB = new SoEventCallback();
    eventCB.addEventCallback(SoKeyboardEvent.class, new CustomKeyboardEventCB());
    viewer3DRoot.addChild(eventCB);

    // Load background logo node
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    viewer3DRoot.addChild(logoBackground);

    // SoTransformerManip allows user to drag/scale/rotate clipping box
    SoTransformSeparator protectManip = new SoTransformSeparator();
    m_manip = new SoTransformerManip();
    protectManip.addChild(m_manip);
    viewer3DRoot.addChild(protectManip);

    // Create the volume clipping group
    SoVolumeClippingGroup clippingGroup = new SoVolumeClippingGroup();
    viewer3DRoot.addChild(clippingGroup);

    // SoTransformSeparator prevents local transform from "leaking" out of
    // clipping group
    SoTransformSeparator localProtection = new SoTransformSeparator();
    clippingGroup.addChild(localProtection);

    // This transform controls the position/size/rotation of the clipping
    // geometry. Its fields are connected from the manip's fields so it
    // automatically tracks the changes that the user makes to the manip.
    SoTransform localTransform = new SoTransform();
    localTransform.setName("Local_Transfo");
    localTransform.translation.connectFrom(m_manip.translation);
    localTransform.rotation.connectFrom(m_manip.rotation);
    localTransform.scaleFactor.connectFrom(m_manip.scaleFactor);
    localTransform.scaleOrientation.connectFrom(m_manip.scaleOrientation);
    localTransform.center.connectFrom(m_manip.center);
    localProtection.addChild(localTransform);

    // Add the actual clipping geometry
    localProtection.addChild(new SoCube());

    // Create and place a red arrow in front of the cube
    SoSeparator arrowSep = new SoSeparator();
    SoTransform arrowTransform = new SoTransform();
    arrowTransform.translation.connectFrom(m_manip.translation);
    arrowTransform.rotation.connectFrom(m_manip.rotation);
    arrowTransform.scaleFactor.connectFrom(m_manip.scaleFactor);
    arrowTransform.scaleOrientation.connectFrom(m_manip.scaleOrientation);
    arrowTransform.center.connectFrom(m_manip.center);
    arrowSep.addChild(arrowTransform);
    SoRotationXYZ arrowRot = new SoRotationXYZ();
    arrowRot.axis.setValue(0);
    arrowRot.angle.setValue(-1.5707f);
    arrowSep.addChild(arrowRot);
    SoTranslation arrowTrans = new SoTranslation();
    arrowTrans.translation.setValue(0, -3, 0);
    arrowSep.addChild(arrowTrans);
    SoMaterial arrowMat = new SoMaterial();
    arrowMat.diffuseColor.setValue(1, 0, 0);
    arrowSep.addChild(arrowMat);
    SoCylinder cylinderArrow = new SoCylinder();
    cylinderArrow.radius.setValue(0.01f);
    arrowSep.addChild(cylinderArrow);
    SoTranslation coneArrowTrans = new SoTranslation();
    coneArrowTrans.translation.setValue(0, 1, 0);
    arrowSep.addChild(coneArrowTrans);
    SoCone coneArrow = new SoCone();
    coneArrow.bottomRadius.setValue(0.05f);
    coneArrow.height.setValue(0.2f);
    arrowSep.addChild(coneArrow);
    viewer3DRoot.addChild(arrowSep);

    // Sub-graph containing the data
    SoSeparator dataSep = new SoSeparator();
    viewer3DRoot.addChild(dataSep);

    // Flip the data (because by default it is upside down)
    SoRotationXYZ dataRot = new SoRotationXYZ();
    dataRot.axis.setValue(0);
    dataRot.angle.setValue((float) Math.PI);
    dataSep.addChild(dataRot);

    // Node holding the volume data
    SoVolumeData volumeData = new SoVolumeData();
    volumeData.fileName.setValue(dataPath);
    dataSep.addChild(volumeData);

    // Describes the association between data data values and colors
    m_transfertFunction = new SoTransferFunction();
    m_transfertFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);
    m_transfertFunction.maxValue.setValue(155);
    dataSep.addChild(m_transfertFunction);

    // Node in charge of drawing the volume
    m_volumeRender = new SoVolumeRender();
    m_volumeRender.renderMode.setValue(SoVolumeRender.RenderModes.AVERAGE_INTENSITY_PROJECTION);
    dataSep.addChild(m_volumeRender);
    m_renderMode = 2;

    viewer3D.setSceneGraph(viewer3DRoot);

    // Adjust camera pose to have a nicer view of the scene
    viewer3D.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    SoOrthographicCamera viewer3DCamera = (SoOrthographicCamera) viewer3D.getRootSceneGraph().getCamera();
    viewer3DCamera.nearDistance.setValue(0);
    viewer3DCamera.farDistance.setValue(1000);
    viewer3DCamera.position.setValue(new SbVec3d(0, 0, 4.5));

    //////////////////////////////////////////////////////
    // 2D Viewer
    //////////////////////////////////////////////////////
    final IRenderAreaInteractive viewer2D = ViewerComponentsFactory.createRenderAreaInteractive();
    viewer2D.setCameraType(CameraMode.ORTHOGRAPHIC);

    SoSeparator viewer2DRoot = new SoSeparator();
    viewer2DRoot.setName(VIEWER_2D);

    // Set up camera
    SoOrthographicCamera viewer2DCamera = (SoOrthographicCamera) viewer2D.getRootSceneGraph().getCamera();
    viewer2DCamera.setName("viewer2D_camera");
    viewer2DCamera.nearDistance.setValue(-50.0f);
    viewer2DCamera.farDistance.setValue(50.0f);

    viewer2DRoot.addChild(logoBackground);
    viewer2DRoot.addChild(clippingGroup);
    viewer2DRoot.addChild(dataSep);

    // Watch the node manip
    SoNodeSensor rotationSensor = new SoNodeSensor();
    rotationSensor.attach(m_manip);
    rotationSensor.setTask(new ManipWatcher());
    rotationSensor.setPriority(0);

    viewer2D.setSceneGraph(viewer2DRoot);

    //////////////////////////////////////////////////////
    // Launch examples
    //////////////////////////////////////////////////////
    displayViewer(viewer3D, VIEWER_3D, new Point(0, 0));
    displayViewer(viewer2D, VIEWER_2D, new Point(MedicalHelper.WINDOW_WIDTH, 0));
  }

  private void displayViewer(final IRenderAreaInteractive renderArea, String viewerName, Point location)
  {
    final JFrame frame = new JFrame(EXAMPLE_NAME + " - " + viewerName);
    frame.setSize(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT);
    frame.setLocation(location);

    final JPanel panel = new JPanel();
    panel.setLayout(new BorderLayout());
    panel.add(renderArea.getComponent());

    frame.add(panel);
    frame.addWindowListener(new WindowAdapter()
    {
      @Override
      public void windowClosing(WindowEvent e)
      {
        renderArea.dispose();
        frame.dispose();
        if ( m_isStandAlone )
          System.exit(0);
      }
    });
    m_viewerList.add(renderArea);
    m_frameList.add(frame);

    EventQueue.invokeLater(new Runnable()
    {
      @Override
      public void run()
      {
        try
        {
          frame.setVisible(true);
        }
        catch (Exception e)
        {
          LOGGER.log(Level.SEVERE, "Failed to launch viewer", e);
        }
      }
    });
  }

  @Override
  public void stop()
  {
    for ( IRenderAreaInteractive renderArea : m_viewerList )
      renderArea.dispose();
    for ( JFrame jFrame : m_frameList )
      jFrame.dispose();
  }
}
