package medical.rendering.visualization.medicalmultiview;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.awt.Point;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JFrame;
import javax.swing.JPanel;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbPlane;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.draggers.SoJackDragger;
import com.openinventor.inventor.events.SoEvent;
import com.openinventor.inventor.events.SoKeyboardEvent;
import com.openinventor.inventor.events.SoMouseWheelEvent;
import com.openinventor.inventor.gui.view.PoView;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.volumeviz.nodes.SoObliqueSlice;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Very useful demonstration that explains how to share a same dataset between
 * three different viewers. In this case, a 3D and a 2D viewers.
 * <p>
 * In this viewer you obtain a rendering of a Dicom stack using the RayCasting
 * technology of VolumeViz. You can manipulate this 3D volume by right clicking
 * the left button of the mouse and by moving the mouse. We load a stack of
 * Dicom or an LDA ( Open Inventor file format that give the capabilities to
 * manipulate out of core data set ) file. With LDA file, even if the size of
 * the file is bigger than the memory size of your computer, the dataset is
 * still immediately loaded.
 * <p>
 * In selection mode ( ESC in the 3D viewer ), it is also possible to change the
 * min and max values of the SoTransferFunction. Still in selection mode (cursor
 * == arrow), you can select the dragger to manipulate the connected
 * SoObliqueSlice + SoClipPlane. The rendering of this manipulation is done in
 * the 2D viewer (bottom left).
 * <p>
 * In this Render Area (top right), we render nine SoOrthoSlice (from
 * SoVolumeData). If you select the DOWN arrow of the keyboard, you decrease by
 * five the index of each orthoSlice and if the UP arrow is selected, you
 * increase by five the index of each SoOrthoSlice.
 * <p>
 * <b>Limitation</b>: This demo is not available on 32 bit architectures as it
 * requires more than 2 GB of memory to be run.
 *
 * @see SoVolumeData
 * @see SoTransferFunction
 * @see SoDataRange
 * @see SoJackDragger
 */
public class Main extends Example
{
  private static final String VIEWER_ORTHO = "Viewer Ortho";
  private static final String VIEWER_OBLIQUE = "Viewer Oblique";
  private static final String VIEWER_3D = "Viewer 3D";
  //public static final String DATA = "/medical/data/files/Thorax.ldm";
  public static final String DATA = "/medical/data/dicomSample/listOfDicomFiles512.dcm";
  public static final String COLORMAP = "/medical/data/resources/volrenGlow.am";
  public static final String EXAMPLE_NAME = "Medical Multiple Viewers";
  public static final String EXAMPLE_DESCRITPION = "This example shows how to create multiple viewers for medical data";
  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private static boolean m_isStandAlone = false;

  private SoObliqueSlice m_obliqueSlice = new SoObliqueSlice();
  private float[] m_volumeBounds;
  private OrthoSliceArray m_orthoSliceArray;

  private IRenderAreaInteractive[] m_viewerArray;
  private List<JFrame> m_frameList;

  public static void printHelp()
  {
    System.out.println("Launching example: " + EXAMPLE_NAME);
    System.out.println(EXAMPLE_DESCRITPION);
    System.out.println("Press Esc in the Viewer 3D to switch to PICKING mode in order to move the dragger");
    System.out.println("Press Esc in the Viewer 3D to switch back to VIEWING mode");
    System.out.println("Press up/down arrows or scroll up/down in the Ortho Viewer to modify the sections");
  }

  public static void main(String[] args)
  {
    m_isStandAlone = true;

    Main example = new Main();
    example.start();
  }

  /**
   * Definition of viewers defined in this example
   */
  public enum Viewer
  {
    VOLUME, OBLIQUE, ORTHO
  }

  /**
   * Structure used to store and ortho slices array
   */
  public class OrthoSliceArray
  {
    public SoOrthoSlice[][] orthoSlicesInZ;
    public PoView[][] poViews;

    public OrthoSliceArray(int rows, int cols)
    {
      orthoSlicesInZ = new SoOrthoSlice[rows][cols];
      poViews = new PoView[rows][cols];
    }
  }

  /**
   * Dragger callback. It is invoked when the dragger in the Viewer 3D is moved.
   * The section shown on the Viewer Oblique is updated accordingly.
   */
  class CustomSoDraggerCB extends SoDraggerCB
  {
    @Override
    public void invoke(SoDragger dragger)
    {
      SoJackDragger jackDragger = (SoJackDragger) dragger;
      SbVec3f draggerPos = jackDragger.translation.getValue();

      // rotate the plane's normal by the dragger rotation
      SbRotation rotation = jackDragger.rotation.getValue();
      SbVec3f planeNormal = rotation.multVec(new SbVec3f(0, 1, 0));

      // translate cross section and cross contour
      m_obliqueSlice.plane.setValue(new SbPlane(planeNormal, draggerPos));

      // Do a viewAll of the volume slide
      SbBox3f bbox = new SbBox3f(m_volumeBounds[0], m_volumeBounds[1], jackDragger.translation.getValue().getValueAt(2),
          m_volumeBounds[3], m_volumeBounds[4], jackDragger.translation.getValue().getValueAt(2));
      SoCamera camera = m_viewerArray[Viewer.OBLIQUE.ordinal()].getRootSceneGraph().getCamera();
      camera.viewAll(bbox, new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    }
  }

  /**
   * Keyboard callback. It is invoked when the "Viewer Ortho" is on focus and
   * when the keyboard is touched.
   */
  private class CustomKeyboardEventCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();
      if ( evt != null )
      {
        if ( SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.UP_ARROW) )
          scrollOrthoSliceArray(49);
        if ( SoKeyboardEvent.isKeyPressEvent(evt, SoKeyboardEvent.Keys.DOWN_ARROW) )
          scrollOrthoSliceArray(-49);
      }
    }
  }

  /**
   * Mouse wheel callback. It is invoked when the "Viewer Ortho" is on focus and
   * when the keyboard is touched.
   */
  private class CustomMouseWheelEventCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoEvent evt = node.getEvent();
      if ( evt != null )
      {
        SoMouseWheelEvent mouseWheelEvent = (SoMouseWheelEvent) evt;
        scrollOrthoSliceArray((mouseWheelEvent.getDelta() > 0 ? 1 : -1));
      }
    }
  }

  /**
   * Modifies the sections displayed on the Viewer Ortho by increment
   *
   * @param increment
   */
  private void scrollOrthoSliceArray(int increment)
  {
    for ( int i = 0; i < m_orthoSliceArray.orthoSlicesInZ.length; i++ )
    {
      for ( int j = 0; j < m_orthoSliceArray.orthoSlicesInZ[i].length; j++ )
      {
        int currentSliceNumber = m_orthoSliceArray.orthoSlicesInZ[i][j].sliceNumber.getValue();
        m_orthoSliceArray.orthoSlicesInZ[i][j].sliceNumber.setValue(currentSliceNumber + increment);
        SoOrthographicCamera camera =
            (SoOrthographicCamera) m_orthoSliceArray.poViews[i][j].getPart("cameraKit.camera", true);
        camera.viewAll(m_viewerArray[Viewer.ORTHO.ordinal()].getRootSceneGraph(),
            new SbViewportRegion((short) 1, (short) 1));
      }
    }
  }

  @Override
  public void start()
  {
    printHelp();

    // Load example resources
    File dataFile, colormapFile;
    try
    {
      dataFile = new File(Main.class.getResource(DATA).toURI());
      colormapFile = new File(Main.class.getResource(COLORMAP).toURI());
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    // Launch examples
    m_viewerArray = new IRenderAreaInteractive[3];
    m_frameList = new ArrayList<JFrame>();

    String dataPath = dataFile.toString();
    String colormapPath = colormapFile.toString();

    //////////////////////////////////////////////////////
    // Define common nodes
    //////////////////////////////////////////////////////

    // Load background logo node
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }

    // Decrease the quality while moving to have better interactivity
    SoInteractiveComplexity interactiveComplexity = new SoInteractiveComplexity();
    interactiveComplexity.fieldSettings.set1(0, "SoComplexity value 0.2 0.5");
    interactiveComplexity.fieldSettings.set1(1, "SoVolumeRender interpolation LINEAR CUBIC");
    interactiveComplexity.refinementDelay.setValue(0);

    // Complexity node for the interact node to control
    SoComplexity complexity = new SoComplexity();

    // Node holding the volume data
    SoVolumeData volumeData = new SoVolumeData();
    volumeData.fileName.setValue(dataPath);

    //////////////////////////////////////////////////////
    // VOLUME
    //////////////////////////////////////////////////////
    // RenderAreaExaminer for the volume to be able to move the camera
    final IRenderAreaExaminer volumeViewer = ViewerComponentsFactory.createRenderAreaExaminer();

    SoSeparator volumeRoot = new SoSeparator();
    volumeRoot.setName(VIEWER_3D);

    SbBox3f volumeSize;
    SbVec3f draggerInitialPosition, draggerNormal;
    SoJackDragger dragger;

    {
      // Use a predefined colorMap (associate data values to colors)
      SoTransferFunction transferFunction = new SoTransferFunction();
      transferFunction.loadColormap(colormapPath);

      // Remap data range to full range of data
      SoDataRange dataRange = new SoDataRange();
      dataRange.min.setValue(176);
      dataRange.max.setValue(476);

      // Property node which allows SoVolumeRender to draw High Quality volumes
      SoVolumeRenderingQuality renderingQuality = new SoVolumeRenderingQuality();
      renderingQuality.interpolateOnMove.setValue(true);
      renderingQuality.preIntegrated.setValue(true);
      renderingQuality.ambientOcclusion.setValue(true);
      renderingQuality.deferredLighting.setValue(true);

      // Node in charge of drawing the volume
      SoVolumeRender volumeRender = new SoVolumeRender();
      volumeRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.AUTOMATIC);
      volumeRender.lowResMode.setValue(SoVolumeRender.LowResModes.DECREASE_SCREEN_RESOLUTION);
      volumeRender.lowScreenResolutionScale.setValue(2);
      volumeRender.subdivideTile.setValue(true);
      volumeRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
      volumeRender.opacityThreshold.setValue(0.17f);

      // Clipping Group
      SoGroup clippingGroup = new SoGroup();
      clippingGroup.setName("Clipping_Group");

      volumeSize = volumeData.extent.getValue();
      draggerInitialPosition = volumeSize.getMin().plus(volumeSize.getMax().minus(volumeSize.getMin()).over(2));
      draggerNormal = new SbVec3f(0, 0, 1);
      SbRotation draggerInitialRotation = new SbRotation(new SbVec3f(0.0f, 1.0f, 0.0f), draggerNormal);
      SbVec3f draggerScaleFactor = new SbVec3f(100, 100, 100);

      dragger = new SoJackDragger();
      dragger.setName("Jack_For_Oblique");
      dragger.addMotionCallback(new CustomSoDraggerCB(), null);
      dragger.rotation.setValue(draggerInitialRotation);
      dragger.scaleFactor.setValue(draggerScaleFactor);
      dragger.translation.setValue(draggerInitialPosition);

      // To clip the volume when moving the dragger
      SoClipPlane clipPlane = new SoClipPlane();
      clipPlane.setName("Oblique_ClipPlane");
      clipPlane.on.setValue(true);
      clipPlane.plane.connectFrom(m_obliqueSlice.plane);

      // Define the scene graph
      volumeRoot.addChild(logoBackground);
      volumeRoot.addChild(interactiveComplexity);
      volumeRoot.addChild(complexity);
      volumeRoot.addChild(volumeData);
      volumeRoot.addChild(transferFunction);
      volumeRoot.addChild(dataRange);
      volumeRoot.addChild(renderingQuality);
      clippingGroup.addChild(dragger);
      clippingGroup.addChild(clipPlane);
      volumeRoot.addChild(clippingGroup);
      volumeRoot.addChild(volumeRender);

      volumeViewer.setSceneGraph(volumeRoot);
      volumeViewer.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    }

    //////////////////////////////////////////////////////
    // OBLIQUE
    //////////////////////////////////////////////////////
    final IRenderAreaInteractive obliqueViewer = ViewerComponentsFactory.createRenderAreaInteractive();
    obliqueViewer.setCameraType(CameraMode.ORTHOGRAPHIC);

    SoSeparator obliqueRoot = new SoSeparator();
    obliqueRoot.setName(VIEWER_OBLIQUE);
    {
      // Describes the association between data data values and colors
      SoTransferFunction transfertFunction = new SoTransferFunction();
      transfertFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);

      // remap data range to full range of data
      SoDataRange dataRange = new SoDataRange();
      dataRange.min.setValue(-1024);
      dataRange.max.setValue(1789);

      // Slice of the volume to visualize
      m_obliqueSlice.interpolation.setValue(SoObliqueSlice.Interpolations.TRILINEAR);
      m_obliqueSlice.enableBumpMapping.setValue(true);
      m_obliqueSlice.bumpScale.setValue(10.0f);
      m_obliqueSlice.plane.setValue(new SbPlane(draggerNormal, draggerInitialPosition));

      // Define the scene graph
      obliqueRoot.addChild(logoBackground);
      obliqueRoot.addChild(interactiveComplexity);
      obliqueRoot.addChild(complexity);
      obliqueRoot.addChild(volumeData);
      obliqueRoot.addChild(transfertFunction);
      obliqueRoot.addChild(dataRange);
      obliqueRoot.addChild(m_obliqueSlice);

      obliqueViewer.setSceneGraph(obliqueRoot);

      // Do a viewAll of the volume slide (to see only that on the viewer)
      m_volumeBounds = volumeSize.getBounds();
      SbBox3f bbox = new SbBox3f(m_volumeBounds[0], m_volumeBounds[1], dragger.translation.getValue().getValueAt(2),
          m_volumeBounds[3], m_volumeBounds[4], dragger.translation.getValue().getValueAt(2));
      obliqueViewer.getRootSceneGraph().getCamera().viewAll(bbox,
          new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    }

    //////////////////////////////////////////////////////
    // ORTHO
    //////////////////////////////////////////////////////
    final IRenderAreaInteractive orthoViewer = ViewerComponentsFactory.createRenderAreaInteractive();

    SoSeparator orthoRoot = new SoSeparator();
    orthoRoot.setName(VIEWER_ORTHO);
    {
      // remap data range to full range of data
      SoDataRange dataRange = new SoDataRange();
      dataRange.min.setValue(-1000);
      dataRange.max.setValue(2138);

      // Describes the association between data data values and colors
      SoTransferFunction transfertFunction = new SoTransferFunction();
      transfertFunction.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.INTENSITY);

      SoSeparator orthoSep = new SoSeparator();
      orthoSep.setName("Ortho_Separator");

      // Define the scene graph
      orthoRoot.addChild(logoBackground);
      orthoRoot.addChild(interactiveComplexity);
      orthoRoot.addChild(complexity);
      orthoRoot.addChild(volumeData);
      orthoSep.addChild(transfertFunction);
      orthoSep.addChild(dataRange);
      orthoRoot.addChild(orthoSep);

      // Create multi-view ortho slices
      int k = 0; // first slice
      m_orthoSliceArray = new OrthoSliceArray(7, 7);
      for ( int i = 0; i < m_orthoSliceArray.orthoSlicesInZ.length; ++i )
      {
        for ( int j = 0; j < m_orthoSliceArray.orthoSlicesInZ[i].length; ++j )
        {
          m_orthoSliceArray.poViews[i][j] = new PoView();
          m_orthoSliceArray.poViews[i][j].sensitiveOnEvents(true);
          m_orthoSliceArray.poViews[i][j].viewportOrigin.setValue(0.0f + (i * 0.142f), 0.0f + (j * 0.142f));
          m_orthoSliceArray.poViews[i][j].viewportSize.setValue(.142f, .142f);
          SoOrthographicCamera camera = new SoOrthographicCamera();
          m_orthoSliceArray.poViews[i][j].setPart("cameraKit.camera", camera);
          orthoSep.addChild(m_orthoSliceArray.poViews[i][j]);

          SoSeparator locSep = new SoSeparator();
          m_orthoSliceArray.orthoSlicesInZ[i][j] = new SoOrthoSlice();
          m_orthoSliceArray.orthoSlicesInZ[i][j].sliceNumber.setValue(k++);
          m_orthoSliceArray.orthoSlicesInZ[i][j].axis.setValue(SoOrthoSlice.AxisType.Z);
          locSep.addChild(m_orthoSliceArray.orthoSlicesInZ[i][j]);
          orthoSep.addChild(locSep);

          camera.viewAll(orthoRoot, new SbViewportRegion((short) 1, (short) 1));
        }
      }

      // An event callback node so we can receive key press and wheel events
      SoEventCallback eventCB = new SoEventCallback();
      eventCB.addEventCallback(SoKeyboardEvent.class, new CustomKeyboardEventCB());
      eventCB.addEventCallback(SoMouseWheelEvent.class, new CustomMouseWheelEventCB());
      orthoRoot.addChild(eventCB);

      orthoViewer.setSceneGraph(orthoRoot);
    }

    //////////////////////////////////////////////////////
    // Launch examples
    //////////////////////////////////////////////////////
    m_viewerArray[Viewer.VOLUME.ordinal()] = volumeViewer;
    m_viewerArray[Viewer.OBLIQUE.ordinal()] = obliqueViewer;
    m_viewerArray[Viewer.ORTHO.ordinal()] = orthoViewer;
    displayViewer(volumeViewer, VIEWER_3D, new Point(0, 0));
    displayViewer(obliqueViewer, VIEWER_OBLIQUE, new Point(MedicalHelper.WINDOW_WIDTH, 0));
    displayViewer(orthoViewer, VIEWER_ORTHO, new Point(0, MedicalHelper.WINDOW_HEIGHT));
  }

  private void displayViewer(final IRenderAreaInteractive renderArea, String viewerName, Point location)
  {
    final JFrame frame = new JFrame(EXAMPLE_NAME + " - " + viewerName);
    frame.setSize(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT);
    frame.setLocation(location);

    final JPanel panel = new JPanel();
    panel.setLayout(new BorderLayout());
    panel.add(renderArea.getComponent());

    frame.add(panel);
    frame.addWindowListener(new WindowAdapter()
    {
      @Override
      public void windowClosing(WindowEvent e)
      {
        renderArea.dispose();
        frame.dispose();
        if ( m_isStandAlone )
          System.exit(0);
      }
    });
    m_frameList.add(frame);

    EventQueue.invokeLater(new Runnable()
    {
      @Override
      public void run()
      {
        try
        {
          frame.setVisible(true);
        }
        catch (Exception e)
        {
          LOGGER.log(Level.SEVERE, "Failed to launch viewer", e);
        }
      }
    });
  }

  @Override
  public void stop()
  {
    for ( IRenderAreaInteractive renderArea : m_viewerArray )
      renderArea.dispose();
    for ( JFrame jFrame : m_frameList )
      jFrame.dispose();
  }
}
