package medical.rendering.visualization.simpleIsosurface;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.readers.SoVolumeReader;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Gnomon;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeIsosurface;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.readers.SoVRDicomFileReader;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  static final boolean HighRenderingQuality = true;
  public static final String EXAMPLE_NAME = "Simple Isosurface";
  public static final String DATA = "/medical/data/files/medicalFoot.ldm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private SoSeparator _root;
  private SoVolumeData _volData;

  private IRenderAreaExaminer _renderArea;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    // Load example resources
    File dataFile;
    try
    {
      dataFile = new File(Main.class.getResource(DATA).toURI());
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    String dataPath = dataFile.toString();
    buildSceneGraph(dataPath);

    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    _renderArea.setSceneGraph(_root);
    _renderArea.viewAll(new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));

    MedicalHelper.orientView(MedicalHelper.Axis.SAGITTAL,
        _renderArea.getRootSceneGraph().getCameraInteractor().getCamera(), _volData);

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  private void buildSceneGraph(String dataPath)
  {
    _root = new SoSeparator();

    // Node to hold the volume data
    _volData = new SoVolumeData();
    _volData.fileName.setValue(dataPath);

    // Get the volume data range.
    // For DICOM volumes use the canonical Hounsfield units range.
    long[] range = new long[2];
    SoVolumeReader reader = _volData.getReader();
    if ( reader instanceof SoVRDicomFileReader )
    {
      range[0] = -1000;
      range[1] = 3000;
    }
    else
      range = _volData.getMinMax();

    SoDataRange volRange = new SoDataRange();
    volRange.min.setValue(range[0]);
    volRange.max.setValue(range[1]);

    // Specify isosurface rendering style and isovalue(s).
    // The GPU surface algorithm allows multiple isosurfaces to be extracted
    // and rendered simultaneously.
    SoVolumeIsosurface isosurface = new SoVolumeIsosurface();
    isosurface.isovalues.set1Value(0, 134); // Bones
    isosurface.isovalues.set1Value(1, -400); // Skin

    // Node in charge of drawing the volume
    SoVolumeRender volRender = new SoVolumeRender();
    volRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.AUTOMATIC);
    volRender.subdivideTile.setValue(true);

    // Material which defines the isosurface diffuse color & transparency,
    SoMaterial matVolRend = new SoMaterial();
    matVolRend.diffuseColor.set1Value(0, 0.8f, 0.8f, 0.8f);
    matVolRend.transparency.set1Value(0, 0.0f);
    matVolRend.diffuseColor.set1Value(1, 1.f, 0.55f, 0.35f);
    matVolRend.transparency.set1Value(1, 0.5f);

    // Assemble the scene graph
    // Note: SoVolumeRender must appear after the SoVolumeData node.
    _root = new SoSeparator();
    { // Assemble scene graph
      _root.addChild(matVolRend);
      _root.addChild(volRange);
      _root.addChild(_volData);
      _root.addChild(isosurface);
      _root.addChild(volRender);
      _root.addChild(new Gnomon());
    }

    // Add Open Inventor logo at the left-bottom corner
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    _root.addChild(logoBackground);

    // Note
    TextBox text = new TextBox();
    text.position.setValue(0, -0.99f, 0); // Normalized device coordinates
    text.alignmentH.setValue(TextBox.AlignmentH.CENTER);
    text.alignmentV.setValue(TextBox.AlignmentV.BOTTOM);
    text.addLine("GPU visualization of two isosurfaces.");
    _root.addChild(text);
  }
}
