package medical.segmentation.medicalfreehandcutting;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbEventListener;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoGLRenderAction.TransparencyTypes;
import com.openinventor.inventor.actions.SoHandleEventAction;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.drawers.SoEllipseScreenDrawer;
import com.openinventor.inventor.drawers.SoLassoScreenDrawer;
import com.openinventor.inventor.drawers.SoPolyLineScreenDrawer;
import com.openinventor.inventor.drawers.SoPolyLineScreenDrawer.EventArg;
import com.openinventor.inventor.drawers.SoPolygonScreenDrawer;
import com.openinventor.inventor.drawers.SoRectangleScreenDrawer;
import com.openinventor.inventor.elements.SoViewVolumeElement;
import com.openinventor.inventor.errors.SoError;
import com.openinventor.inventor.misc.SbExtrusionGenerator;
import com.openinventor.inventor.nodes.SoCSGShape;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSeparator.FastEditings;
import com.openinventor.inventor.nodes.SoShape;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer.InteractionMode;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor.CameraMode;
import com.openinventor.ldm.SoLDMGlobalResourceParameters;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.Gnomon;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.nodes.SoVolumeClippingGroup;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Medical example program.<br>
 * Purpose : How to extract a volume from a medical dataset.
 *
 */
public class Main extends Example
{

  private enum Drawers
  {
    LASSO("Lasso"),
    POLYGON("Polygon"),
    ELLIPSE_CORNER("Ellipse - From corner"),
    ELLIPSE_CENTER("Ellipse - From center"),
    RECTANGLE_CORNER("Rectangle - From corner"),
    RECTANGLE_CENTER("Rectangle - From center");

    private String m_label;

    private Drawers(String label)
    {
      m_label = label;
    }

    @Override
    public String toString()
    {
      return m_label;
    }
  }

  private static final String EXAMPLE_NAME = "Medical Free Hand Cutting";
  private static final String SCENE_GRAPH = "/medical/segmentation/medicalfreehandcutting/model.iv";
  private static final String ANNOTATION = "/medical/data/dicomSample/CVH001.dcm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());

  private IRenderAreaExaminer _renderArea;
  private SoCSGShape.CsgOperationTypes m_currentOperation;

  private SoSeparator m_sceneRoot;
  private SoGroup m_csgShapeRoot;
  private SoGroup m_volumeDataRoot;
  private SoVolumeData m_volumeData;
  private SoSwitch m_volumeClippingSwitch;
  private SoSwitch m_lineDrawerSwitch;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    // Load example resources
    File sceneFile;
    File annotationFile;
    try
    {
      sceneFile = new File(Main.class.getResource(SCENE_GRAPH).toURI());
      annotationFile = new File(Main.class.getResource(ANNOTATION).toURI());
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    // Create render area with an orthographic camera
    _renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    _renderArea.setTransparencyType(TransparencyTypes.DELAYED_BLEND);
    SceneExaminer root = _renderArea.getRootSceneGraph();
    root.setCameraMode(CameraMode.ORTHOGRAPHIC);
    root.setInteractionMode(InteractionMode.SELECTION);

    // Build scene graph
    String sceneGraphFilePath = sceneFile.toString();
    String annotationFilePath = annotationFile.toString();
    SoSeparator sceneGraph = buildSceneGraph(sceneGraphFilePath, annotationFilePath);

    _renderArea.setSceneGraph(sceneGraph);

    // Adjust camera
    SoCamera camera = root.getCameraInteractor().getCamera();
    MedicalHelper.orientView(MedicalHelper.Axis.CORONAL, camera, m_volumeData);

    final Component canvas = _renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));

    JPanel optionsPanel = buildOptionsPanel();

    setLayout(new BorderLayout());
    add(canvas, BorderLayout.CENTER);
    add(optionsPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    _renderArea.dispose();
  }

  private SoSeparator buildSceneGraph(String sceneGraphFilePath, String annotationFilePath)
  {
    SoPreferences.setValue("OIV_BUFFER_REGION_ENABLE", "0");
    SoCSGShape.setRescueOperation(SoCSGShape.CsgOperationTypes.LEFT_ONLY);

    // Medical Gnomon.
    Gnomon gnomon = new Gnomon();

    // DICOM annotation
    SoNode annotation = MedicalHelper.exampleDicomAnnotation(annotationFilePath);

    // Instructions
    TextBox text = new TextBox();
    text.position.setValue(-0.98f, 0.f, 0.f); // Normalized device coords -1..1
    text.alignmentH.setValue(TextBox.AlignmentH.LEFT);
    text.alignmentV.setValue(TextBox.AlignmentV.MIDDLE);
    text.addLine("Click and drag to outline clip region.");

    m_sceneRoot = MedicalHelper.readFile(sceneGraphFilePath);
    if ( m_sceneRoot == null )
      return null;
    m_sceneRoot.setName("ROOT");

    // Define Open Inventor logo
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }

    m_volumeDataRoot = MedicalHelper.find(m_sceneRoot, SoGroup.class, "VOLUME_DATA_ROOT");
    // If cannot find VOLUME_DATA_ROOT, use root.
    if ( m_volumeDataRoot == null )
      m_volumeDataRoot = m_sceneRoot;

    m_volumeData = MedicalHelper.find(m_sceneRoot, SoVolumeData.class, "VOLUME_DATA");

    m_volumeClippingSwitch = MedicalHelper.find(m_sceneRoot, SoSwitch.class, "VOLUME_CLIPPING_ROOT");

    m_csgShapeRoot = MedicalHelper.find(m_sceneRoot, SoGroup.class, "CSG_ROOT");
    if ( m_csgShapeRoot == null )
      m_csgShapeRoot = m_sceneRoot;

    // Line drawers
    LineDrawerListener drawerListener = new LineDrawerListener();
    // lasso
    SoLassoScreenDrawer lasso = new SoLassoScreenDrawer();
    lasso.color.setValue(1, 0, 0);
    lasso.setName("LASSO_SCREEN_DRAWER");
    lasso.onFinish.addEventListener(drawerListener);
    lasso.isClosed.setValue(true);

    // polygon
    SoPolygonScreenDrawer polygon = new SoPolygonScreenDrawer();
    polygon.color.setValue(0, 1, 0);
    polygon.setName("POLYGON_SCREEN_DRAWER");
    polygon.onFinish.addEventListener(drawerListener);

    // ellipse corner
    SoEllipseScreenDrawer ellipseCorner = new SoEllipseScreenDrawer();
    ellipseCorner.color.setValue(0, 0, 1);
    ellipseCorner.method.setValue(SoEllipseScreenDrawer.CreationMethods.CORNER_CORNER);
    ellipseCorner.setName("ELLIPSE_CORNER_SCREEN_DRAWER");
    ellipseCorner.onFinish.addEventListener(drawerListener);

    // ellipse center
    SoEllipseScreenDrawer ellipseCenter = new SoEllipseScreenDrawer();
    ellipseCenter.color.setValue(1, 1, 0);
    ellipseCenter.method.setValue(SoEllipseScreenDrawer.CreationMethods.CENTER_CORNER);
    ellipseCenter.setName("ELLIPSE_CENTER_SCREEN_DRAWER");
    ellipseCenter.onFinish.addEventListener(drawerListener);

    // Rectangle corner
    SoRectangleScreenDrawer rectangleCorner = new SoRectangleScreenDrawer();
    rectangleCorner.color.setValue(1, 1, 1);
    rectangleCorner.method.setValue(SoRectangleScreenDrawer.CreationMethods.CORNER_CORNER);
    rectangleCorner.setName("RECTANGLE_CORNER_SCREEN_DRAWER");
    rectangleCorner.onFinish.addEventListener(drawerListener);

    // Rectangle center
    SoRectangleScreenDrawer rectangleCenter = new SoRectangleScreenDrawer();
    rectangleCenter.color.setValue(0.5f, 1, 0);
    rectangleCenter.method.setValue(SoRectangleScreenDrawer.CreationMethods.CENTER_CORNER);
    rectangleCenter.setName("RECTANGLE_CENTER_SCREEN_DRAWER");
    rectangleCenter.onFinish.addEventListener(drawerListener);

    m_lineDrawerSwitch = new SoSwitch();
    m_lineDrawerSwitch.setName("LINE_DRAWER_SWITCH");
    {
      // add drawers in same order as Drawers enum
      m_lineDrawerSwitch.addChild(lasso);
      m_lineDrawerSwitch.addChild(polygon);
      m_lineDrawerSwitch.addChild(ellipseCorner);
      m_lineDrawerSwitch.addChild(ellipseCenter);
      m_lineDrawerSwitch.addChild(rectangleCorner);
      m_lineDrawerSwitch.addChild(rectangleCenter);
    }
    // lasso by default
    m_lineDrawerSwitch.whichChild.setValue(0);

    SoSeparator drawers = new SoSeparator();
    drawers.fastEditing.setValue(FastEditings.CLEAR_ZBUFFER);
    drawers.boundingBoxIgnoring.setValue(true);
    drawers.setName("DRAWERS");
    {
      drawers.addChild(m_lineDrawerSwitch);
    }

    // Assemble the scene graph
    SoSeparator root = new SoSeparator();
    {
      root.addChild(gnomon);
      root.addChild(annotation);
      root.addChild(text);
      root.addChild(m_sceneRoot);
      if ( logoBackground != null )
        root.addChild(logoBackground);
      root.addChild(drawers);
    }

    return root;
  }

  /**
   * Add a new SoCSGShape which right child is specified shape. By adding
   * extruded shapes, we will build the following CSGTree:
   *
   * <pre>
   *               S
   *            S__|__
   *         S__|__   |
   *      R__|__   |  N4
   *    __|__   |  N3
   *   |     |  N2
   *  NULL   N1
   * </pre>
   * <p>
   * Where S are SoCSGShape, N1, N2, ... The first added is N1, the last is N4.
   * The last added shape must be in top-right of tree to respect
   * non-associativity and non-commutativity of CSG Operations.<br>
   * For example, the following operations: N1 + N2 - N3, correspond to ((N1 +
   * N2) - N3), which tree is (R represent RIGHT_ONLY):
   *
   * <pre>
   *             -
   *          +__|__
   *       R__|__   |
   *     __|__   |  N3
   *    |     |  N2
   *   NULL   N1
   * </pre>
   */
  private void addShape(SoGroup csgTreeRoot, SoNode newShape, SoCSGShape.CsgOperationTypes operation)
  {
    SoCSGShape firstCSGShape = MedicalHelper.find(csgTreeRoot, SoCSGShape.class);
    if ( firstCSGShape == null )
    {
      firstCSGShape = new SoCSGShape();
      firstCSGShape.rightOperand.setValue(newShape);
      firstCSGShape.csgOperation.setValue(SoCSGShape.CsgOperationTypes.RIGHT_ONLY);
      csgTreeRoot.addChild(firstCSGShape);
    }
    else
    {
      SoCSGShape newCSGShape = new SoCSGShape();
      newCSGShape.leftOperand.setValue(firstCSGShape);
      newCSGShape.csgOperation.setValue(operation);
      newCSGShape.rightOperand.setValue(newShape);
      csgTreeRoot.replaceChild(firstCSGShape, newCSGShape);
    }
  }

  private JPanel buildOptionsPanel()
  {
    JPanel panel = new JPanel();

    GridBagLayout gridBagLayout = new GridBagLayout();
    gridBagLayout.columnWidths = new int[] { 0, 0, 0, 0, 0 };
    gridBagLayout.rowHeights = new int[] { 0, 0, 0, 0 };
    panel.setLayout(gridBagLayout);

    JLabel lblFreeHandCut = new JLabel("Free Hand Cut Tools");
    GridBagConstraints gbc_lblFreeHandCut = new GridBagConstraints();
    gbc_lblFreeHandCut.insets = new Insets(5, 5, 5, 5);
    gbc_lblFreeHandCut.anchor = GridBagConstraints.WEST;
    gbc_lblFreeHandCut.gridx = 0;
    gbc_lblFreeHandCut.gridy = 0;
    panel.add(lblFreeHandCut, gbc_lblFreeHandCut);

    JComboBox<Drawers> drawersComboBox = new JComboBox<Drawers>(Drawers.values());
    drawersComboBox.setSelectedIndex(m_lineDrawerSwitch.whichChild.getValue());
    drawersComboBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        m_lineDrawerSwitch.whichChild.setValue(drawersComboBox.getSelectedIndex());
      }
    });
    GridBagConstraints gbc_drawersComboBox = new GridBagConstraints();
    gbc_drawersComboBox.insets = new Insets(5, 0, 5, 5);
    gbc_drawersComboBox.anchor = GridBagConstraints.WEST;
    gbc_drawersComboBox.gridx = 1;
    gbc_drawersComboBox.gridy = 0;
    panel.add(drawersComboBox, gbc_drawersComboBox);

    JLabel lblShow = new JLabel("Show");
    GridBagConstraints gbc_lblShow = new GridBagConstraints();
    gbc_lblShow.weightx = 1.0;
    gbc_lblShow.anchor = GridBagConstraints.EAST;
    gbc_lblShow.insets = new Insets(5, 0, 5, 5);
    gbc_lblShow.gridx = 4;
    gbc_lblShow.gridy = 0;
    panel.add(lblShow, gbc_lblShow);

    JComboBox<String> showComboBox = new JComboBox<String>();
    GridBagConstraints gbc_showComboBox = new GridBagConstraints();
    gbc_showComboBox.insets = new Insets(5, 0, 5, 5);
    gbc_showComboBox.anchor = GridBagConstraints.NORTH;
    gbc_showComboBox.gridx = 5;
    gbc_showComboBox.gridy = 0;
    panel.add(showComboBox, gbc_showComboBox);
    SoSwitch showSwitch = MedicalHelper.find(m_sceneRoot, SoSwitch.class, "SHOW_SWITCH");
    if ( showSwitch == null )
    {
      SoError.post("SoSwitch node named SHOW_SWITCH not found");
      showComboBox.setEnabled(false);
    }
    else
    {
      final int numChildren = showSwitch.getNumChildren();
      for ( int i = 0; i < numChildren; i++ )
        showComboBox.addItem(showSwitch.getChild(i).getName());
      showComboBox.setSelectedIndex(showSwitch.whichChild.getValue());
      showComboBox.addActionListener(new ActionListener()
      {
        @Override
        public void actionPerformed(ActionEvent e)
        {
          showSwitch.whichChild.setValue(showComboBox.getSelectedIndex());
        }
      });
    }

    JCheckBox chckbxClipOutside = new JCheckBox("Clip Outside");
    GridBagConstraints gbc_chckbxClipOutside = new GridBagConstraints();
    gbc_chckbxClipOutside.anchor = GridBagConstraints.NORTHWEST;
    gbc_chckbxClipOutside.insets = new Insets(0, 0, 5, 5);
    gbc_chckbxClipOutside.gridx = 0;
    gbc_chckbxClipOutside.gridy = 1;
    panel.add(chckbxClipOutside, gbc_chckbxClipOutside);
    SoVolumeClippingGroup volumeClippingGroup = MedicalHelper.find(m_sceneRoot, SoVolumeClippingGroup.class);
    if ( volumeClippingGroup == null )
    {
      SoError.post("SoVolumeClippingGroup node not found");
      chckbxClipOutside.setEnabled(false);
    }
    else
    {
      chckbxClipOutside.setSelected(volumeClippingGroup.clipOutside.getValue());
      chckbxClipOutside.addActionListener(new ActionListener()
      {
        @Override
        public void actionPerformed(ActionEvent e)
        {
          volumeClippingGroup.clipOutside.setValue(chckbxClipOutside.isSelected());
        }
      });
    }

    JCheckBox chckbxShowLdmTopology = new JCheckBox("Show LDM Topology");
    GridBagConstraints gbc_chckbxShowLdmTopology = new GridBagConstraints();
    gbc_chckbxShowLdmTopology.anchor = GridBagConstraints.NORTHEAST;
    gbc_chckbxShowLdmTopology.gridwidth = 2;
    gbc_chckbxShowLdmTopology.insets = new Insets(0, 0, 5, 5);
    gbc_chckbxShowLdmTopology.gridx = 4;
    gbc_chckbxShowLdmTopology.gridy = 1;
    gbc_chckbxShowLdmTopology.weightx = 1.0;
    panel.add(chckbxShowLdmTopology, gbc_chckbxShowLdmTopology);
    SoVolumeRender volumeRender = MedicalHelper.find(m_sceneRoot, SoVolumeRender.class);
    if ( volumeRender == null )
    {
      SoError.post("SoVolumeRender node not found");
      chckbxShowLdmTopology.setEnabled(false);
    }
    else
    {
      chckbxShowLdmTopology.setSelected(SoLDMGlobalResourceParameters
          .getVisualFeedbackParam(SoLDMGlobalResourceParameters.VisualFeedbackParams.DRAW_TOPOLOGY));
      chckbxShowLdmTopology.addActionListener(new ActionListener()
      {
        @Override
        public void actionPerformed(ActionEvent e)
        {
          SoLDMGlobalResourceParameters.setVisualFeedbackParam(
              SoLDMGlobalResourceParameters.VisualFeedbackParams.DRAW_TOPOLOGY, chckbxShowLdmTopology.isSelected());
        }
      });
    }

    JPanel actionPanel = new JPanel();
    actionPanel.setBorder(new TitledBorder(null, "Action", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    actionPanel.setLayout(new GridLayout(0, 1, 0, 0));
    GridBagConstraints gbc_actionPanel = new GridBagConstraints();
    gbc_actionPanel.gridheight = 2;
    gbc_actionPanel.insets = new Insets(5, 0, 0, 5);
    gbc_actionPanel.fill = GridBagConstraints.BOTH;
    gbc_actionPanel.gridx = 2;
    gbc_actionPanel.gridy = 0;
    gbc_actionPanel.weighty = 1.0;
    gbc_actionPanel.weightx = 0.0;
    panel.add(actionPanel, gbc_actionPanel);

    ButtonGroup actionButtonGroup = new ButtonGroup();
    JRadioButton rdbtnSelect = new JRadioButton("Select");
    rdbtnSelect.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( rdbtnSelect.isSelected() )
          m_currentOperation = SoCSGShape.CsgOperationTypes.INTERSECTION;
      }
    });
    actionButtonGroup.add(rdbtnSelect);
    actionPanel.add(rdbtnSelect);

    JRadioButton rdbtnAdd = new JRadioButton("Add");
    rdbtnAdd.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( rdbtnAdd.isSelected() )
          m_currentOperation = SoCSGShape.CsgOperationTypes.ADD;
      }
    });
    actionButtonGroup.add(rdbtnAdd);
    actionPanel.add(rdbtnAdd);

    JRadioButton rdbtnSubstract = new JRadioButton("Substract");
    rdbtnSubstract.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( rdbtnSubstract.isSelected() )
          m_currentOperation = SoCSGShape.CsgOperationTypes.SUB;
      }
    });
    actionButtonGroup.add(rdbtnSubstract);
    actionPanel.add(rdbtnSubstract);

    // INTERSECTION by default
    rdbtnSelect.setSelected(true);
    m_currentOperation = SoCSGShape.CsgOperationTypes.INTERSECTION;

    JButton btnClear = new JButton("Clear");
    btnClear.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        m_csgShapeRoot.removeAllChildren();
        m_volumeClippingSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });
    GridBagConstraints gbc_btnClear = new GridBagConstraints();
    gbc_btnClear.insets = new Insets(0, 0, 5, 5);
    gbc_btnClear.gridheight = 2;
    gbc_btnClear.gridx = 3;
    gbc_btnClear.gridy = 0;
    panel.add(btnClear, gbc_btnClear);

    return panel;
  }

  class LineDrawerListener implements SbEventListener<EventArg>
  {
    @Override
    public void onEvent(EventArg event)
    {
      SoPolyLineScreenDrawer lineDrawer = event.getSource();
      SoHandleEventAction action = event.getAction();

      final int nbPoints = lineDrawer.point.getNum();

      // If less than 1 point, shape cannot be generated.
      if ( nbPoints < 1 )
      {
        lineDrawer.clear();
        return;
      }

      // retrieve points of line in cam space
      Collection<SbVec2f> pointsInCam = new ArrayList<>();
      for ( int i = 0; i < nbPoints; i++ )
        pointsInCam.add(lineDrawer.point.getValueAt(i));

      // create a new extruded shape :

      // retrieve path of extruded shape root
      SoSearchAction searchAction = new SoSearchAction();
      searchAction.setNode(m_csgShapeRoot);
      searchAction.apply(m_sceneRoot);
      SoPath pathToExtrudedShapeRoot = searchAction.getPath();

      // retrieve bounding box of volumeData node.
      SbBox3f bbox = MedicalHelper.getBoundingBox(m_volumeDataRoot);

      // create an extruded shape from specified line. Line is extruded along
      // view direction between bounding box enclosing planes.
      SoShape extrudedShape = SbExtrusionGenerator.createFrom2DPoints(pointsInCam, pathToExtrudedShapeRoot,
          SoViewVolumeElement.get(action.getState()), bbox);
      if ( extrudedShape == null )
      {
        lineDrawer.clear();
        return;
      }

      // Clip extrudedShape with volumeData bbox.
      SoSeparator bboxSep = MedicalHelper.createCube(bbox);
      SoCSGShape extrudedShapeClipped = new SoCSGShape();
      extrudedShapeClipped.leftOperand.setValue(extrudedShape);
      extrudedShapeClipped.rightOperand.setValue(bboxSep);
      extrudedShapeClipped.csgOperation.setValue(SoCSGShape.CsgOperationTypes.INTERSECTION);

      // if it's the first time a CSGShape is added to m_csgShapeRoot,
      // Initialize CSGTree with the box representing volumeData.
      SoCSGShape initialBox = MedicalHelper.find(m_csgShapeRoot, SoCSGShape.class);
      if ( initialBox == null )
      {
        initialBox = new SoCSGShape();
        initialBox.rightOperand.setValue(bboxSep);
        initialBox.csgOperation.setValue(SoCSGShape.CsgOperationTypes.RIGHT_ONLY);
        m_csgShapeRoot.addChild(initialBox);

        // and enable VolumeClippingGroup
        m_volumeClippingSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
      }

      // Add this clippedShape to current CSGTree
      addShape(m_csgShapeRoot, extrudedShapeClipped, m_currentOperation);

      // don't forget to clear line
      lineDrawer.clear();

      action.setHandled();
    }
  }
}
