package medical.tools.rotateroi;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Panel;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JPanel;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.draggers.SoTransformerDragger;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaInteractive;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.volumeviz.nodes.SoVolumeClippingGroup;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.ViewerComponentsFactory;

public class RotateRoiPanel extends Panel
{
  private static final Logger LOGGER = Logger.getLogger(RotateRoiPanel.class.getName());
  private IRenderAreaInteractive m_renderArea;
  private JPanel m_renderAreaPanel;
  private JButton m_resetRoiButton;
  private SoTransformerDragger m_dragger;
  private SbBox3f m_initialRoiBox;

  /**
   * Constructor
   *
   * @param dataPath
   */
  public RotateRoiPanel(String dataPath)
  {
    initializeUI();
    SoSeparator rootSep = buildSceneGraph(dataPath);
    m_renderArea = ViewerComponentsFactory.createRenderAreaInteractive();
    m_renderArea.setSceneGraph(rootSep);

    SoCamera camera = m_renderArea.getRootSceneGraph().getCamera();
    camera.viewAll(rootSep, new SbViewportRegion(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    camera.nearDistance.setValue(0.1f);
    camera.farDistance.setValue(200f);

    m_renderAreaPanel.add(m_renderArea.getComponent());
  }

  /**
   * Initialize the panel
   */
  private void initializeUI()
  {
    setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

    m_renderAreaPanel = new JPanel();
    m_renderAreaPanel.setLayout(new BorderLayout());
    m_renderAreaPanel.setPreferredSize(new Dimension(0, 0));
    this.add(m_renderAreaPanel);

    JPanel settingsPanel = new JPanel();
    this.add(settingsPanel);

    JPanel resetRoiPanel = new JPanel();
    m_resetRoiButton = new JButton("Reset ROI");
    m_resetRoiButton.setFocusPainted(false);
    resetRoiPanel.add(m_resetRoiButton);
    settingsPanel.setLayout(new BoxLayout(settingsPanel, BoxLayout.Y_AXIS));
    settingsPanel.add(resetRoiPanel);

    // When the Reset button is clicked, the dragger is re-initialized
    m_resetRoiButton.addMouseListener(new MouseAdapter()
    {
      @Override
      public void mouseClicked(MouseEvent e)
      {
        SbVec3f center = m_initialRoiBox.getCenter();
        SbVec3f scale = m_initialRoiBox.getSize().over(2);
        m_dragger.scaleFactor.setValue(scale);
        m_dragger.rotation.setValue(SbRotation.identity());
        m_dragger.translation.setValue(center);
      }
    });
  }

  /**
   * @param dataPath
   * @return Scene graph
   */
  public SoSeparator buildSceneGraph(String dataPath)
  {
    SoSeparator rootSep = new SoSeparator();
    rootSep.setName("root");

    // Create scene elements

    SoSeparator sceneSep = new SoSeparator();
    sceneSep.setName("scene");
    rootSep.addChild(sceneSep);

    // Add the background logo node
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    sceneSep.addChild(logoBackground);

    SoSeparator volSep = new SoSeparator();
    volSep.setName("data");
    sceneSep.addChild(volSep);

    // Decrease the quality while moving to have better interactivity
    SoInteractiveComplexity interactComplexity = new SoInteractiveComplexity();
    interactComplexity.fieldSettings.set1Value(0, "SoComplexity value 0.25 0.5");
    interactComplexity.fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
    interactComplexity.refinementDelay.setValue(0);
    volSep.addChild(interactComplexity);

    // Complexity node for the interact node to control
    SoComplexity complexity = new SoComplexity();
    volSep.addChild(complexity);

    // Node holding the volume data
    SoVolumeData volData = new SoVolumeData();
    volData.fileName.setValue(dataPath);
    volSep.addChild(volData);

    // The following must be a group so the clipping group can affect subsequent
    // volume rendering node
    SoGroup rotatedRoiGroup = new SoGroup();
    rotatedRoiGroup.setName("rotated_roi");
    volSep.addChild(rotatedRoiGroup);

    // Group for ROI dragger (cannot be part of the volume clipping group)
    SoSeparator manipGroup = new SoSeparator();
    manipGroup.setName("roi_dragger");
    rotatedRoiGroup.addChild(manipGroup);

    // Generate a region of interest for the dragger
    m_initialRoiBox = makeRoiBox(volData);
    SbVec3f center = m_initialRoiBox.getCenter();
    SbVec3f scale = m_initialRoiBox.getSize().over(2);

    // Add dragger
    m_dragger = new SoTransformerDragger();
    m_dragger.translation.setValue(center);
    m_dragger.scaleFactor.setValue(scale);
    manipGroup.addChild(m_dragger);

    // Group for clipping geometry
    SoVolumeClippingGroup clipGroup = new SoVolumeClippingGroup();
    clipGroup.setName("roi_geometry");
    rotatedRoiGroup.addChild(clipGroup);

    // Controls size, position and rotation of clipping geometry.
    // Fields will be automatically updated when the user moves the dragger.
    SoTransform clipTrans = new SoTransform();
    clipTrans.scaleFactor.connectFrom(m_dragger.scaleFactor);
    clipTrans.rotation.connectFrom(m_dragger.rotation);
    clipTrans.translation.connectFrom(m_dragger.translation);
    clipGroup.addChild(clipTrans);

    // Clipping geometry.
    SoCube clipBox = new SoCube();
    clipGroup.addChild(clipBox);

    // Use a predefined colorMap (associate data values to colors)
    SoTransferFunction volTF = new SoTransferFunction();
    volTF.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);
    volTF.minValue.setValue(69); // Make low intensity voxels transparent
    volSep.addChild(volTF);

    // Improve data rendering by adding the following node
    SoVolumeRenderingQuality volQual = new SoVolumeRenderingQuality();
    volQual.interpolateOnMove.setValue(true);
    volQual.preIntegrated.setValue(true);
    volQual.deferredLighting.setValue(true);
    volQual.lightingModel.setValue(SoVolumeRenderingQuality.LightingModels.OPENGL);
    volQual.surfaceScalarExponent.setValue(5);
    volSep.addChild(volQual);

    // Node in charge of drawing the volume
    SoVolumeRender volRend = new SoVolumeRender();
    volRend.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.AUTOMATIC);
    volRend.lowScreenResolutionScale.setValue(2);
    volRend.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
    volSep.addChild(volRend);

    // Add volume boundaries
    SoSeparator dataBoundariesSep = getCube(volData.extent.getValue());
    dataBoundariesSep.setName("data_boundaries");
    sceneSep.addChild(dataBoundariesSep);

    return rootSep;
  }

  /**
   * Generate a pre-defined region of interest from a given volume
   *
   * @param volData
   */
  private SbBox3f makeRoiBox(SoVolumeData volData)
  {
    SbVec3i32 volDim = volData.data.getSize();
    SbVec3i32 volMin = new SbVec3i32(0, 0, 0);
    SbVec3i32 volMax = volDim.minus(new SbVec3i32(1, 1, 1));

    int roiWidth = volDim.getValueAt(0) / 5;
    int offset = (volDim.getValueAt(0) - roiWidth) / 2;
    SbVec3i32 roiMin = volMin.plus(new SbVec3i32(offset, 0, 0));
    SbVec3i32 roiMax = volMax.minus(new SbVec3i32(offset, 0, 0));

    // Convert ROI in IJK voxel coords to box in XYZ 3D coordinates.
    // Note the conversion method requires voxel coordinates as float, not int.
    SbVec3f roiMinF = new SbVec3f(roiMin.getValueAt(0), roiMin.getValueAt(1), roiMin.getValueAt(2));
    SbVec3f roiMaxF = new SbVec3f(roiMax.getValueAt(0), roiMax.getValueAt(1), roiMax.getValueAt(2));
    SbBox3f roiF = new SbBox3f(roiMinF, roiMaxF);
    return volData.voxelToXYZ(roiF);
  }

  /**
   * @param box
   * @return Bounding box node
   */
  private SoSeparator getCube(SbBox3f box)
  {
    SoLightModel lightModel = new SoLightModel();
    lightModel.model.setValue(SoLightModel.Models.BASE_COLOR);

    SoDrawStyle drawStyle = new SoDrawStyle();
    drawStyle.style.setValue(SoDrawStyle.Styles.LINES);

    SoMaterial material = new SoMaterial();
    material.diffuseColor.setValue(1, 0, 0); // RED

    SoCube cube = new SoCube();
    cube.width.setValue(box.getSize().getX());
    cube.height.setValue(box.getSize().getY());
    cube.depth.setValue(box.getSize().getZ());

    SoTransform transform = new SoTransform();
    transform.translation.setValue(box.getCenter());

    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    // Assemble elements
    SoSeparator boxSep = new SoSeparator();
    boxSep.addChild(material);
    boxSep.addChild(pickStyle);
    boxSep.addChild(lightModel);
    boxSep.addChild(drawStyle);
    boxSep.addChild(transform);
    boxSep.addChild(cube);
    return boxSep;
  }

  public void destroy()
  {
    // Destroy NEWT resources
    m_renderArea.dispose();
  }
}
