package medical.tools.volumePickingGPU;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.errors.SoError;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.SoCamera;
import com.openinventor.inventor.nodes.SoEventCallback;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoText2;
import com.openinventor.inventor.viewercomponents.awt.IRenderAreaExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.medical.nodes.TextBox;
import com.openinventor.volumeviz.details.SoVolumeDetail.TransparentLongValue;
import com.openinventor.volumeviz.details.SoVolumeRenderDetail;
import com.openinventor.volumeviz.nodes.SoVolumeData;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  public static final String INFO = "/medical/tools/volumePickingGPU/info.iv";
  public static final String SCENE_GRAPH = "/medical/tools/volumePickingGPU/simpleVolumeRender.iv";
  public static final String EXAMPLE_NAME = "Volume Picking GPU";
  public static final String DATA = "/medical/data/files/3DHEAD.ldm";

  private final static Logger LOGGER = Logger.getLogger(Main.class.getName());
  private SoSeparator _root;
  private SoText2 _infoText;
  private SoMaterial _materialVoxel;
  private boolean _mousePressed;

  private static String _infoFile;
  private static String _sceneGraphFile;
  private static String _dataFile;

  private IRenderAreaExaminer m_renderArea;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain(EXAMPLE_NAME);
  }

  @Override
  public void start()
  {
    // Load example resources
    try
    {
      _infoFile = (new File(Main.class.getResource(INFO).toURI().getPath())).toString();
      _sceneGraphFile = (new File(Main.class.getResource(SCENE_GRAPH).toURI())).toString();
      _dataFile = (new File(Main.class.getResource(DATA).toURI())).toString();
    }
    catch (Exception e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load resources", e);
      return;
    }

    buildSceneGraph();

    m_renderArea = ViewerComponentsFactory.createRenderAreaExaminer();
    m_renderArea.setSceneGraph(_root);
    m_renderArea.getRootSceneGraph().setInteractionMode(SceneExaminer.InteractionMode.SELECTION);

    // Adjust camera to have first nice and detailed view.
    SoCamera camera = m_renderArea.getRootSceneGraph().getCameraInteractor().getCamera();
    camera.position.setValue(-0.41f, 0.01f, 0.62f);
    camera.orientation.setValue(new SbVec3f(-0.72f, -0.62f, -0.3f), 0.29f);
    camera.focalDistance.setValue(1.0f);
    camera.nearDistance.setValue(0.1f);
    m_renderArea.saveCamera();

    final Component canvas = m_renderArea.getComponent();
    canvas.setPreferredSize(new java.awt.Dimension(MedicalHelper.WINDOW_WIDTH, MedicalHelper.WINDOW_HEIGHT));
    setLayout(new BorderLayout());
    add(canvas);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  /**
   * Build the scene graph to be displayed.
   */
  private void buildSceneGraph()
  {
    _root = new SoSeparator();

    // Add event handler
    SoEventCallback eventCB = new SoEventCallback();
    eventCB.addEventCallback(SoMouseButtonEvent.class, new MyMousePressCB());
    eventCB.addEventCallback(SoLocation2Event.class, new MyMouseMoveCB());
    _root.addChild(eventCB);

    // Load main scene graph from file.
    SoSeparator scene = MedicalHelper.readFile(_sceneGraphFile);
    if ( scene == null )
      return;

    SoVolumeData volumeData = MedicalHelper.find(scene, SoVolumeData.class);
    if ( volumeData == null )
      SoError.post("SoVolumeData node not found");
    else
      volumeData.fileName.setValue(_dataFile);

    _root.addChild(scene);

    // Add annotation display
    _root.addChild(displayInfo(_infoFile));

    // Add Open Inventor logo at the left-bottom corner
    SoNode logoBackground = null;
    try
    {
      logoBackground = MedicalHelper.getExampleLogoNode();
    }
    catch (FileNotFoundException e)
    {
      LOGGER.log(Level.SEVERE, "Failed to load logo", e);
    }
    _root.addChild(logoBackground);

    TextBox text = new TextBox();
    text.position.setValue(0.0f, -0.99f, 0.0f); // Normalized device coordinates
    text.alignmentH.setValue(TextBox.AlignmentH.CENTER);
    text.alignmentV.setValue(TextBox.AlignmentV.BOTTOM);
    text.addLine("\"Voxelized rendering\" displays each voxel as a cube for accurate selection.");
    text.addLine("Click or Press-and-drag to interactively display voxel properties.");
    _root.addChild(text);
  }

  /**
   * Load annotation scene graph from file
   */
  SoSeparator displayInfo(String infoFile)
  {
    SoSeparator infoSep = MedicalHelper.readFile(infoFile);
    if ( infoSep == null )
      return null;

    // Find material and text nodes that event handlers will modify
    _materialVoxel = MedicalHelper.find(infoSep, SoMaterial.class, "VOXELMAT");
    if ( _materialVoxel == null )
      SoError.post("SoMaterial node not found");

    _infoText = MedicalHelper.find(infoSep, SoText2.class);
    if ( _infoText == null )
      SoError.post("SoText2 node not found");

    return infoSep;
  }

  void updateInfo(SoPickedPoint p)
  {
    if ( p == null )
      return;

    // Get detail object for picked point.
    SoVolumeRenderDetail detail = (SoVolumeRenderDetail) p.getDetail();
    if ( detail != null )
    {
      TransparentLongValue value = detail.getFirstNonTransparentValue();
      SbVec3i32 dataPos = value.pos;
      SbVec3f objPos = value.objPos;
      long yValue = value.value;

      // Get the color from the colorMap.
      // First need to get the index of the corresponding data value using
      // the inherited dataRange.
      // Then, get the corresponding color of the index and update the
      // text information and the top left color marker material.
      SoTransferFunction tFunction = MedicalHelper.find(_root, SoTransferFunction.class);
      if ( tFunction == null )
        SoError.post("SoTransferFunction node not found");
      SoDataRange range = MedicalHelper.find(_root, SoDataRange.class);
      if ( range == null )
        SoError.post("SoDataRange node not found");

      float[] rgba = tFunction.actualColorMap.getValues(0);
      int numColors = rgba.length / 4;
      int index =
          (int) Math.abs((yValue - range.min.getValue()) * (numColors) / (range.max.getValue() - range.min.getValue()));
      if ( index > numColors - 1 )
        index = numColors - 1;

      float r = rgba[index * 4];
      float g = rgba[index * 4 + 1];
      float b = rgba[index * 4 + 2];
      float a = rgba[index * 4 + 3];

      // Display info for closest intersected voxel.
      _materialVoxel.diffuseColor.set1Value(0, new SbColor(r, g, b));

      _infoText.string.set1Value(0,
          "Position IJK : " + dataPos.getX() + " , " + dataPos.getY() + " , " + dataPos.getZ());
      _infoText.string.set1Value(1, "Position XYZ: " + objPos.getX() + " , " + objPos.getY() + " , " + objPos.getZ());
      _infoText.string.set1Value(2, "RGBA            : " + r + " , " + g + " , " + b + " , " + a);
    }
    else
    {
      _materialVoxel.diffuseColor.set1Value(0, 1, 1, 1);

      _infoText.string.set1Value(0, "Position IJK : ");
      _infoText.string.set1Value(1, "Position XYZ: ");
      _infoText.string.set1Value(2, "RGBA            : ");
    }
  }

  /**
   * MouseMove event handler
   */
  private class MyMouseMoveCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      if ( _mousePressed ) // If mouse button is currently pressed
      {
        // update info according the picked point.
        updateInfo(node.getPickedPoint());
      }
      node.setHandled();
    }
  }

  /**
   * MousePress event handler
   */
  private class MyMousePressCB extends SoEventCallbackCB
  {
    @Override
    public void invoke(SoEventCallback node)
    {
      SoMouseButtonEvent mouseEvent = (SoMouseButtonEvent) node.getEvent();

      if ( SoMouseButtonEvent.isButtonPressEvent(mouseEvent, mouseEvent.getButton()) )
      {
        _mousePressed = true;
        updateInfo(node.getPickedPoint());
      }
      else if ( SoMouseButtonEvent.isButtonReleaseEvent(mouseEvent, mouseEvent.getButton()) )
      {
        _mousePressed = false;
      }
      node.setHandled();
    }
  }
}
