package medical.web.medicalremotermpr.service;

import java.io.FileNotFoundException;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SbViewportRegion;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.actions.SoGetBoundingBoxAction;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.errors.SoError;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.nodes.SoGroup;
import com.openinventor.inventor.nodes.SoLineSet;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoVertexProperty;
import com.openinventor.inventor.viewercomponents.SoCameraInteractor;
import com.openinventor.inventor.viewercomponents.nodes.SceneExaminer;
import com.openinventor.inventor.viewercomponents.nodes.SceneInteractor;
import com.openinventor.medical.helpers.MedicalHelper;
import com.openinventor.remoteviz.rendering.RenderArea;
import com.openinventor.remoteviz.rendering.RenderAreaListener;
import com.openinventor.remoteviz.rendering.ServiceListener;
import com.openinventor.volumeviz.draggers.SoOrthoSliceDragger;
import com.openinventor.volumeviz.nodes.SoOrthoSlice;
import com.openinventor.volumeviz.nodes.SoSlice;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;

public class MedicalRemoteMprService extends ServiceListener
{
  private final static Logger LOGGER = Logger.getLogger(MedicalRemoteMprService.class.getName());
  private final String m_ivFilePath;
  private boolean firstRead = true;
  private SoVolumeData m_localVolumeData = null;
  private SoGroup m_commonGroup = null;
  private SceneExaminer m_rootTL = null;
  private SceneExaminer m_rootTR = null;
  private SceneExaminer m_rootBL = null;
  private SceneExaminer m_rootBR = null;
  private SoSeparator m_sagittalBorderSep = null;
  private SoSeparator m_coronalBorderSep = null;
  private SoSeparator m_transverseBorderSep = null;

  public MedicalRemoteMprService(String ivFilePath)
  {
    super();
    m_ivFilePath = ivFilePath;

    // Create borders
    m_sagittalBorderSep = new SoSeparator();
    m_sagittalBorderSep.setName("sagittalBorder");
    m_coronalBorderSep = new SoSeparator();
    m_coronalBorderSep.setName("coronalBorder");
    m_transverseBorderSep = new SoSeparator();
    m_transverseBorderSep.setName("coronalBorder");
    createBorderSep(m_sagittalBorderSep, new float[] { 1, 0, 0 });
    createBorderSep(m_coronalBorderSep, new float[] { 0, 1, 0 });
    createBorderSep(m_transverseBorderSep, new float[] { 0, 0, 1 });
  }

  private void createBorderSep(SoSeparator borderSep, float borderColor[])
  {
    borderSep.fastEditing.setValue(SoSeparator.FastEditings.KEEP_ZBUFFER);

    // Border Color
    SoMaterial borderMaterial = new SoMaterial();
    borderMaterial.diffuseColor.setValue(borderColor);
    borderSep.addChild(borderMaterial);

    // Lineset that define the border
    SoLineSet border = new SoLineSet();
    border.numVertices.setValue(5);
    SoVertexProperty borderVertexProperty = new SoVertexProperty();
    border.vertexProperty.setValue(borderVertexProperty);
    borderSep.addChild(border);
  }

  // Dragger callback
  class CustomSoDraggerCB extends SoDraggerCB
  {
    @Override
    public void invoke(SoDragger dragger)
    {
      SoOrthoSliceDragger orthoDragger = (SoOrthoSliceDragger) dragger;
      if ( orthoDragger == null )
        return;

      SoPath localOrthoPath = orthoDragger.orthoSlicePath.getValue();
      if ( localOrthoPath == null )
        return;
      SoOrthoSlice localOrtho = (SoOrthoSlice) localOrthoPath.regular.getTail();

      SoGetBoundingBoxAction bboxAction = new SoGetBoundingBoxAction(new SbViewportRegion((short) 1, (short) 1));
      bboxAction.apply(localOrthoPath);
      SbBox3f bbox = bboxAction.getBoundingBox();

      float[] bounds = bbox.getBounds();
      float xmin = bounds[0], ymin = bounds[1], zmin = bounds[2], xmax = bounds[3], ymax = bounds[4], zmax = bounds[5];

      SoSeparator localSep = null;
      switch ( localOrtho.axis.getValue() )
      {
      case 0 :// SAGITTAL
        localSep = m_sagittalBorderSep;
        break;
      case 1 :// CORONAL
        localSep = m_coronalBorderSep;
        break;
      case 2 :// TRANSVERS
        localSep = m_transverseBorderSep;
        break;
      }

      if ( localSep == null )
        return;

      if ( localSep.getNumChildren() < 2 )
        return;

      SoLineSet ls = (SoLineSet) localSep.getChild(1);
      if ( ls == null )
        return;

      SoVertexProperty localVertexProperty = (SoVertexProperty) ls.vertexProperty.getValue();
      if ( localVertexProperty == null )
        return;

      localVertexProperty.vertex.set1Value(0, xmin, ymin, zmin);
      localVertexProperty.vertex.set1Value(2, xmax, ymax, zmax);
      localVertexProperty.vertex.set1Value(4, xmin, ymin, zmin);
      if ( localOrtho.axis.getValue() == 0 ) // SAGITTAL
      {
        localVertexProperty.vertex.set1Value(1, xmin, ymax, zmin);
        localVertexProperty.vertex.set1Value(3, xmax, ymin, zmax);
      }
      else
      {
        localVertexProperty.vertex.set1Value(1, xmin, ymax, zmax);
        localVertexProperty.vertex.set1Value(3, xmax, ymin, zmin);
      }
    }
  }

  // Common method to create sceneGraph 2D viewers.
  private void createSceneGraph(SoGroup initialScene, SceneExaminer sceneExaminer, RenderArea renderArea, int axis,
      SoSeparator borderSep)
  {
    SoSeparator scene = new SoSeparator();
    scene.addChild(initialScene);

    SoOrthoSlice orthoSlice = new SoOrthoSlice();
    orthoSlice.interpolation.setValue(SoSlice.Interpolations.MULTISAMPLE_12);

    switch ( axis )
    {
    case 0 : // Sagittal
      orthoSlice.axis.setValue(0);
      orthoSlice.sliceNumber.setValue(m_localVolumeData.data.getSize().getValueAt(0) / 2);
      orthoSlice.setName("orthoSliceSagittal");
      break;
    case 1 :
      orthoSlice.axis.setValue(1);
      orthoSlice.sliceNumber.setValue(m_localVolumeData.data.getSize().getValueAt(1) / 2);
      orthoSlice.setName("orthoSliceCoronal");
      break;
    case 2 :
      orthoSlice.axis.setValue(2);
      orthoSlice.sliceNumber.setValue(m_localVolumeData.data.getSize().getValueAt(2) / 2);
      orthoSlice.setName("orthoSliceTransvers");
      break;
    }
    scene.addChild(orthoSlice);

    // Create path to slice node
    // Note: Can be a partial path but must include the slice node.
    SoPath path = new SoPath(scene);
    path.regular.append(orthoSlice);

    SoOrthoSliceDragger orthoDragger = new SoOrthoSliceDragger();
    orthoDragger.orthoSlicePath.setValue(path);
    orthoDragger.volumeDimension.setValue(m_localVolumeData.data.getSize());
    orthoDragger.volumeExtent.setValue(m_localVolumeData.extent.getValue());
    CustomSoDraggerCB customSoDraggerCB = new CustomSoDraggerCB();
    orthoDragger.addMotionCallback(customSoDraggerCB, null);
    scene.addChild(orthoDragger);

    scene.addChild(borderSep);
    sceneExaminer.addChild(scene);
    // Apply the sceneExaminer node as renderArea scene graph
    renderArea.getSceneManager().setSceneGraph(sceneExaminer);

    // initialize border
    customSoDraggerCB.invoke(orthoDragger);
  }

  @Override
  public void onInstantiatedRenderArea(RenderArea renderArea)
  {
    RenderAreaListener renderAreaListener = new RenderAreaListener();
    renderArea.addListener(renderAreaListener);
    renderArea.getTouchManager().addDefaultRecognizers();

    SoInput myInput = new SoInput();
    if ( myInput.openFile(m_ivFilePath) == true )
    {
      if ( firstRead )
      {
        // Read common sceneGraph (iv file)
        SoDB.readAll(myInput);
        m_commonGroup = (SoGroup) SoNode.getByName("CommonGroup");
        firstRead = false;

        // Adjust volume size and range
        m_localVolumeData = (SoVolumeData) SoNode.getByName("volumeData");
        MedicalHelper.dicomAdjustVolume(m_localVolumeData, true);
      }

      if ( m_commonGroup != null )
      {
        // 3D View (top left viewer)
        if ( renderArea.getId().equals("renderArea3D_TOP_LEFT") )
        {
          // Instantiate a sceneExaminer to interact with the camera
          m_rootTL = new SceneExaminer();
          m_rootTL.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);
          renderArea.getSceneManager().getGLRenderAction()
              .setTransparencyType(SoGLRenderAction.TransparencyTypes.DELAYED_SORTED_PIXELS_BLEND);
          renderArea.getSceneManager().setAntialiasing(0.9f);
          m_rootTL.getCameraInteractor().setOrientation(new SbRotation(new SbVec3f(1, 0, 0), (float) Math.PI / 2.0f));

          // Scene graph
          SoSeparator topLeftSG = new SoSeparator();
          topLeftSG.addChild(m_commonGroup);
          SoVolumeRender volumeRender = new SoVolumeRender();
          volumeRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.AUTOMATIC);
          volumeRender.numSlices.setValue(-1);
          volumeRender.samplingAlignment.setValue(SoVolumeRender.SamplingAlignments.BOUNDARY_ALIGNED);
          topLeftSG.addChild(volumeRender);
          m_rootTL.addChild(topLeftSG);
          m_rootTL.addChild(m_sagittalBorderSep);
          m_rootTL.addChild(m_coronalBorderSep);
          m_rootTL.addChild(m_transverseBorderSep);

          try
          {
            // OIV Logo
            SoNode logoBackground = MedicalHelper.getExampleLogoNode();
            m_rootTL.addChild(logoBackground);
          }
          catch (FileNotFoundException e)
          {
            LOGGER.log(Level.SEVERE, "Failed to load logo", e);
          }

          // Apply the sceneExaminer node as renderArea scene graph
          renderArea.getSceneManager().setSceneGraph(m_rootTL);
          m_rootTL.viewAll(renderArea.getSceneManager().getViewportRegion());
        }

        // SAGITTAL View (top right viewer)
        if ( renderArea.getId().equals("renderArea2D_TOP_RIGHT") )
        {
          // Instantiate a sceneExaminer to interact with the camera
          m_rootTR = new SceneExaminer();
          m_rootTR.enableOrbit(false);
          m_rootTR.setInteractionMode(SceneExaminer.InteractionMode.SELECTION);
          m_rootTR.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);

          SoCameraInteractor camSagittalInteractor = m_rootTR.getCameraInteractor();
          camSagittalInteractor.setOrientation(new SbRotation(new SbVec3f(0, 1, 0), (float) Math.PI / 2.0f));
          camSagittalInteractor.setRotationAxis(new SbVec3f(0, 0, 1));
          camSagittalInteractor.rotate((float) Math.PI / 2.0f);

          // Scene graph
          m_rootTR.addChild(m_coronalBorderSep);
          m_rootTR.addChild(m_transverseBorderSep);
          createSceneGraph(m_commonGroup, m_rootTR, renderArea, 0, m_sagittalBorderSep);
          m_rootTR.viewAll(renderArea.getSceneManager().getViewportRegion());
        }

        // CORONAL View (bottom left viewer)
        if ( renderArea.getId().equals("renderArea2D_BOTTOM_LEFT") )
        {
          // Instantiate a sceneExaminer to interact with the camera
          m_rootBL = new SceneExaminer();
          m_rootBL.enableOrbit(false);
          m_rootBL.setInteractionMode(SceneExaminer.InteractionMode.SELECTION);
          m_rootBL.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);
          m_rootBL.getCameraInteractor().setOrientation(new SbRotation(new SbVec3f(1, 0, 0), (float) Math.PI / 2.0f));

          // Scene graph
          m_rootBL.addChild(m_sagittalBorderSep);
          m_rootBL.addChild(m_transverseBorderSep);
          createSceneGraph(m_commonGroup, m_rootBL, renderArea, 1, m_coronalBorderSep);
          m_rootBL.viewAll(renderArea.getSceneManager().getViewportRegion());
        }

        // TRANSVERSE View (bottom right viewer)
        if ( renderArea.getId().equals("renderArea2D_BOTTOM_RIGHT") )
        { //
          // Instantiate a sceneExaminer to interact with the camera
          m_rootBR = new SceneExaminer();
          m_rootBR.enableOrbit(false);
          m_rootBR.setInteractionMode(SceneExaminer.InteractionMode.SELECTION);
          m_rootBR.setCameraMode(SceneInteractor.CameraMode.ORTHOGRAPHIC);

          // Scene graph
          m_rootBR.addChild(m_sagittalBorderSep);
          m_rootBR.addChild(m_coronalBorderSep);
          createSceneGraph(m_commonGroup, m_rootBR, renderArea, 2, m_transverseBorderSep);
          m_rootBR.viewAll(renderArea.getSceneManager().getViewportRegion());
        }
      }
    }
    else
    {
      SoError.post("Cannot open " + m_ivFilePath);
    }
  }
}
