package meshviz.mesh.advanced.meshViewer;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbPlane;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.draggers.SoTrackballDragger;
import com.openinventor.inventor.draggers.SoTranslate1Dragger;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.nodes.SoClipPlane;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.nodes.SoTransform;
import com.openinventor.inventor.nodes.SoTranslation;
import com.openinventor.meshviz.data.PoMesh;
import com.openinventor.meshviz.data.PoMeshCrossContour;
import com.openinventor.meshviz.data.PoMeshCrossSection;
import com.openinventor.meshviz.graph.PoBase;
import com.openinventor.meshviz.misc.callbacks.PoRebuildCB;

public class MeshCutPlaneNode extends Scalar3DRepresentNode {
  PoMeshCrossSection m_meshCrossSection;
  PoMeshCrossContour m_meshCrossContour;
  SoSwitch m_meshCrossSectionSwitch;
  SoSwitch m_meshCrossContourSwitch;
  SoClipPlane m_clipPlane;
  SoSwitch m_meshCutPlaneSwitch;

  SoTransform m_transDraggerTransform;
  SoTranslation m_rotateDraggerTrans;
  SoTranslate1Dragger m_cutPlaneTranslateDragger;
  SoTrackballDragger m_cutPlaneRotateDragger;
  SoSwitch m_cutPlaneRotateDraggerSwitch;

  public SbVec3f m_cutPlanePos;
  public int m_whichColoringType;

  private SoNode[] m_inventorNodes;
  private DataNode m_parent;
  private String m_name;
  private int m_number;
  private MeshCutPlanePanel m_panel;
  private float m_transparencyValue;
  private SbVec3f m_planeNormal;
  private float[] m_bboxBounds;

  public MeshCutPlaneNode(SbBox3f bbox) {
    m_number = m_cutPlanesVector.size()+1;
    m_name = "Cut Plane " + m_number;
    m_transparencyValue = 0.f;
    m_whichColoringType = PoMesh.ColoringTypes.COLOR_MAPPING.getValue();

    // Cut Plane position
    SbVec3f bbox_center = bbox.getCenter();
    m_bboxBounds = bbox.getBounds();
    m_cutPlanePos = new SbVec3f();
    if (m_number == 1)
      m_cutPlanePos = bbox_center;
    else {
      float cut_plane_space = (m_bboxBounds[5] - m_bboxBounds[2]) / 10.f;
      SbVec3f previous_cut_plane_pos
          = ((MeshCutPlaneNode)m_cutPlanesVector.lastElement()).m_cutPlanePos;
      m_cutPlanePos.setX(bbox_center.getX());
      m_cutPlanePos.setY(bbox_center.getY());
      float cut_plane_pos_Z = previous_cut_plane_pos.getZ() + cut_plane_space;
      if (cut_plane_pos_Z > m_bboxBounds[5])
        m_cutPlanePos.setZ(m_bboxBounds[2] + cut_plane_space);
      else
        m_cutPlanePos.setZ(cut_plane_pos_Z);
    }

    // Mesh CrossSection
    m_planeNormal = new SbVec3f(0, 0, 1);
    SbPlane cross_section_plane = new SbPlane(m_planeNormal, m_cutPlanePos);
    m_meshCrossSection = new PoMeshCrossSection();
    m_meshCrossSection.plane.setValue(cross_section_plane);
    m_meshCrossSection.addPostRebuildCallback(new TransparencyCB(), null);

    m_meshCrossSectionSwitch = new SoSwitch();
    m_meshCrossSectionSwitch.addChild(m_meshCrossSection);
    m_meshCrossSectionSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    // Mesh CrossContour
    m_meshCrossContour = new PoMeshCrossContour();
    m_meshCrossContour.set("appearance.drawStyle", "lineWidth 2");
    m_meshCrossContour.plane.setValue(cross_section_plane);

    m_meshCrossContourSwitch = new SoSwitch();
    m_meshCrossContourSwitch.addChild(m_meshCrossContour);
    m_meshCrossContourSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    SoSeparator cut_plane_sep = new SoSeparator();
    {
      cut_plane_sep.addChild(m_pickSwitch);
      cut_plane_sep.addChild(m_meshCrossSectionSwitch);
      cut_plane_sep.addChild(m_meshCrossContourSwitch);
    }

    /********* Draggers *********/
    // SoTranslate1Dragger
    SoPath path = new SoPath(cut_plane_sep);

    m_cutPlaneTranslateDragger = new SoTranslate1Dragger();
    m_cutPlaneTranslateDragger.setPartAsPath("translator", null);
    m_cutPlaneTranslateDragger.setPartAsPath("translatorActive", path);
    m_cutPlaneTranslateDragger.translation.setValue(m_cutPlanePos.getZ(),
                                                    m_cutPlanePos.getY(),
                                                    -m_cutPlanePos.getX());

    // Positioning the dragger within the scene :
    // the default translate dragger is along X axis, we need to rotate it
    m_transDraggerTransform = new SoTransform();
    m_transDraggerTransform.rotation.setValue(new SbRotation(new SbVec3f(1, 0, 0),
        m_planeNormal));

    SoSeparator trans_dragger_sep = new SoSeparator();
    {
      trans_dragger_sep.addChild(m_transDraggerTransform);
      trans_dragger_sep.addChild(m_cutPlaneTranslateDragger);
    }
    // SoTrackballDragger
    float[] bbox_size = bbox.getSize().array;
    float max = bbox_size[0];
    if (bbox_size[1] > max)
      max = bbox_size[1];
    if (bbox_size[2] > max)
      max = bbox_size[2];

    float scale_factor = max / 10;
    m_cutPlaneRotateDragger = new SoTrackballDragger();
    m_cutPlaneRotateDragger.scaleFactor.setValue(scale_factor, scale_factor,
                                                 scale_factor);

    m_cutPlaneRotateDraggerSwitch = new SoSwitch();
    m_cutPlaneRotateDraggerSwitch.addChild(m_cutPlaneRotateDragger);
    m_cutPlaneRotateDraggerSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);

    m_rotateDraggerTrans = new SoTranslation();
    m_rotateDraggerTrans.translation.setValue(m_cutPlanePos);

    SoSeparator rotate_dragger_sep = new SoSeparator();
    {
      rotate_dragger_sep.addChild(m_rotateDraggerTrans);
      rotate_dragger_sep.addChild(m_cutPlaneRotateDraggerSwitch);
    }

    // Clip plane
    m_clipPlane = new SoClipPlane();
    m_clipPlane.on.setValue(false);
    SbVec3f clip_normal = m_planeNormal;
    clip_normal.negate();
    m_clipPlane.plane.setValue(new SbPlane(clip_normal, m_cutPlanePos));

    m_meshCutPlaneSwitch = new SoSwitch();
    {
      m_meshCutPlaneSwitch.addChild(trans_dragger_sep);
      m_meshCutPlaneSwitch.addChild(rotate_dragger_sep);
      m_meshCutPlaneSwitch.addChild(cut_plane_sep);
      m_meshCutPlaneSwitch.addChild(m_clipPlane);

      m_meshCutPlaneSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    }

    m_panel = new MeshCutPlanePanel(this);
    m_cutPlaneRotateDragger.addValueChangedCallback(new CutPlaneDraggerCB(),
                                                    null);
    m_cutPlaneTranslateDragger.addValueChangedCallback(new CutPlaneDraggerCB(),
        null);

    m_inventorNodes = new SoNode[] {
        m_meshCrossSection, m_meshCrossContour, m_cutPlaneRotateDragger
    };
  }

  public void decreaseNumber() {
    m_number--;
    m_name = "Cut Plane " + m_number;
    m_panel.changeTitleName(m_name);
  }

  public String toString() {
    return m_name;
  }

  public RepresentNodePanel getPanel() {
    return m_panel;
  }

  public SoNode[] addToParent(DataNode parent, boolean is_drop_action) {
    m_parent = parent;
    if (m_parent instanceof Mesh3DNode)
      ( (Mesh3DNode) parent).addCutPlane(this);
    else
      ( (Scalar3DNode) m_parent).addCutPlane(this);

    if (! is_drop_action) {
      m_cutPlanesVector.add(this);
    }

    return m_inventorNodes;
  }

  public SoNode[] delete(boolean is_drop_action) {
    if (m_parent instanceof Mesh3DNode)
      ((Mesh3DNode)m_parent).removeCutPlane(this);
    else
      ((Scalar3DNode)m_parent).removeCutPlane(this);

    if (! is_drop_action) {
      for (int i = m_number; i < m_cutPlanesVector.size(); i++)
        ((MeshCutPlaneNode)m_cutPlanesVector.get(i)).decreaseNumber();
      m_cutPlanesVector.remove(m_number-1);
    }

    return m_inventorNodes;
  }

  public void show(boolean show) {
    if (show)
      m_meshCutPlaneSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    else
      m_meshCutPlaneSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
  }

  public void setTransparency(float value) {
    m_transparencyValue = value;
    SoMaterial material =
        (SoMaterial)m_meshCrossSection.getPart("appearance.material", true);
    for (int i = 0; i < material.diffuseColor.getNum(); i++)
      material.transparency.set1Value(i, value);
  }

  public DataNode getDataNodeParent() {
    return m_parent;
  }

  public void updateDraggers(int selected_orient) {
    SbVec3f dragger_normal = new SbVec3f();
    SbVec3f dragger_trans = new SbVec3f();
    switch (selected_orient) {
      case 0:
        dragger_normal.setValue(1, 0, 0);
        dragger_trans.setValue(m_cutPlanePos);
        break;
      case 1:
        dragger_normal.setValue(0, 1, 0);
        dragger_trans.setValue(m_cutPlanePos.getY(),
                               m_cutPlanePos.getX(),
                               m_cutPlanePos.getZ());
        break;
      case 2:
        dragger_normal.setValue(0, 0, 1);
        dragger_trans.setValue(m_cutPlanePos.getZ(),
                               m_cutPlanePos.getY(),
                               -m_cutPlanePos.getX());
        break;
    }

    m_cutPlaneRotateDragger.rotation.setValue(new SbRotation(new SbVec3f(0, 0, 1),
        dragger_normal));
    m_transDraggerTransform.rotation.setValue(new SbRotation(new SbVec3f(1, 0, 0),
        dragger_normal));
    m_cutPlaneTranslateDragger.translation.setValue(dragger_trans);
  }

  class TransparencyCB extends PoRebuildCB {
    public void invoke(PoBase base) {
      setTransparency(m_transparencyValue);
    }
  }

  class CutPlaneDraggerCB extends SoDraggerCB {
    public void invoke(SoDragger d) {
      // rotate the plane's normal by the dragger rotation
      SbRotation rotation = m_cutPlaneRotateDragger.rotation.getValue();
      m_planeNormal = rotation.multVec(new SbVec3f(0, 0, 1));

      if (d instanceof SoTranslate1Dragger) {
        SoTranslate1Dragger dragger = ( (SoTranslate1Dragger) d);

        SbVec3f dragger_pos = dragger.translation.getValue();
        float new_pos = dragger_pos.getX();
        SbVec3f dragger_trans = null;

        int selected_orient = m_panel.m_cutPlaneOrient.getSelectedIndex();
        switch (selected_orient) {
          case 0:
            if (new_pos <= m_bboxBounds[3] && new_pos >= m_bboxBounds[0])
              m_cutPlanePos.setX(dragger_pos.getX());
            else
               dragger_trans = new SbVec3f(m_cutPlanePos);
            break;
          case 1:
            if (new_pos <= m_bboxBounds[4] && new_pos >= m_bboxBounds[1])
              m_cutPlanePos.setY(dragger_pos.getX());
            else
              dragger_trans = new SbVec3f(m_cutPlanePos.getY(),
                                          m_cutPlanePos.getX(),
                                          m_cutPlanePos.getZ());
            break;
          case 2:
            if (new_pos <= m_bboxBounds[5] && new_pos >= m_bboxBounds[2])
              m_cutPlanePos.setZ(dragger_pos.getX());
            else
              dragger_trans = new SbVec3f(m_cutPlanePos.getZ(),
                                          m_cutPlanePos.getY(),
                                          -m_cutPlanePos.getX());
            break;
        }

        if (dragger_trans != null) {
          dragger.enableValueChangedCallbacks(false);
          dragger.translation.setValue(dragger_trans);
          dragger.enableValueChangedCallbacks(true);
        }

        // translate TrackballDragger
        m_cutPlaneRotateDragger.enableValueChangedCallbacks(false);
        m_rotateDraggerTrans.translation.setValue(m_cutPlanePos);
        m_cutPlaneRotateDragger.enableValueChangedCallbacks(true);
      }

      m_meshCrossSection.plane.setValue(new SbPlane(m_planeNormal, m_cutPlanePos));
      m_meshCrossContour.plane.setValue(new SbPlane(m_planeNormal, m_cutPlanePos));

      // translate clip plane
      SbVec3f clip_normal = m_planeNormal;
      if (!m_panel.m_frontSide.isSelected())
        clip_normal.negate();
      m_clipPlane.plane.setValue(new SbPlane(clip_normal, m_cutPlanePos));
    }
  }
}
