package meshviz.mesh.advanced.meshViewer;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;

import javax.swing.JOptionPane;

import com.openinventor.hardcopy.SoVectorizeAction;
import com.openinventor.hardcopy.SoVectorizeCGMAction;
import com.openinventor.hardcopy.SoVectorizeHPGLAction;
import com.openinventor.hardcopy.SoVectorizePSAction;
import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPath;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoSearchAction;
import com.openinventor.inventor.actions.SoWriteAction;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.meshviz.data.PbMesh;
import com.openinventor.meshviz.graph.PoNonLinearValueLegend3;
import com.openinventor.meshviz.nodes.PoDomain;
import com.openinventor.meshviz.nodes.PoIsovaluesList;
import com.openinventor.meshviz.nodes.PoMeshProperty;
import com.openinventor.meshviz.nodes.PoMiscTextAttr;
import com.openinventor.meshviz.nodes.PoNonLinearDataMapping2;

import util.ViewerComponentsFactory;

public class MeshViewer {
  public static String m_filePrefix;
  private int m_numIsoValues;
  private String m_courtesyTitle;
  private MeshViewTreeFrame m_treeFrame;

  IViewerExaminer m_viewer;
  SoSeparator m_sceneRoot;
  SoSeparator m_scene3D;
  SoAnnotation m_scene2D;
  SoSwitch m_probeTextInfoSwitch;
  SoSwitch m_dataMappingSwitch;
  SoSwitch m_isoValuesSwitch;
  SoSwitch m_dataSwitch;
  SoSwitch m_dataLegendSwitch;
  PoNonLinearDataMapping2 m_moduleDataMapping;
  PoMeshProperty m_meshNode;
  PoDomain m_domain;
  SbBox3f m_bbox;
  PbMesh m_mesh;
  SoEventCallback m_mouseEventCB;
  SoGradientBackground m_background;
  int m_initDatasetIndex;

  protected SbColor[] m_colors = {
      new SbColor(0, 0, 1),
      new SbColor(0, 1, 1),
      new SbColor(0, 1, 0),
      new SbColor(1, 1, 0),
      new SbColor(1, 0, 0)
  };

  public MeshViewer() {
    m_numIsoValues = 16;
    m_courtesyTitle = " ";
    m_domain = null;
    m_bbox = null;
    m_initDatasetIndex = 0;
    m_viewer = null;

    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    m_filePrefix =
        SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;
  }

  public IViewerExaminer show(String iv_filename) {
    if (loadIvFile(iv_filename)) {
      m_treeFrame = new MeshViewTreeFrame(this);
      m_treeFrame.setVisible(true);
      return m_viewer;
    }
    return null;
  }

  public IViewerExaminer show(PoMeshProperty mesh_node) {
    m_meshNode = mesh_node;
    m_mesh = m_meshNode.getMesh();
    if (m_bbox == null)
      m_bbox = m_mesh.getBoundingBox();
    buildSceneGraph();

    m_treeFrame = new MeshViewTreeFrame(this);
    m_treeFrame.setVisible(true);
    return m_viewer;
  }

  public IViewerExaminer show(PoMeshProperty mesh_node, PoDomain domain) {
    m_domain = domain;
    m_bbox = new SbBox3f();
    m_bbox.setBounds(domain.min.getValue(), domain.max.getValue());
    return show(mesh_node);
  }

  public void stop() {
    m_treeFrame.dispose();
    if ( m_viewer != null )
      m_viewer.dispose();
  }

  public void setCourtesyTitle(String title) {
    m_courtesyTitle = title;
  }

  public void setDatasetIndex(int index) {
    m_initDatasetIndex = index;
  }

  public void saveIvFile(String iv_filename, boolean binary) {
    // Save the iv file in "FOLD" format,
    // Attention : This saved file cannot be read by an application that
    // is not linked with MeshViz (used UNFOLD for that!)
    // PoBase.setNodeWriteFormat(PoBase.FOLD_NODE_WRITE_FORMAT);
    SoWriteAction write_action = new SoWriteAction();
    write_action.getOutput().openFile(iv_filename);
    write_action.getOutput().setBinary(binary);
    write_action.apply(m_sceneRoot);
    write_action.getOutput().closeFile();
  }

  public void vectorizeCGM(String filename) {
    SoVectorizeCGMAction action = new SoVectorizeCGMAction();
    action.getCGMVectorOutput().openFile(filename);
    action.setHLHSRMode(SoVectorizeAction.HLHSRModes.HLHSR_RASTER);
    action.apply(m_sceneRoot);
    action.getCGMVectorOutput().closeFile();
  }
  public void vectorizeHPGL(String filename) {
    SoVectorizeHPGLAction action = new SoVectorizeHPGLAction();
    action.getHPGLVectorOutput().openFile(filename);
    action.setHLHSRMode(SoVectorizeAction.HLHSRModes.HLHSR_RASTER);
    action.apply(m_sceneRoot);
    action.getHPGLVectorOutput().closeFile();
  }
  public void vectorizePS(String filename) {
    SoVectorizePSAction action = new SoVectorizePSAction();
    action.getPSVectorOutput().openFile(filename);
    action.setHLHSRMode(SoVectorizeAction.HLHSRModes.HLHSR_RASTER);
    action.apply(m_sceneRoot);
    action.getPSVectorOutput().closeFile();
  }

  public boolean loadIvFile(String iv_filename) {
    Container parent = null;

    if (getMeshFromIvFile(iv_filename)) {
      if (m_viewer != null) {
        parent = m_viewer.getComponent().getParent();
        parent.removeAll();
      }

      buildSceneGraph();

      if ( parent != null )
      {
        parent.add(m_viewer.getComponent(), BorderLayout.CENTER);
        parent.validate();
        parent.repaint();
      }
      return true;
    }
    return false;
  }

  public static URL getURL(String fileName)
  {
    URL url = null;
    try
    {
      url = new URL("file:" + (new File(fileName)).getAbsolutePath());
    }
    catch (MalformedURLException exc)
    {
      System.err.println(exc + " : (" + url + ") " + fileName + " invalid URL");
      url = null;
    }
    catch (Exception e)
    {
      System.err.println(e + " : cant load File " + fileName);
      url = null;
    }
    return url;
  }

  private boolean getMeshFromIvFile(String iv_file_name) {
    SoInput input = new SoInput();

    if (! input.openFile(iv_file_name)) {
      JOptionPane.showMessageDialog(null, "File not found",
                                    "Error",
                                    JOptionPane.ERROR_MESSAGE);
      return false;
    }

    SoGroup group = SoDB.readAll(input);
    if (group == null) {
      JOptionPane.showMessageDialog(null, "Invalid file format",
                                    "Error",
                                    JOptionPane.ERROR_MESSAGE);
      return false;
    }

    SoSearchAction s_action = new SoSearchAction();
    s_action.setNodeClass(PoMeshProperty.class, true);
    s_action.apply(group);

    SoPath path = s_action.getPath();
    if (path == null) {
      JOptionPane.showMessageDialog(null,
                                    "This iv file do not contain any mesh definition",
                                    "Incorrect iv file",
                                    JOptionPane.ERROR_MESSAGE);
      return false;
    }

    m_meshNode = (PoMeshProperty)path.regular.getTail();
    m_mesh = m_meshNode.getMesh();
    m_bbox = m_mesh.getBoundingBox();
    return true;
  }

  private void buildSceneGraph() {
    // Add an event callback to catch mouse button presses.
    m_mouseEventCB = new SoEventCallback();

    // background
    m_background = new SoGradientBackground();

    { // 3D scene
      m_scene3D = new SoSeparator();
      m_scene3D.setName("Scene3D");
      m_scene3D.addChild(m_background);
      m_scene3D.addChild(m_mouseEventCB);
      m_scene3D.addChild(m_meshNode);
    }

    buildDataMapping();

    // Camera for the 2D scene
    SoOrthographicCamera camera2D = new SoOrthographicCamera();
    camera2D.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);

    buildScene2D();

    m_sceneRoot = new SoSeparator();
    { // scene graph
      m_sceneRoot.addChild(m_scene3D);
      m_sceneRoot.addChild(camera2D);
      m_sceneRoot.addChild(m_dataSwitch);
      m_sceneRoot.addChild(m_scene2D);
    }

    // viewer
    if ( m_viewer == null )
    {
      m_viewer = ViewerComponentsFactory.createViewerExaminer();
    }
    m_viewer.setSceneGraph(m_sceneRoot);
    m_viewer.viewAll();
    int screen_width = Toolkit.getDefaultToolkit().getScreenSize().width;
    m_viewer.getComponent().setPreferredSize(new Dimension(screen_width/2, 500));
  }

  private void buildDataMapping() {
    // Data mapping
    int dataset_num = m_mesh.getNumValuesSet();
    float[] vmin = new float[dataset_num];
    float[] vmax = new float[dataset_num];
    m_dataMappingSwitch = new SoSwitch();
    m_isoValuesSwitch = new SoSwitch();
    m_dataSwitch = new SoSwitch();
    m_dataSwitch.whichChild.setValue(0);

    for (int i = 0; i < dataset_num; i++) {
      PoNonLinearDataMapping2 data_mapping = new PoNonLinearDataMapping2();
      vmin[i] = m_mesh.getMinValuesSet(i);
      vmax[i] = m_mesh.getMaxValuesSet(i);
      data_mapping.type.setValue(PoNonLinearDataMapping2.Types.LINEAR_PER_LEVEL);
      float dv = (vmax[i] - vmin[i]) / 4;
      float val = vmin[i];
      for (int j = 0; j < 5; j++, val += dv) {
        data_mapping.value.set1Value(j, val);
        data_mapping.color.set1Value(j, m_colors[j]);
      }

      // define the list of iso-value associated to i-th data-set
      PoIsovaluesList iso_values = new PoIsovaluesList();
      iso_values.setRegularIsoList(vmin[i], vmax[i], m_numIsoValues);

      m_dataMappingSwitch.addChild(data_mapping);
      m_isoValuesSwitch.addChild(iso_values);

      SoGroup group = new SoGroup();
      group.addChild(data_mapping);
      group.addChild(iso_values);

      m_dataSwitch.addChild(group);
    }

    // define the data-mapping associated to the module of vectors
    m_moduleDataMapping = new PoNonLinearDataMapping2();
    m_moduleDataMapping.type.setValue(PoNonLinearDataMapping2.Types.LINEAR_PER_LEVEL);
    if (m_mesh.getNumVecsSet() > 0) {
      float min = m_mesh.getMinVecsSet(0);
      float max = m_mesh.getMaxVecsSet(0);
      float dv = (max - min) / 4;
      float val = min;
      for (int j = 0; j < 5; j++, val += dv) {
        m_moduleDataMapping.value.set1Value(j, val);
        m_moduleDataMapping.color.set1Value(j, m_colors[j]);
      }
    }
  }

  private void buildScene2D() {
    // build courtesy title
    SoFont font = new SoFont();
    font.size.setValue(12);
    SoTranslation courtesy_trans = new SoTranslation();
    courtesy_trans.translation.setValue(0.95F, -0.95F, -1);
    SoText2 courtesy_title = new SoText2();
    courtesy_title.string.setValue(m_courtesyTitle);
    courtesy_title.justification.setValue(SoText2.Justifications.RIGHT);

    // build colored legend
    // set the background legend box invisible.
    PoNonLinearValueLegend3 data_legend
        = new PoNonLinearValueLegend3(new SbVec2f( -0.95F, -0.95F),
                                      new SbVec2f( -0.75F, 0.95F));
    data_legend.set("backgroundApp.drawStyle", "style INVISIBLE");
    data_legend.set("backgroundBorderApp.drawStyle", "style INVISIBLE");

    // define the font used by the legend
    PoMiscTextAttr legend_text_attr = new PoMiscTextAttr();
    legend_text_attr.fontName.setValue("Courier New");

    SoSeparator legend_sep = new SoSeparator();
    legend_sep.addChild(legend_text_attr);
    legend_sep.addChild(data_legend);

    m_dataLegendSwitch = new SoSwitch();
    m_dataLegendSwitch.addChild(legend_sep) ;
    m_dataLegendSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    // pick style node
    SoPickStyle pick_style = new SoPickStyle();
    pick_style.style.setValue(SoPickStyle.Styles.UNPICKABLE);

    // Text property for the 2D scene
    SoAnnoText3Property annoText3Property = new SoAnnoText3Property();
    annoText3Property.renderPrintType.setValue(SoAnnoText3Property.RenderPrintTypes.
                                               RENDER2D_PRINT_RASTER);
    annoText3Property.fontSizeHint.setValue(SoAnnoText3Property.FontSizeHints.ANNOTATION);

    m_probeTextInfoSwitch = new SoSwitch();
    m_probeTextInfoSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);

    m_scene2D = new SoAnnotation();
    m_scene2D.addChild(pick_style);
    m_scene2D.addChild(annoText3Property);
    m_scene2D.addChild(m_dataLegendSwitch);
    m_scene2D.addChild(font);
    m_scene2D.addChild(courtesy_trans);
    m_scene2D.addChild(courtesy_title);
    m_scene2D.addChild(m_probeTextInfoSwitch);
  }
}
