package meshviz.mesh.sample.dialog;

import java.awt.*;

/**
  *  PoAwtLabelData
  *
  * Class to define a label. This element is only composed of a string.
  * By default the text of the label is an empty string.
  *
  * @author Rodolphe Albout
  * @author Laurent ISTIN
  * @author Loic Vigneras
  * @author Patrick Vigneras
  */
public class PoAwtLabelData extends PoAwtElementData {
  static Color UIForegroundColor = Color.blue;

  // Constants
  // ---------

  /** Constant for alignment to the left or the beginning.
    * You can also use Label.LEFT.
    */
  public static final int ALIGNMENT_BEGINNING = Label.LEFT;
  /** Constant for alignment to the center.
    * You can also use Label.CENTER.
    */
  public static final int ALIGNMENT_CENTER = Label.CENTER;
  /** Constant for alignment to the right or the end.
    * You can also use Label.RIGHT.
    */
  public static final int ALIGNMENT_END = Label.RIGHT;

  // Constructors
  // ------------

  /** Construct a new centered PoAwtLabelData and without label text
    * (set to 'null').
    */
  public PoAwtLabelData () {
    this (null, ALIGNMENT_CENTER);
  }

  /** Construct a new centered PoAwtLabelData with the given label text.
    */
  public PoAwtLabelData (String text) {
    this (text, ALIGNMENT_CENTER);
  }

  /** Construct a new PoAwtLabelData with the given label text and alignement.
    * @param text The text to display in the label
    * @parem alignement Any alignment constant (see above)
    */
  public PoAwtLabelData (String text, int alignment) {
    super ();
    _label = new Label (text, alignment);
    _label.setForeground(UIForegroundColor);
    linkInnerComponents ();
    setLabel (text);  // set also _labelWidth and _labelWidthPixel
  }

  /** Construct a new PoAwtLabelData with the same values as the given one.
    */
  protected PoAwtLabelData (PoAwtLabelData label) {
    super ();
    _label = new Label (label.getLabel (), label.getAlignment ());
    linkInnerComponents ();
    setLabel (getLabel ());  // set also _labelWidth and _labelWidthPixel
  }

  // Methods
  // -------

  /** Set the text of the label.
    */
  public void setLabel (String text) {
    _label.setText (text);
    _labelWidth = text.length ();
    setLabelWidthPixel ();
  }

  /** Get the text of the label.
    */
  public String getLabel () {
    return _label.getText ();
  }

  /** Returns the type of the element, i.e. PoAwtElementData.LABEL.
    */
  public int getType () {
    return LABEL;
  }

  /** Creates and returns a new element which is a copy of this element.
    */
  protected Object clone () {
    return new PoAwtLabelData (this);
  }

  /** Affect the internal data awtElement to current object.
    * The current object and awtElement have to be the same object type.
    * This method should be called by PoAwtDialog, and not by user code
    * directly.
    */
  public void updateValues (PoAwtElementData element) {
    if (element.getType () == LABEL) {
      PoAwtLabelData label = (PoAwtLabelData) element;

      setLabel (label.getLabel ());
      setLength (label.getLength ());
      setID (label.getID ());
    }
  }

  // For change the size of label form
  public void setLength (int length) {
    _labelWidthPixel = length;
  }
  public int getLength () {
    return _labelWidthPixel;
  }
  public int getTextLength () {
    return _labelWidth;
  }
  public int getAlignment () {
    return _label.getAlignment ();
  }

  public void addNotify () {
    super.addNotify ();
    setLabelWidthPixel ();
  }

  private void linkInnerComponents () {
    GridBagLayout gridbag = new GridBagLayout ();
    this.setLayout (gridbag);
    GridBagConstraints c = new GridBagConstraints ();
    c.fill = GridBagConstraints.BOTH;
    c.weightx = 1.0;
    c.weighty = 1.0;
    gridbag.setConstraints (_label, c);
    this.add (_label);
  }

  private void setLabelWidthPixel () {
    Font font = this.getFont ();
    if (font == null)
      return;
    FontMetrics fm = this.getFontMetrics (font);
    if (fm == null)
      return;
    _labelWidthPixel = fm.stringWidth (getLabel ());
  }

  // Fields
  // ------

  // Inner component
  private Label _label;  // includes text of label and alignment

  private int _labelWidth;
  private int _labelWidthPixel;
}
