package meshvizxlm.eclipsemeshviz.pillargrid;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;

import meshvizxlm.eclipsemeshviz.DemoSettings;

public class PillarGridBinReader extends PillarGridFileIO
{

  @Override
  public void write(PillarGrid pillarGrid, File file) throws IOException
  {
    System.out.println("saving pillar grid into " + file.getName());
    long startTime = System.currentTimeMillis();

    long filePosition = 0;

    RandomAccessFile fileStream = new RandomAccessFile(file, "rw");
    FileChannel fileChannel = fileStream.getChannel();
    MappedByteBuffer dimensionMappedByteBuffer = fileChannel.map(MapMode.READ_WRITE, filePosition, Long.SIZE / 8 * 7);
    dimensionMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);

    // writes grid dimensions
    dimensionMappedByteBuffer.putLong(pillarGrid.getDimI());
    dimensionMappedByteBuffer.putLong(pillarGrid.getDimJ());
    dimensionMappedByteBuffer.putLong(pillarGrid.getDimK());
    dimensionMappedByteBuffer.putLong(pillarGrid.m_numHFacesAtKLevel);

    dimensionMappedByteBuffer.putLong(4);
    dimensionMappedByteBuffer.putLong(1);
    // writes num zcorn
    dimensionMappedByteBuffer.putLong(pillarGrid.m_numZcorns);
    dimensionMappedByteBuffer.force();
    filePosition += dimensionMappedByteBuffer.position();

    // writes data
    int i;
    for ( int j = 0; j < 4; j++ )
    {
      MappedByteBuffer zcornMappedByteBuffer =
          fileChannel.map(MapMode.READ_WRITE, filePosition, pillarGrid.m_numZcorns * Float.SIZE / 8 / 4);
      zcornMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
      for ( i = 0; i < pillarGrid.m_numZcorns / 4; i++ )
      {
        zcornMappedByteBuffer.putFloat(pillarGrid.m_zcorn[(int) (i + j * pillarGrid.m_numZcorns / 4)]);
      }
      zcornMappedByteBuffer.force();
      filePosition += zcornMappedByteBuffer.position();
    }

    MappedByteBuffer coordMappedByteBuffer =
        fileChannel.map(MapMode.READ_WRITE, filePosition, pillarGrid.m_coord.length * Float.SIZE / 8);
    coordMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
    for ( i = 0; i < pillarGrid.m_coord.length; i++ )
      coordMappedByteBuffer.putFloat(pillarGrid.m_coord[i]);
    filePosition += coordMappedByteBuffer.position();
    coordMappedByteBuffer.force();

    MappedByteBuffer longMappedByteBuffer = fileChannel.map(MapMode.READ_WRITE, filePosition, Long.SIZE / 8);
    longMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
    longMappedByteBuffer.putLong(pillarGrid.m_numNonActiveCells);
    filePosition += longMappedByteBuffer.position();
    longMappedByteBuffer.force();

    MappedByteBuffer actnumMappedByteBuffer =
        fileChannel.map(MapMode.READ_WRITE, filePosition, pillarGrid.m_dimI * pillarGrid.m_dimJ * pillarGrid.m_dimK
            * Byte.SIZE / 8);
    actnumMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
    if ( pillarGrid.hasNonActiveCells() )
      for ( i = 0; i < pillarGrid.m_dimI * pillarGrid.m_dimJ * pillarGrid.m_dimK; i++ )
      {
        byte[] actCell = new byte[] { (byte) (pillarGrid.m_actnum[i] ? 1 : 0) };
        actnumMappedByteBuffer.put(actCell);
      }
    filePosition += actnumMappedByteBuffer.position();
    actnumMappedByteBuffer.force();

    MappedByteBuffer boolMappedByteBuffer = fileChannel.map(MapMode.READ_WRITE, filePosition, Byte.SIZE / 8);
    boolMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
    boolean flag = pillarGrid.isPorosityAvailable();
    byte[] porosityExists = new byte[] { (byte) (flag ? 1 : 0) };
    boolMappedByteBuffer.put(porosityExists);
    boolMappedByteBuffer.force();
    filePosition += boolMappedByteBuffer.position();

    if ( flag )
    {
      MappedByteBuffer poroMappedByteBuffer =
          fileChannel.map(MapMode.READ_WRITE, filePosition, pillarGrid.m_dimI * pillarGrid.m_dimJ * pillarGrid.m_dimK
              * Float.SIZE / 8);
      poroMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
      for ( i = 0; i < pillarGrid.m_poro.length; i++ )
        poroMappedByteBuffer.putFloat(pillarGrid.m_poro[i]);
      poroMappedByteBuffer.force();
      filePosition += poroMappedByteBuffer.position();
    }

    fileChannel.close();
    fileStream.close();
    DemoSettings.displayTime("pillar grid saved", System.currentTimeMillis() - startTime);

  }

  @Override
  public void read(PillarGrid pillarGrid, File file) throws Exception
  {
    System.out.println("reading pillar grid from " + file.getName());
    RandomAccessFile inputFile = new RandomAccessFile(file, "r");
    FileChannel fileChannel = inputFile.getChannel();
    try
    {
      long startTime = System.currentTimeMillis();
      long filePosition = 0;
      MappedByteBuffer dataMappedByteBuffer =
          fileChannel.map(FileChannel.MapMode.READ_ONLY, filePosition, 8 * 7).load();
      dataMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
      long dimI = dataMappedByteBuffer.getLong();
      long dimJ = dataMappedByteBuffer.getLong();
      long dimK = dataMappedByteBuffer.getLong();
      pillarGrid.m_numHFacesAtKLevel = (int) dataMappedByteBuffer.getLong();
      pillarGrid.setDim((int) dimI, (int) dimJ, (int) dimK);

      long CoorTSize = dataMappedByteBuffer.getLong();
      long ActiveTSize = dataMappedByteBuffer.getLong();

      long storedNumZCorns = dataMappedByteBuffer.getLong();
      filePosition += dataMappedByteBuffer.position();
      if ( pillarGrid.m_numZcorns == storedNumZCorns
          && (!DemoSettings.HORIZONTAL_FAULTS || pillarGrid.m_numHFacesAtKLevel == 2) )
      {
        // read data
        int i;
        for ( int j = 0; j < 4; j++ )
        {
          MappedByteBuffer zcornMappedByteBuffer =
              fileChannel.map(MapMode.READ_ONLY, filePosition, pillarGrid.m_numZcorns * 1);
          zcornMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
          zcornMappedByteBuffer.load();
          for ( i = 0; i < pillarGrid.m_numZcorns / 4; i++ )
          {
            pillarGrid.m_zcorn[(int) (i + j * pillarGrid.m_numZcorns / 4)] = zcornMappedByteBuffer.getFloat();
          }
          filePosition += zcornMappedByteBuffer.position();
        }

        MappedByteBuffer coordMappedByteBuffer =
            fileChannel.map(MapMode.READ_ONLY, filePosition, pillarGrid.m_coord.length * 4);
        coordMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
        coordMappedByteBuffer.load();
        for ( i = 0; i < pillarGrid.m_coord.length; i++ )
        {
          pillarGrid.m_coord[i] = coordMappedByteBuffer.getFloat();
        }
        filePosition += coordMappedByteBuffer.position();

        MappedByteBuffer longMappedByteBuffer = fileChannel.map(FileChannel.MapMode.READ_ONLY, filePosition, 8).load();
        longMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);

        pillarGrid.m_numNonActiveCells = longMappedByteBuffer.getLong();
        filePosition += longMappedByteBuffer.position();

        if ( pillarGrid.m_numNonActiveCells > 0 )
        {
          MappedByteBuffer actNumMappedByteBuffer =
              fileChannel.map(MapMode.READ_ONLY, filePosition, pillarGrid.m_dimI * pillarGrid.m_dimJ
                  * pillarGrid.m_dimK * 1);
          actNumMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
          actNumMappedByteBuffer.load();
          byte actCell;
          for ( i = 0; i < pillarGrid.m_dimI * pillarGrid.m_dimJ * pillarGrid.m_dimK; i++ )
          {
            actCell = actNumMappedByteBuffer.get();
            pillarGrid.m_actnum[i] = actCell != 0;
          }
          filePosition += actNumMappedByteBuffer.position();
        }

        MappedByteBuffer boolMappedByteBuffer = fileChannel.map(FileChannel.MapMode.READ_ONLY, filePosition, 1).load();
        boolMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);

        byte flagByte = boolMappedByteBuffer.get();
        filePosition += boolMappedByteBuffer.position();

        boolean flag = flagByte != 0;
        pillarGrid.setAvailablePorosity(flag);
        if ( flag == true )
        {
          MappedByteBuffer poroMappedByteBuffer =
              fileChannel.map(MapMode.READ_ONLY, filePosition, pillarGrid.m_dimI * pillarGrid.m_dimJ
                  * pillarGrid.m_dimK * 4);
          poroMappedByteBuffer.order(ByteOrder.LITTLE_ENDIAN);
          poroMappedByteBuffer.load();
          for ( i = 0; i < pillarGrid.m_dimI * pillarGrid.m_dimJ * pillarGrid.m_dimK; i++ )
          {
            pillarGrid.m_poro[i] = poroMappedByteBuffer.getFloat();
          }
          filePosition += poroMappedByteBuffer.position();
        }
      }
      else
        throw new Exception("wrong file format " + file.getName());

      DemoSettings.displayTime("pillar grid read", System.currentTimeMillis() - startTime);
    }
    catch (Exception ex)
    {
      throw ex;
    }
    finally
    {
      fileChannel.close();
      inputFile.close();
    }
  }
}
