package meshvizxlm.eclipsemeshviz.pillargrid;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import meshvizxlm.eclipsemeshviz.DemoSettings;

public class PillarGridGenerator
{
  private static final double EPSILON = 1e-4;
  private static final int[][] NEIGHBOUR_PILLAR = { { 0, 0 }, { 1, 0 }, { 1, 1 }, { 0, 1 } };
  private static final int[][] NEIGHBOUR_CELLS = { { 0, 0 }, { -1, 0 }, { -1, -1 }, { 0, -1 } };
  private static final String CONST = "const";
  private static final String COS = "cos";
  private static final String SIN = "sin";
  private static final String COSSIN = "cossin";
  private static final String STEPICRACK = "stepicrack";
  private static final String STEPJCRACK = "stepjcrack";
  private static final String LINEARCRACK = "linearcrack";

  private List<LevelFunctor> m_levelFctors;
  private List<CrackFunctor> m_crackFctors;

  public PillarGridGenerator()
  {
    m_levelFctors = new ArrayList<LevelFunctor>();
    m_crackFctors = new ArrayList<CrackFunctor>();
  }

  public <T extends PillarGrid> void generate(T pillarGrid, int numCellI, int numCellJ, int numCellK, float maxX,
      float maxY)
  {
    pillarGrid.clear();

    if ( m_levelFctors.size() >= 2 )
    {
      long t1 = System.currentTimeMillis();

      pillarGrid.m_numHFacesAtKLevel = DemoSettings.HORIZONTAL_FAULTS ? 2 : 1;
      pillarGrid.setDim(numCellI, numCellJ, numCellK);

      generateCoord(pillarGrid, maxX, maxY);
      generateZCornTopDown(pillarGrid, maxX, maxY);

      long t2 = System.currentTimeMillis();
      DemoSettings.displayTime("PillarGrid generated", t2 - t1);
    }
    else
    {
      pillarGrid.clear();
      System.err.println("PillarGridGenerator: Bad number of level functions. Can not generate synthetic mesh");
    }
  }

  public <T extends PillarGrid> void generate(T pillarGrid, String filename, int[] dim)
      throws UnsupportedOperationException
  {
    generate(pillarGrid, filename, dim, 1.0);
  }

  public <T extends PillarGrid> void generate(T pillarGrid, String filename, int[] dim, double ratio)
      throws UnsupportedOperationException
  {
    try
    {
      FileInputStream file = new FileInputStream(filename);
      BufferedReader reader = new BufferedReader(new InputStreamReader(file));
      String line;
      String[] words;
      String firstWord;

      int dimI;
      int dimJ;
      int dimK;
      float maxX;
      float maxY;
      int ni;
      int nj;
      int nk;

      line = reader.readLine();
      if ( line == null || (words = line.split("\\s+")).length < 5 )
      {
        reader.close();
        throw new UnsupportedOperationException("PillarGridGenerator: I/O exception occured in " + filename);
      }
      else
      {
        dimI = Integer.parseInt(words[0]);
        dimJ = Integer.parseInt(words[1]);
        dimK = Integer.parseInt(words[2]);
        maxX = Float.parseFloat(words[3]);
        maxY = Float.parseFloat(words[4]);
        ni = (int) (dimI * ratio);
        nj = (int) (dimJ * ratio);
        nk = (int) (dimK * ratio);
      }

      float crackmax = 0;
      float crackmin = 0;

      while ( (line = reader.readLine()) != null )
      {
        words = line.split("\\s+");

        if ( words.length > 0 )
        {
          firstWord = words[0];

          if ( CONST.equalsIgnoreCase(firstWord) )
          {
            float cst = Float.parseFloat(words[1]);
            m_levelFctors.add(new ConstantFctor(cst));
          }
          else if ( COS.equalsIgnoreCase(firstWord) )
          {
            float a = Float.parseFloat(words[1]);
            float b = Float.parseFloat(words[2]);
            float cst = Float.parseFloat(words[3]);
            m_levelFctors.add(new CosFctor(a, b, cst));
          }
          else if ( SIN.equalsIgnoreCase(firstWord) )
          {
            float a = Float.parseFloat(words[1]);
            float b = Float.parseFloat(words[2]);
            float cst = Float.parseFloat(words[3]);
            m_levelFctors.add(new SinFctor(a, b, cst));
          }
          else if ( COSSIN.equalsIgnoreCase(firstWord) )
          {
            float a1 = Float.parseFloat(words[1]);
            float b1 = Float.parseFloat(words[2]);
            float a2 = Float.parseFloat(words[3]);
            float b2 = Float.parseFloat(words[4]);
            float cst = Float.parseFloat(words[5]);
            m_levelFctors.add(new CosSinFctor(a1, b1, a2, b2, cst));
          }
          else if ( STEPICRACK.equalsIgnoreCase(firstWord) )
          {
            float a = Float.parseFloat(words[1]);
            float cst = Float.parseFloat(words[2]);
            m_crackFctors.add(new StepICrackFctor((int) (ni * a), cst));
            if ( cst > 0 )
              crackmax += cst;
            if ( cst < 0 )
              crackmin += cst;
          }
          else if ( STEPJCRACK.equalsIgnoreCase(firstWord) )
          {
            float a = Float.parseFloat(words[1]);
            float cst = Float.parseFloat(words[2]);
            m_crackFctors.add(new StepJCrackFctor((int) (nj * a), cst));
            if ( cst > 0 )
              crackmax += cst;
            if ( cst < 0 )
              crackmin += cst;
          }
          else if ( LINEARCRACK.equalsIgnoreCase(firstWord) )
          {
            float a = Float.parseFloat(words[1]);
            float b = Float.parseFloat(words[2]);
            float cst = Float.parseFloat(words[3]);
            m_crackFctors.add(new LinearCrackFctor(b, (int) (nj * a), cst));
            if ( cst > 0 )
              crackmax += cst;
            if ( cst < 0 )
              crackmin += cst;
          }
        }
      }
      reader.close();

      pillarGrid.m_min[0] = 0;
      pillarGrid.m_max[0] = maxX;
      pillarGrid.m_min[1] = 0;
      pillarGrid.m_max[1] = maxY;
      pillarGrid.m_min[2] = Float.POSITIVE_INFINITY;
      pillarGrid.m_max[2] = Float.NEGATIVE_INFINITY;
      for ( int level = 0; level < m_levelFctors.size(); ++level )
      {
        pillarGrid.m_max[2] = Math.max(pillarGrid.m_max[2], m_levelFctors.get(level).getMax());
        pillarGrid.m_min[2] = Math.min(pillarGrid.m_min[2], m_levelFctors.get(level).getMin());
      }

      pillarGrid.m_max[2] += crackmax;
      pillarGrid.m_min[2] += crackmin;
      pillarGrid.m_minMaxUpdated = true;

      generate(pillarGrid, ni, nj, nk, maxX, maxY);

      if ( dim != null && dim.length > 2 )
      {
        dim[0] = dimI;
        dim[1] = dimJ;
        dim[2] = dimK;
      }
    }
    catch (FileNotFoundException e)
    {
      throw new UnsupportedOperationException("PillarGridGenerator: file not found: " + filename);
    }
    catch (IOException e)
    {
      throw new UnsupportedOperationException("PillarGridGenerator: I/O exception occured in " + filename);
    }
    // catch (Exception e)
    // {
    // throw new
    // UnsupportedOperationException("PillarGridGenerator: file format not supported.");
    // }
  }

  public void clear()
  {
    m_levelFctors.clear();
    m_crackFctors.clear();
  }

  private int generateCoord(PillarGrid mesh, float maxX, float maxY)
  {
    final int dimI = mesh.getDimI();
    final int dimJ = mesh.getDimJ();
    final int numLevel = m_levelFctors.size() - 1;
    int nfloats = 0;
    float stepI = maxX / dimI;
    float stepJ = maxY / dimJ;
    float x;
    float y;
    float tmpX;
    float tmpY;

    for ( int j = 0; j < dimJ + 1; ++j )
      for ( int i = 0; i < dimI + 1; ++i )
      {
        x = i * stepI;
        y = j * stepJ;
        tmpX = x / maxX;
        tmpY = y / maxY;

        mesh.setCoord(nfloats, x, y, m_levelFctors.get(0).apply(tmpX, tmpY));
        nfloats += 3;
        mesh.setCoord(nfloats, x, y, m_levelFctors.get(numLevel).apply(tmpX, tmpY));
        nfloats += 3;
      }

    return nfloats;
  }

  private <T extends PillarGrid> int generateZCornTopDown(T grid, float maxX, float maxY)
  {
    if ( grid instanceof PillarEGrid )
      return generateZCornTopDown((PillarEGrid) grid, maxX, maxY);
    else
      return generateZCornTopDown((PillarGridJIK) grid, maxX, maxY);
  }

  private int generateZCornTopDown(PillarEGrid mesh, float maxX, float maxY)
  {
    final int numLevels = m_levelFctors.size() - 1;
    final int dimI = mesh.getDimI();
    final int dimJ = mesh.getDimJ();
    final int dimK = mesh.getDimK();
    final int dimIxJ = dimI * dimJ;
    final int numLayerPerLevel = dimK / numLevels;
    final float stepI = maxX / dimI;
    final float stepJ = maxY / dimJ;
    float stepK;

    int numCells = 0;
    float height;
    float floor;
    float layerFloorHeight;
    int kstart;
    int kstop;
    float x;
    float y;
    float z;
    int zcornId;
    float offset;
    // for each level in mesh (starting from top)
    for ( int l = numLevels; l > 0; --l )
    {
      if ( l == numLevels )
        kstart = dimK - 1;
      else
        kstart = l * numLayerPerLevel - 1;
      kstop = (l - 1) * numLayerPerLevel;

      layerFloorHeight = 0;
      stepK = (m_levelFctors.get(l).apply(0, 0) - m_levelFctors.get(l - 1).apply(0, 0)) / (kstart + 1 - kstop);

      for ( int k = kstart; k > kstop - 1; --k )
      {
        for ( int j = 0; j < dimJ; ++j )
          for ( int i = 0; i < dimI; ++i )
          {
            int numPos = DemoSettings.HORIZONTAL_FAULTS ? 2 : k < mesh.m_dimK - 1 ? 1 : 2;
            for ( int kPos = 0; kPos < numPos; ++kPos )
            {
              height = (1 - kPos) * stepK;

              for ( int n = 0; n < 4; ++n )
              {
                x = (i + NEIGHBOUR_PILLAR[n][0]) * stepI;
                y = (j + NEIGHBOUR_PILLAR[n][1]) * stepJ;
                offset = computeOffset(i, j);
                if ( DemoSettings.HORIZONTAL_FAULTS && k < mesh.m_dimK - 1 && kPos > 0 )
                {
                  zcornId =
                      (mesh.m_numHFacesAtKLevel * (k + 1)) * (4 * dimIxJ) + (2 * j + NEIGHBOUR_PILLAR[n][1])
                          * (2 * mesh.m_dimI) + (2 * i + NEIGHBOUR_PILLAR[n][0]);
                  z = mesh.m_zcorn[zcornId];
                }
                else
                  z = m_levelFctors.get(l).apply(x / maxX, y / maxY) - height - layerFloorHeight + offset;
                floor = m_levelFctors.get(l - 1).apply(x / maxX, y / maxY) + offset;
                zcornId =
                    (mesh.m_numHFacesAtKLevel * k + kPos) * (4 * dimIxJ) + (2 * j + NEIGHBOUR_PILLAR[n][1])
                        * (2 * mesh.m_dimI) + (2 * i + NEIGHBOUR_PILLAR[n][0]);
                if ( z - floor < EPSILON )
                  mesh.m_zcorn[zcornId] = floor;
                else
                  mesh.m_zcorn[zcornId] = z;
              }
            }

            ++numCells;
          }

        layerFloorHeight += stepK;
      }
    }

    return numCells * 8;
  }

  private int generateZCornTopDown(PillarGridJIK mesh, float maxX, float maxY)
  {
    int numZcorns = 0;
    int zcornId, ni, nj;
    float x, y, z, offset, value;
    int numLevels = m_levelFctors.size() - 1;
    int numLayerPerLevel = mesh.m_dimK / numLevels;
    float floor;
    float stepI = maxX / mesh.m_dimI;
    float stepJ = maxY / mesh.m_dimJ;

    // for each level in mesh (starting from top)
    for ( int l = numLevels; l > 0; --l )
    {
      int kstart = l == numLevels ? mesh.m_dimK : l * numLayerPerLevel;
      int kstop = (l - 1) * numLayerPerLevel;
      float stepK = (m_levelFctors.get(l).apply(0, 0) - m_levelFctors.get(l - 1).apply(0, 0)) / (kstart - kstop);
      for ( int j = 0; j < mesh.m_dimJ + 1; ++j )
        for ( int i = 0; i < mesh.m_dimI + 1; ++i )
        {
          x = i * stepI;
          y = j * stepJ;
          floor = m_levelFctors.get(l - 1).apply(x / maxX, y / maxY);
          z = m_levelFctors.get(l).apply(x / maxX, y / maxY);
          zcornId = mesh.computeZCornId(i, j, kstart);
          for ( int k = kstart; k >= kstop; --k, z -= stepK )
          {
            value = (z - floor < EPSILON) ? floor : z;
            for ( int n = 0; n < 4; ++n, ++zcornId )
            {
              ni = i + NEIGHBOUR_CELLS[n][0];
              nj = j + NEIGHBOUR_CELLS[n][1];

              if ( ni < 0 || nj < 0 )
                continue;

              if ( ni < mesh.m_dimI && nj < mesh.m_dimJ )
              {
                offset = computeOffset(ni, nj);
                if ( !DemoSettings.HORIZONTAL_FAULTS || k < mesh.m_dimK )
                {
                  mesh.m_zcorn[zcornId] = value + offset;
                  ++numZcorns;
                }

                if ( DemoSettings.HORIZONTAL_FAULTS && k != 0 )
                {
                  mesh.m_zcorn[zcornId + 4] = value + offset;
                  ++numZcorns;
                }
              }
            }
            zcornId -= (mesh.m_numCorners + 4);
          }
        }
    }
    return numZcorns;
  }

  private float computeOffset(int i, int j)
  {
    float offset = 0;
    final int numCracks = m_crackFctors.size();

    for ( int n = 0; n < numCracks; ++n )
      offset += m_crackFctors.get(n).apply(i, j);

    return offset;
  }
}
