package meshvizxlm.eclipsemeshviz.scenegraph;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import com.openinventor.inventor.SbPlane;
import com.openinventor.inventor.SbRotation;
import com.openinventor.inventor.SbTime;
import com.openinventor.inventor.SbVec3f;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.actions.SoAction;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.details.SoDetail;
import com.openinventor.inventor.draggers.SoDragger;
import com.openinventor.inventor.draggers.SoJackDragger;
import com.openinventor.inventor.elements.SoInteractionElement;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.manips.SoClipPlaneManip;
import com.openinventor.inventor.misc.callbacks.SoDraggerCB;
import com.openinventor.inventor.misc.callbacks.SoEventCallbackCB;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.sensors.SoAlarmSensor;
import com.openinventor.inventor.sensors.SoNodeSensor;
import com.openinventor.meshvizxlm.extractors.MiExtractorCallback;
import com.openinventor.meshvizxlm.mapping.details.MoFaceDetailIjk;
import com.openinventor.meshvizxlm.mapping.nodes.*;
import com.openinventor.meshvizxlm.mesh.Dimension;

import meshvizxlm.eclipsemeshviz.DemoSettings;
import meshvizxlm.eclipsemeshviz.pillargrid.PillarGrid;
import meshvizxlm.eclipsemeshviz.wrappers.CellFilter;
import meshvizxlm.eclipsemeshviz.wrappers.EclipseMesh;
import meshvizxlm.eclipsemeshviz.wrappers.PillarMesh;
import meshvizxlm.eclipsemeshviz.wrappers.property.AnimatedPillarPorosity;
import meshvizxlm.eclipsemeshviz.wrappers.property.AnimatedPillarProperty;
import meshvizxlm.eclipsemeshviz.wrappers.property.BoundedProperty;
import meshvizxlm.eclipsemeshviz.wrappers.property.PillarProperty;
import meshvizxlm.eclipsemeshviz.wrappers.subsampled.MiAdjustableHexaMeshIjk;
import meshvizxlm.eclipsemeshviz.wrappers.subsampled.SubSampledMesh;
import meshvizxlm.eclipsemeshviz.wrappers.topology.PillarTopology;

public class OivSceneGraph

{
  private static final DecimalFormat scFormat = new DecimalFormat("#.###E0", new DecimalFormatSymbols(Locale.ENGLISH));
  private static final DecimalFormat decFormat =
      new DecimalFormat("#.######", new DecimalFormatSymbols(Locale.ENGLISH));

  private SoSeparator m_root;
  private SoTransform m_transform;
  private MoPredefinedColorMapping m_dataMapping;
  private SoSwitch m_switchMesh;
  private SoSwitch m_roiSwitch;

  private CellFilter m_roiFilter;
  private CellFilter m_lowResROIFilter;

  private MoDrawStyle m_drawStyle;

  private SoClipPlaneManip m_clipPlaneManip;
  private SoClipPlane m_clipPlane;
  private SoSwitch m_clipPlaneManipSwitch;
  private boolean m_facesPickingEnabled;
  private SoSwitch m_cellInfoDisplaySwitch;
  private SoText2 m_cellInfoDisplay;
  private MoScalarSetI m_moIsoScalarSetI;

  private SoAlarmSensor m_fullResSensor;
  private double m_delayToFullRes;
  private long[] m_slabOriginalPos;
  private long[] m_slabOriginalWidth;
  private int[] m_gap;

  private boolean m_alwaysDownsampled;
  private double m_resolution;
  private SoText2 m_gridInfoDisplay;
  private SoSwitch m_gridInfoDisplaySwitch;

  private ResolutionCallback m_timer;
  private ResolutionCallback m_lowRestimer;
  private MoMaterial m_skinMaterial;
  private MoMaterial m_hullMaterial;
  public SoText2 m_gridTriangleDisplay;

  public MeshSceneGraph m_fullResSceneGraph;
  public MeshSceneGraph m_lowResSceneGraph;

  public PillarMesh<?> m_fullResMesh;
  public SubSampledMesh<PillarGrid> m_lowResMesh;

  public OivSceneGraph()
  {
    m_timer = new ResolutionCallback();
    m_lowRestimer = new ResolutionCallback();

    m_alwaysDownsampled = false;
    m_resolution = 1;

    m_facesPickingEnabled = false;

    m_root = buildSceneGraph();
    m_gap = new int[3];

    // init
    m_delayToFullRes = DemoSettings.BACK_TO_FULL_RES_TIME;
    m_fullResSensor = new SoAlarmSensor();
    m_fullResSensor.setTask(new Runnable()
    {
      @Override
      public void run()
      {
        System.out.println("Back to full resolution...");
        restoreFullRes();
        m_fullResSensor.unschedule();
      }
    });

    // display slabs
    getSwitchLogicalSlice(0).whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
    getSwitchLogicalSlice(1).whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
    getSwitchLogicalSlice(2).whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
    // display hull
    m_fullResSceneGraph.m_switchHull.whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
  }

  public SoSeparator getRoot()
  {
    return m_root;
  }

  private SoSeparator buildSceneGraph()
  {
    // 3D view
    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    m_transform = new SoTransform();
    m_transform.scaleFactor.setValue(1.f, 1.f, 1.f);

    // Data mapping
    m_dataMapping = new MoPredefinedColorMapping();
    m_dataMapping.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.BLUE_WHITE_RED);

    // Data binding
    MoDataBinding dataBinding = new MoDataBinding();
    dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_CELL);

    m_switchMesh = new SoSwitch();
    m_switchMesh.setName("MeshSceneGraph");

    m_hullMaterial = new MoMaterial();
    // Be careful this value refers to the default value of the transparency
    // slider
    // Here we set the displayed transparency but not the value of the slider
    // See HullPanel
    m_hullMaterial.transparency.setValue((float) 0.8);

    m_drawStyle = new MoDrawStyle();
    m_drawStyle.displayFaces.setValue(true);
    m_drawStyle.displayEdges.setValue(true);
    m_drawStyle.fadingThreshold.setValue(5);
    m_skinMaterial = new MoMaterial();
    m_skinMaterial.lineColor.setValue(0, 0, 0);
    m_skinMaterial.lineColoring.setValue(MoMaterial.ColoringType.COLOR);
    m_skinMaterial.pointColor.setValue(1, 0, 0);
    m_skinMaterial.pointColoring.setValue(MoMaterial.ColoringType.COLOR);

    MoMaterial material = new MoMaterial();
    material.lineColor.setValue(0, 0, 0);
    material.lineColoring.setValue(MoMaterial.ColoringType.COLOR);
    material.pointColor.setValue(1, 0, 0);
    material.pointColoring.setValue(MoMaterial.ColoringType.COLOR);

    m_clipPlaneManip = new SoClipPlaneManip();
    m_clipPlaneManip.on.setValue(false);

    m_moIsoScalarSetI = new MoScalarSetI();

    m_clipPlaneManipSwitch = new SoSwitch();
    m_clipPlaneManipSwitch.addChild(m_clipPlaneManip);

    m_clipPlane = new SoClipPlane();
    m_clipPlane.on.setValue(false);
    m_clipPlane.plane.connectFrom(m_clipPlaneManip.plane);

    // Slab for each axis
    Dimension[] allAxis = Dimension.values();
    m_slabOriginalPos = new long[allAxis.length];
    m_slabOriginalWidth = new long[allAxis.length];

    SoEventCallback mouseMovedCb = new SoEventCallback();
    mouseMovedCb.addEventCallback(SoLocation2Event.class, new MouseMovedEventCallbackCB());

    SoCallback delayFullResolution = new SoCallback();
    delayFullResolution.setCallback(new SoCallback.CB()
    {
      @Override
      public void invoke(SoAction action)
      {
        if ( action instanceof SoGLRenderAction && SoInteractionElement.isInteracting(action.getState())
            && !m_alwaysDownsampled )
          delayFullRes();
      }
    });

    m_fullResSceneGraph = new MeshSceneGraph(m_skinMaterial, m_hullMaterial, m_timer);
    m_fullResSceneGraph.setName("Full Resolution");

    m_lowResSceneGraph = new MeshSceneGraph(m_skinMaterial, m_hullMaterial, m_lowRestimer);
    m_lowResSceneGraph.setName("Low Resolution");

    m_switchMesh.addChild(m_fullResSceneGraph);
    m_switchMesh.addChild(m_lowResSceneGraph);
    m_switchMesh.whichChild.setValue(0);

    m_roiSwitch = getSwitchROIDataRangeFilter();

    SoSeparator scene3D = new SoSeparator();
    scene3D.setName("Scene_3D");
    {
      scene3D.addChild(sh);
      scene3D.addChild(m_clipPlaneManipSwitch);
      scene3D.addChild(m_dataMapping);
      scene3D.addChild(material);
      scene3D.addChild(m_drawStyle);
      scene3D.addChild(m_transform);
      scene3D.addChild(dataBinding);
      scene3D.addChild(m_moIsoScalarSetI);
      scene3D.addChild(m_switchMesh);
      scene3D.addChild(mouseMovedCb);
      scene3D.addChild(delayFullResolution);

    }

    // 2D view
    SoOrthographicCamera camera2D = new SoOrthographicCamera();
    camera2D.viewportMapping.setValue(SoCamera.ViewportMappings.LEAVE_ALONE);

    SoSeparator scene2D = new SoSeparator();
    scene2D.boundingBoxIgnoring.setValue(true);
    scene2D.setName("Scene_2D");
    {
      SoPickStyle unpickable = new SoPickStyle();
      unpickable.style.setValue(SoPickStyle.Styles.UNPICKABLE);

      SoAnnoText3Property annoText3Property = new SoAnnoText3Property();
      annoText3Property.renderPrintType.setValue(SoAnnoText3Property.RenderPrintTypes.RENDER2D_PRINT_RASTER);
      annoText3Property.fontSizeHint.setValue(SoAnnoText3Property.FontSizeHints.ANNOTATION);

      SoFont font = new SoFont();
      font.size.setValue(15);

      scene2D.addChild(camera2D);
      scene2D.addChild(unpickable);
      scene2D.addChild(annoText3Property);
      scene2D.addChild(font);
      scene2D.addChild(buildCellInfoTextSceneGraph());
      scene2D.addChild(buildGridInfoTextSceneGraph());
    }

    SoSeparator root = new SoSeparator();
    { // assemble scene graph
      root.addChild(new SoGradientBackground());
      root.addChild(scene3D);
      root.addChild(scene2D);
    }

    return root;
  }

  private SoNode buildGridInfoTextSceneGraph()
  {
    SoTranslation trans = new SoTranslation();
    trans.translation.setValue(-0.95F, 0.90F, -0.1F);

    m_gridInfoDisplay = new SoText2();
    m_gridInfoDisplay.justification.setValue(SoText2.Justifications.LEFT);

    m_gridTriangleDisplay = new SoText2();
    m_gridTriangleDisplay.justification.setValue(SoText2.Justifications.LEFT);

    SoSeparator gridInfoSep = new SoSeparator();
    {
      gridInfoSep.addChild(trans);
      gridInfoSep.addChild(m_gridInfoDisplay);
      gridInfoSep.addChild(m_gridTriangleDisplay);
    }

    m_gridInfoDisplaySwitch = new SoSwitch();
    m_gridInfoDisplaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);

    {
      m_gridInfoDisplaySwitch.addChild(gridInfoSep);
    }

    SoAnnotation annotation = new SoAnnotation();
    annotation.addChild(m_gridInfoDisplaySwitch);

    return annotation;
  }

  private SoNode buildCellInfoTextSceneGraph()
  {
    SoTranslation trans1 = new SoTranslation();
    trans1.translation.setValue(-0.95F, 0.90F, -1.F);

    m_cellInfoDisplay = new SoText2();
    m_cellInfoDisplay.justification.setValue(SoText2.Justifications.LEFT);

    // Text display
    SoSeparator cellTextSep = new SoSeparator();
    {
      cellTextSep.addChild(trans1);
      cellTextSep.addChild(m_cellInfoDisplay);
    }

    m_cellInfoDisplaySwitch = new SoSwitch();
    m_cellInfoDisplaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    {
      m_cellInfoDisplaySwitch.addChild(cellTextSep);
    }

    SoAnnotation annotation = new SoAnnotation();
    annotation.addChild(m_cellInfoDisplaySwitch);

    return annotation;
  }

  private void displayCellInfo()
  {
    if ( !isSliceVisible() )
    {
      m_cellInfoDisplaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
      m_gridInfoDisplaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
    }
  }

  public void enableFacesPicking(boolean enable)
  {
    m_facesPickingEnabled = enable;
  }

  public void displayGridInfo()
  {
    m_gridInfoDisplaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
    m_cellInfoDisplaySwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
  }

  @SuppressWarnings("unchecked")
  public <T extends PillarGrid> void setMesh(PillarMesh<T> mesh)
  {
    m_fullResMesh = mesh;
    m_lowResMesh = new SubSampledMesh<PillarGrid>((PillarMesh<PillarGrid>) mesh);
    m_fullResSceneGraph.m_moMesh.setMesh(mesh);
    m_fullResSceneGraph.m_switchPlaneSlice.addChild(m_clipPlane);
    m_lowResSceneGraph.m_switchPlaneSlice.addChild(m_clipPlane);

    m_lowResSceneGraph.m_moMesh.setName("LowResMesh");

    int oldFrame = 1;
    PillarProperty dataset = (PillarProperty) ((EclipseMesh) m_fullResMesh).getDataSet();
    if ( m_fullResSceneGraph.m_moMesh.getMesh() != null )
      if ( dataset instanceof AnimatedPillarProperty )
        oldFrame = ((AnimatedPillarProperty) ((EclipseMesh) m_fullResMesh).getDataSet()).getFrame();
      else if ( dataset instanceof AnimatedPillarPorosity )
        oldFrame = ((AnimatedPillarPorosity) ((EclipseMesh) m_fullResMesh).getDataSet()).getFrame();

    m_fullResSceneGraph.m_moScalarSetIjk.setScalarSet(m_fullResMesh.getDataSet());

    m_lowResSceneGraph.m_moMesh.setMesh(m_lowResMesh);
    m_lowResSceneGraph.m_moScalarSetIjk.setScalarSet(m_lowResMesh.getDataSet());
    m_lowResSceneGraph.connectFrom(m_fullResSceneGraph);

    float xCenter = (float) mesh.getXCenter();
    float yCenter = (float) mesh.getYCenter();
    float zCenter = (float) mesh.getZCenter();
    m_transform.center.setValue(xCenter, yCenter, zCenter);

    BoundedProperty sset = m_fullResMesh.getDataSet();
    m_fullResSceneGraph.m_moScalarSetIjk.setScalarSet(sset);

    m_dataMapping.minValue.setValue((float) sset.getMin());
    m_dataMapping.maxValue.setValue((float) sset.getMax());

    m_roiFilter = new CellFilter();
    m_lowResROIFilter = new CellFilter();
    m_roiFilter.setMesh(mesh);
    m_lowResROIFilter.setMesh(m_lowResMesh);

    m_clipPlaneManip.draggerPosition.setValue(xCenter, yCenter, zCenter);
    m_clipPlaneManip.plane.setValue(new SbPlane(new SbVec3f(1, 0, 0), new SbVec3f(xCenter, yCenter, zCenter)));

    SoJackDragger dragger = (SoJackDragger) m_clipPlaneManip.getDragger();
    double[] bbox = mesh.getBBSize();
    dragger.scaleFactor.setValue((float) bbox[0] / 15, (float) bbox[1] / 15, (float) bbox[2] / 15);

    dragger.addMotionCallback(new ClipPlanerDraggerMotionCB(), null);

    PillarTopology<T> topology = (PillarTopology<T>) m_fullResMesh.getTopology();
    String unit = "";
    long numActiveCells = topology.getNumActiveCells();
    if ( numActiveCells > 1e6 )
    {
      unit = " million";
      numActiveCells /= 1e6;
    }

    if ( isosurfaceVisible() )
      m_moIsoScalarSetI.setScalarSet(m_fullResMesh.getPerNodeDataSet());

    m_gridInfoDisplay.string.setNum(3);
    m_gridInfoDisplay.string.set1Value(0, "I/J/K: " + topology.getNumCellsI() + " x " + topology.getNumCellsJ() + " x "
        + topology.getNumCellsK());
    m_gridInfoDisplay.string.set1Value(1, "#Active cells: " + numActiveCells + unit);
    m_gridInfoDisplay.string.set1Value(2, "Resolution 100%");

    m_timer.reset();
    setPlaybackPosition(oldFrame);
  }

  private boolean downsampledMeshVisible()
  {
    return m_switchMesh.whichChild.getValue() == 1;
  }

  private EclipseMesh getVisibleMesh()
  {
    if ( downsampledMeshVisible() )
      return m_lowResMesh;
    return m_fullResMesh;
  }

  private boolean isosurfaceVisible()
  {
    return getSwitchIsoSurface().whichChild.getValue() == SoSwitch.SO_SWITCH_ALL;
  }

  private void activateDownsampling(boolean activate)
  {
    if ( activate )
    {
      m_switchMesh.whichChild.setValue(1);
      m_gridInfoDisplay.string.set1Value(2, "Resolution : " + (int) (getLowResolution() * 100) + "%");
    }
    else
    {
      m_switchMesh.whichChild.setValue(0);
      m_fullResSceneGraph.touch();
      m_gridInfoDisplay.string.set1Value(2, "Resolution 100%");
    }
  }

  private void restoreFullRes()
  {
    m_fullResSensor.unschedule();
    activateDownsampling(false);
  }

  private void delayFullRes()
  {
    if ( downsampledMeshVisible() )
    {
      m_fullResSensor.unschedule();
      m_fullResSensor.setTimeFromNow(new SbTime(m_delayToFullRes));
      m_fullResSensor.schedule();
    }
  }

  private boolean isSliceVisible()
  {
    return getSwitchPlaneSlice().whichChild.getValue() == SoSwitch.SO_SWITCH_ALL;
  }

  public void setLowResolution(double resolution)
  {
    if ( resolution <= 0.5 )
    {
      // Needed by checkVisibleMesh() routine to
      // switch to full resolution if requested resolution
      // is higher than 50%
      m_resolution = resolution;
      adjustLowResolution(resolution);
      checkVisibleMesh();
    }
    else
    {
      restoreFullRes();
      m_resolution = 1;
    }
  }

  public void adjustLowResolution(double resolution)
  {
    m_lowResMesh.adjust(resolution);
    Dimension[] axis = Dimension.values();
    if ( downsampledMeshVisible() )
    {
      MoMeshSlab fullResSlab;
      MoMeshSlab lowResSlab;
      int slabIndex;
      int numSlabs;

      for ( int numAxis = 0; numAxis < axis.length; numAxis++ )
      {
        numSlabs = m_lowResSceneGraph.m_slabGroup[numAxis].getNumChildren();

        for ( int i = 0; i < numSlabs; ++i )
        {
          fullResSlab = (MoMeshSlab) m_fullResSceneGraph.m_slabGroup[numAxis].getChild(i);
          lowResSlab = (MoMeshSlab) m_lowResSceneGraph.m_slabGroup[numAxis].getChild(i);
          slabIndex = fullResSlab.index.getValue() / m_lowResMesh.getStep();

          lowResSlab.index.setValue(slabIndex);
          lowResSlab.thickness.setValue((int) Math.ceil(fullResSlab.thickness.getValue()
              / (double) m_lowResMesh.getStep()));
        }
      }

      m_lowResSceneGraph.m_moMesh.touch();
      m_fullResSceneGraph.m_moMesh.touch();
      m_gridInfoDisplay.string.set1Value(2, "Resolution : " + (int) (getLowResolution() * 100) + "%");
    }
  }

  public double getLowResolution()
  {
    final MiAdjustableHexaMeshIjk lowResMesh = (MiAdjustableHexaMeshIjk) m_lowResSceneGraph.m_moMesh.getMesh();
    return lowResMesh.getRatio();
  }

  public double getCurrentResolution()
  {
    return m_resolution;
  }

  public void setPlaybackPosition(int value)
  {
    checkVisibleMesh();
    final BoundedProperty dataset = ((EclipseMesh) m_fullResMesh).getDataSet();
    if ( dataset instanceof AnimatedPillarProperty )
      ((AnimatedPillarProperty) dataset).setFrame(value);
    else if ( dataset instanceof AnimatedPillarPorosity )
      ((AnimatedPillarPorosity) dataset).setFrame(value);
    m_fullResSceneGraph.m_moScalarSetIjk.touch();
    m_lowResSceneGraph.m_moScalarSetIjk.touch();

    // Re-extract scalar set for isosurface.
    if ( isosurfaceVisible() )
      m_moIsoScalarSetI.setScalarSet(getVisibleMesh().getPerNodeDataSet());
  }

  public void setZScale(float factor)
  {
    checkVisibleMesh();
    SbVec3f vec = m_transform.scaleFactor.getValue();
    vec.setValueAt(2, factor);
    m_transform.scaleFactor.setValue(vec);
  }

  // Full mesh

  public void setSkinVisibility(boolean visible)
  {
    getSwitchSkin().whichChild.setValue(visible ? SoSwitch.WhichChild.SO_SWITCH_ALL.getValue()
        : SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
  }

  public void setFading(float value)
  {
    m_drawStyle.fadingThreshold.setValue(value);
  }

  public void setPointsVisibility(boolean visible)
  {
    m_drawStyle.displayPoints.setValue(visible);
  }

  public void setEdgesVisibility(boolean visible)
  {
    m_drawStyle.displayEdges.setValue(visible);
  }

  public void setFacesVisibility(boolean visible)
  {
    m_drawStyle.displayFaces.setValue(visible);
  }

  public void setSkinTransparency(float value)
  {
    m_skinMaterial.transparency.setValue(value);
  }

  public void setSkinColorProperty(int index)
  {
    m_fullResSceneGraph.m_skin.colorScalarSetId.setValue(index);
    m_lowResSceneGraph.m_skin.colorScalarSetId.setValue(index);
  }

  // Slabs
  public void setSlabVisibility(Dimension axis, boolean visible)
  {
    getSwitchLogicalSlice(axis.ordinal()).whichChild.setValue(visible ? SoSwitch.WhichChild.SO_SWITCH_ALL.getValue()
        : SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
  }

  public void setSlabPosition(Dimension axis, int position)
  {
    checkVisibleMesh();

    final int axisOrdinal = axis.ordinal();
    int subSlabPosition;

    // Low resolution
    final SoGroup lowResSlabGroup = m_lowResSceneGraph.m_slabGroup[axisOrdinal];
    float factor = m_fullResMesh.getDimI() / m_lowResMesh.getTopology().getNumCellsI();
    int startPosition = (int) (position / factor);

    for ( int i = 0; i < lowResSlabGroup.getNumChildren(); i++ )
    {
      subSlabPosition = (int) (startPosition + (i * m_gap[axisOrdinal]) / factor);

      ((MoMeshSlab) lowResSlabGroup.getChild(i)).index.setValue(subSlabPosition);
    }

    // Full resolution
    final SoGroup fullResSlabGroup = m_fullResSceneGraph.m_slabGroup[axisOrdinal];
    startPosition = position;

    for ( int i = 0; i < fullResSlabGroup.getNumChildren(); i++ )
    {
      subSlabPosition = startPosition + m_gap[axisOrdinal] * i;

      ((MoMeshSlab) fullResSlabGroup.getChild(i)).index.setValue(subSlabPosition);
    }
    m_slabOriginalPos[axisOrdinal] = position;
  }

  public void setSlabWidth(Dimension axis, int width)
  {
    checkVisibleMesh();
    // low res
    float factor = m_fullResMesh.getDimI() / m_lowResMesh.getTopology().getNumCellsI();
    int newWidth;
    if ( width < factor && width > 0 )
      newWidth = 1;
    else
      newWidth = (int) (width / factor);

    for ( int i = 0; i < m_lowResSceneGraph.m_slabGroup[axis.ordinal()].getNumChildren(); i++ )
    {
      ((MoMeshSlab) m_lowResSceneGraph.m_slabGroup[axis.ordinal()].getChild(i)).thickness.setValue(newWidth);
    }
    for ( int i = 0; i < m_fullResSceneGraph.m_slabGroup[axis.ordinal()].getNumChildren(); i++ )
    {
      ((MoMeshSlab) m_fullResSceneGraph.m_slabGroup[axis.ordinal()].getChild(i)).thickness.setValue(width);
    }
    m_slabOriginalWidth[axis.ordinal()] = width;
  }

  public void setSlabGap(Dimension axis, int gap)
  {
    if ( gap == 1 )
      return;
    else
    {
      m_gap[axis.ordinal()] = gap;
      int numSlice = 0;

      if ( axis.ordinal() == 0 )
        numSlice = m_fullResMesh.getDimI() / m_gap[axis.ordinal()] + 1;
      else if ( axis.ordinal() == 1 )
        numSlice = m_fullResMesh.getDimJ() / m_gap[axis.ordinal()] + 1;
      else
        numSlice = m_fullResMesh.getDimK() / m_gap[axis.ordinal()] + 1;

      int currentIndex = ((MoMeshSlab) m_fullResSceneGraph.m_slabGroup[axis.ordinal()].getChild(0)).index.getValue();
      int currentThickness =
          ((MoMeshSlab) m_fullResSceneGraph.m_slabGroup[axis.ordinal()].getChild(0)).thickness.getValue();

      updateSlabs(axis, m_gap[axis.ordinal()], numSlice, currentIndex, currentThickness);
    }
  }

  public void updateSlabs(Dimension axis, int gap, int numSlice, int currentIndex, int currentThickness)
  {
    int numChildren = m_fullResSceneGraph.m_slabGroup[axis.ordinal()].getNumChildren();
    for ( int j = 0; j < numChildren; ++j )
    {
      m_fullResSceneGraph.m_slabGroup[axis.ordinal()].removeChild(0);
      m_lowResSceneGraph.m_slabGroup[axis.ordinal()].removeChild(0);
    }

    int subSlabPosition;
    int startPosition = currentIndex;

    for ( int i = 0; i < numSlice; ++i )
    {
      subSlabPosition = startPosition + i * m_gap[axis.ordinal()];

      MoMeshSlab slab = new MoMeshSlab();
      m_fullResSceneGraph.m_slabGroup[axis.ordinal()].addChild(slab);
      slab.dimension.setValue(axis);
      slab.thickness.setValue(currentThickness);
      slab.colorScalarSetId.setValue(0);
      slab.index.setValue(subSlabPosition);
    }

    float factor = m_fullResMesh.getDimI() / m_lowResMesh.getTopology().getNumCellsI();
    startPosition = (int) (currentIndex / factor);

    for ( int i = 0; i < numSlice; ++i )
    {
      subSlabPosition = (int) (startPosition + (i * m_gap[axis.ordinal()]) / factor);

      MoMeshSlab slab = new MoMeshSlab();
      m_lowResSceneGraph.m_slabGroup[axis.ordinal()].addChild(slab);
      slab.dimension.setValue(axis);
      slab.thickness.setValue(currentThickness);
      slab.colorScalarSetId.setValue(0);
      slab.index.setValue(subSlabPosition);
    }
    m_lowResSceneGraph.touch();
  }

  // Geometry slice
  public void setSliceVisibility(boolean visible)
  {
    getSwitchPlaneSlice().whichChild.setValue(visible ? SoSwitch.WhichChild.SO_SWITCH_ALL.getValue()
        : SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
    m_clipPlaneManipSwitch.whichChild.setValue(visible ? SoSwitch.WhichChild.SO_SWITCH_ALL.getValue()
        : SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
  }

  public void setSliceClipping(boolean active)
  {
    m_clipPlane.on.setValue(active);
  }

  public void setSliceRotAxis(int axis)
  {
    float xCenter = (float) m_fullResMesh.getXCenter();
    float yCenter = (float) m_fullResMesh.getYCenter();
    float zCenter = (float) m_fullResMesh.getZCenter();
    SbVec3f normal = new SbVec3f();
    normal.setValueAt(axis, 1.f);

    m_clipPlaneManip.plane.setValue(new SbPlane(normal, new SbVec3f(xCenter, yCenter, zCenter)));
  }

  public void setSliceRotAngle(int axis, float angle)
  {
    float xCenter = (float) m_fullResMesh.getXCenter();
    float yCenter = (float) m_fullResMesh.getYCenter();
    float zCenter = (float) m_fullResMesh.getZCenter();
    SbVec3f normal = new SbVec3f();
    normal.setValueAt(axis, 1.f);
    SbVec3f rotAxis = new SbVec3f();
    rotAxis.setValueAt((axis + 1) % 3, 1.f);
    SbRotation rotation = new SbRotation(rotAxis, (float) Math.PI / 180.f * angle);

    m_clipPlaneManip.plane.setValue(new SbPlane(rotation.multVec(normal), new SbVec3f(xCenter, yCenter, zCenter)));
  }

  public void setAutoDownsamplingState(boolean activate, double resolution)
  {
    m_alwaysDownsampled = !activate;
    activateDownsampling(m_alwaysDownsampled);
    setLowResolution(resolution);
  }

  // ROI
  public void setFilteringState(boolean active)
  {
    checkVisibleMesh();
    m_roiSwitch.whichChild.setValue(active ? SoSwitch.WhichChild.SO_SWITCH_ALL.getValue()
        : SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());

    if ( active && m_roiFilter.isFiltering() )
    {
      if ( m_fullResSceneGraph.m_moCellFilter.getCellFilter() != null )
      {
        m_fullResSceneGraph.m_moCellFilter.touch();
        m_lowResSceneGraph.m_moCellFilter.touch();
      }
      else
      {
        m_fullResSceneGraph.m_moCellFilter.setCellFilter(m_roiFilter);
        m_lowResSceneGraph.m_moCellFilter.setCellFilter(m_lowResROIFilter);
      }
    }
    else
    {
      m_fullResSceneGraph.m_moCellFilter.reset();
      m_lowResSceneGraph.m_moCellFilter.reset();
    }
  }

  public void setDataRange(float min, float max)
  {
    checkVisibleMesh();
    m_roiFilter.setDataRange(min, max);
    m_lowResROIFilter.setDataRange(min, max);
    if ( m_roiFilter.isFiltering() )
    {
      if ( m_fullResSceneGraph.m_moCellFilter.getCellFilter() != null )
      {
        m_fullResSceneGraph.m_moCellFilter.touch();
        m_lowResSceneGraph.m_moCellFilter.touch();
      }
      else
      {
        m_fullResSceneGraph.m_moCellFilter.setCellFilter(m_roiFilter);
        m_lowResSceneGraph.m_moCellFilter.setCellFilter(m_lowResROIFilter);
      }
    }
    else
    {
      m_fullResSceneGraph.m_moCellFilter.reset();
      m_lowResSceneGraph.m_moCellFilter.reset();
    }
  }

  public void setFilter(int imin, int imax, int jmin, int jmax, int kmin, int kmax)
  {
    checkVisibleMesh();
    m_roiFilter.setROI(imin, imax, jmin, jmax, kmin, kmax);
    m_lowResROIFilter.setROI(imin, imax, jmin, jmax, kmin, kmax);

    if ( m_roiFilter.isFiltering() )
    {
      if ( m_fullResSceneGraph.m_moCellFilter.getCellFilter() != null )
      {
        m_fullResSceneGraph.m_moCellFilter.touch();
        m_lowResSceneGraph.m_moCellFilter.touch();
      }
      else
      {
        m_fullResSceneGraph.m_moCellFilter.setCellFilter(m_roiFilter);
        m_lowResSceneGraph.m_moCellFilter.setCellFilter(m_lowResROIFilter);
      }
    }
    else
    {
      m_fullResSceneGraph.m_moCellFilter.reset();
      m_lowResSceneGraph.m_moCellFilter.reset();
    }

  }

  private void checkVisibleMesh()
  {
    if ( m_alwaysDownsampled && !downsampledMeshVisible() && m_resolution <= 0.5 )
      activateDownsampling(true);

  }

  private static class CombinedCallback implements MiExtractorCallback
  {
    private List<MiExtractorCallback> m_callbacks = new ArrayList<MiExtractorCallback>();

    public CombinedCallback(MiExtractorCallback cb1, MiExtractorCallback cb2)
    {
      m_callbacks.add(cb1);
      m_callbacks.add(cb2);
    }

    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      for ( MiExtractorCallback callback : m_callbacks )
        callback.beginCallback(geomChanged, topoChanged, dataSetChanged);
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      for ( MiExtractorCallback callback : m_callbacks )
        callback.endCallback(geomChanged, topoChanged, dataSetChanged);
    }
  }

  public static class TimerCallback implements MiExtractorCallback
  {
    private long t1;
    private String name;
    private boolean lowResTimer;

    public TimerCallback(String name)
    {
      this(name, false);
    }

    public TimerCallback(String name, boolean lowResTimer)
    {
      this.name = name;
      this.lowResTimer = lowResTimer;
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      long t2 = System.currentTimeMillis();
      if ( !lowResTimer || DemoSettings.LOW_RES_TIMINGS )
      {
        double time = (t2 - t1) / 1000.;
        if ( dataSetChanged )
          System.out.println(name + " dataset extracted in " + time + "s");
        else
          System.out.println(name + " extracted in " + time + "s");
      }
    }

    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      t1 = System.currentTimeMillis();
    }
  }

  private class AutoResolutionTimerCallback extends TimerCallback
  {
    public AutoResolutionTimerCallback(String name)
    {
      super(name);
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( !downsampledMeshVisible() || DemoSettings.LOW_RES_TIMINGS )
        super.endCallback(geomChanged, topoChanged, dataSetChanged);
      else if ( downsampledMeshVisible() )
        System.out.println(" extracted in low res ");
    }
  }

  private final class ClipPlanerDraggerMotionCB extends SoDraggerCB
  {
    @Override
    public void invoke(SoDragger dragger)
    {
      startDownsampling();
    }
  }

  private final class MouseMovedEventCallbackCB extends SoEventCallbackCB
  {
    // Method called each time the mouse is moved in the viewer window
    @Override
    public void invoke(SoEventCallback eventCB)
    {
      if ( !m_facesPickingEnabled )
        return;

      // Get the picked point (if any)
      // Picking is automatically performed using the event location
      SoPickedPoint pickedPoint = eventCB.getPickedPoint();
      if ( pickedPoint != null )
      {
        // Something has been picked : It could be either an Open Inventor
        // shape like SoCone or a mesh representation. Get detail and check
        // type.
        SoDetail detail = pickedPoint.getDetail();
        if ( detail instanceof MoFaceDetailIjk )
        {
          displayCellInfo();

          // A face of a mesh representation of a unstructured IJK mesh
          // has been picked. Get detail about the picked face.
          MoFaceDetailIjk fdetail = (MoFaceDetailIjk) detail;

          // Get the picked cell and the value at the picked point.
          long cellI = fdetail.getCellIndexI();
          long cellJ = fdetail.getCellIndexJ();
          long cellK = fdetail.getCellIndexK();
          double value = fdetail.getValue(pickedPoint.getPoint());

          // Process the detail information retreived by picking,
          // for example...
          display(value, cellI, cellJ, cellK);
        }
      }
      else
        displayGridInfo();
    }

    private void display(double value, long... cell)
    {
      StringBuffer buf = new StringBuffer("Cell: ");
      for ( long index : cell )
      {
        buf.append(index);
        buf.append(" ");
      }
      m_cellInfoDisplay.string.set1Value(0, buf.toString());
      m_cellInfoDisplay.string.set1Value(1, "Value: " + format(value));
    }

    private String format(double d)
    {
      if ( d == 0 || (d >= 0.00001 && d < 10000) || (d <= -0.00001 && d > -10000) )
        return decFormat.format(d);
      return scFormat.format(d);
    }
  }

  public void setIsosurfaceVisibility(boolean visible)
  {
    if ( visible )
      m_moIsoScalarSetI.setScalarSet(getVisibleMesh().getPerNodeDataSet());
    getSwitchIsoSurface().whichChild.setValue(visible ? SoSwitch.WhichChild.SO_SWITCH_ALL.getValue()
        : SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
  }

  public void setIsosurfaceValue(float value)
  {
    m_fullResSceneGraph.m_isosurface.isoValue.setValue(value);
  }

  public void requestFullResolution()
  {
    double resolution = m_resolution;
    restoreFullRes();
    m_resolution = resolution;
  }

  public void setDelayToFullResolution(int value)
  {
    m_delayToFullRes = value;
  }

  public void disableFullResolution()
  {
    if ( m_fullResSensor.isScheduled() )
      m_fullResSensor.unschedule();
  }

  public void startDownsampling()
  {
    checkVisibleMesh();
    if ( !m_alwaysDownsampled )
      activateDownsampling(true);
  }

  public void finishDownsampling()
  {
    if ( !m_alwaysDownsampled )
      delayFullRes();
  }

  private class ResolutionCallback implements MiExtractorCallback
  {
    private boolean m_activated;
    private long m_exceedCounter;
    protected long m_count;
    protected long m_startTime;
    protected boolean m_geomChanged;
    protected boolean m_topoChanged;
    private double m_maxExtractionTime = 0;

    public void reset()
    {
      m_activated = true;
      m_exceedCounter = 0;
      m_maxExtractionTime = 0;
    }

    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( m_count == 0 && (topoChanged || geomChanged) && !dataSetChanged )
      {
        m_startTime = System.currentTimeMillis();
        if ( m_activated )
        {
          m_geomChanged = geomChanged;
          m_topoChanged = topoChanged;
        }
      }
      ++m_count;
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( downsampledMeshVisible() )
        endCallbackLowRes(geomChanged, topoChanged, dataSetChanged);
      else
        endCallbackFullRes(geomChanged, topoChanged, dataSetChanged);
    }

    private void endCallbackLowRes(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      --m_count;
      if ( m_count == 0 && !m_alwaysDownsampled )
      {
        long time = System.currentTimeMillis() - m_startTime;

        if ( m_topoChanged || m_geomChanged )
        {
          if ( DemoSettings.LOW_RES_TIMINGS )
            DemoSettings.displayTime("low resolution extraction", time);
          if ( time > DemoSettings.LOW_RESOLUTION_EXTRACTION_TIME_REFERENCE1 )
            m_exceedCounter++;
          else
            m_exceedCounter = 0;
          // if the number of consecutive time that extraction exceeds the
          // reference time
          if ( m_exceedCounter > 3 )
          {
            m_exceedCounter = 0;
            // low resolution is not fast enough, reduce low resolution
            if ( !m_alwaysDownsampled )
            {
              final MiAdjustableHexaMeshIjk lowResMesh =
                  (MiAdjustableHexaMeshIjk) m_lowResSceneGraph.m_moMesh.getMesh();
              double step = 1 / lowResMesh.getRatio();
              adjustLowResolution(1 / (step + 1.01));
            }
          }
          m_geomChanged = false;
          m_topoChanged = false;
        }
      }
    }

    private void endCallbackFullRes(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      --m_count;
      if ( m_count == 0 )
      {
        long time = System.currentTimeMillis() - m_startTime;
        if ( m_topoChanged || m_geomChanged )
        {
          if ( time > m_maxExtractionTime )
          {
            if ( time < DemoSettings.LOW_RESOLUTION_EXTRACTION_TIME_REFERENCE1 )
            {
              // low resolution not needed, extraction is fast enough
              if ( !m_alwaysDownsampled )
                adjustLowResolution(1.0);
            }
            else
            {
              double timeFactor = time / DemoSettings.LOW_RESOLUTION_EXTRACTION_TIME_REFERENCE1;
              // Since extraction is not linear regarding to the size of the
              // grid
              // the step to group cells for low resolution grid is computed
              // based on nlog(n2)
              // so that it increases faster as the size of the grid does. This
              // is lowered by a step of 2.
              double step = Math.ceil(Math.pow(timeFactor * Math.log(timeFactor), 1 / 3.0));
              if ( !m_alwaysDownsampled )
                adjustLowResolution(1 / step);
              // if (!m_alwaysDownsampled)
              // for (DownsamplingParamListener listener : m_listeners)
              // listener.downsamplingParamChanged(m_resolution);
            }
            // max extraction is set to 10% above so that resolution is only
            // reevaluated if a slower
            // representation is extracted
            m_maxExtractionTime = time * 1.1;
          }
        }
        m_geomChanged = false;
        m_topoChanged = false;
      }
    }
  }

  public void setHullVisibility(boolean visible)
  {
    getSwitchHull().whichChild.setValue(visible ? SoSwitch.WhichChild.SO_SWITCH_ALL.getValue()
        : SoSwitch.WhichChild.SO_SWITCH_NONE.getValue());
  }

  public void setHullTransparency(float value)
  {
    m_hullMaterial.transparency.setValue(value);
  }

  public void setOutlineVisibility(boolean visible)
  {
    getSwitchHullOutline().whichChild.setValue(visible ? 0 : -1);
  }

  public void attachFullReshSceneGraph(SoNodeSensor m_triangleNodesensor)
  {
    m_triangleNodesensor.attach(m_fullResSceneGraph);
  }

  public void setReverseFiltering(boolean active)
  {
    m_roiFilter.setReverseDataFiltering(active);
    m_lowResROIFilter.setReverseDataFiltering(active);
    m_lowResROIFilter.setMesh(m_lowResMesh, m_lowResMesh.getDimI() / m_lowResMesh.getTopology().getNumCellsI(), false);
    m_roiFilter.setMesh(m_fullResMesh, 1.0f, false);
  }

  public SoSwitch getSwitchROIDataRangeFilter()
  {
    return m_fullResSceneGraph.m_switchROIDataRangeFilter;
  }

  public SoSwitch getSwitchSkin()
  {
    return m_fullResSceneGraph.m_switchSkin;
  }

  public SoSwitch getSwitchHull()
  {
    return m_fullResSceneGraph.m_switchHull;
  }

  public SoSwitch getSwitchPlaneSlice()
  {
    return m_fullResSceneGraph.m_switchPlaneSlice;
  }

  public SoSwitch getSwitchLogicalSlice(int sliceID)
  {
    return m_fullResSceneGraph.m_switchLogicalSlice[sliceID];
  }

  public SoSwitch getSwitchIsoSurface()
  {
    return m_fullResSceneGraph.m_switchIsosurface;
  }

  public SoSwitch getSwitchHullOutline()
  {
    return m_fullResSceneGraph.m_switchHullOutline;
  }

  public class MeshSceneGraph extends SoSeparator
  {
    public MoMesh m_moMesh;
    public MoScalarSetIjk m_moScalarSetIjk;

    public MoMeshSkin m_skin;
    public MoMeshSkin m_hullSkin;
    public SoGroup m_slabGroup[];
    public MoCellFilter m_moCellFilter;

    public MoMeshOutline m_hullOutline;

    public MoMeshPlaneSlice m_planeSlice;
    public MoMeshIsosurface m_isosurface;

    public SoSwitch m_switchSkin;
    public SoSwitch m_switchHullOutline;
    public SoSwitch m_switchHull;
    public SoSwitch m_switchPlaneSlice;
    public SoSwitch m_switchComposeLogicalSlice;
    public SoSwitch m_switchLogicalSlice[];
    public SoSwitch m_switchROIDataRangeFilter;
    public SoSwitch m_switchIsosurface;

    private MeshSceneGraph(MoMaterial skinMat, MoMaterial hullMat, MiExtractorCallback extractorCB)
    {
      Dimension[] allAxis = Dimension.values();

      m_moMesh = new MoMesh();
      m_moMesh.setName("Mesh");
      m_moScalarSetIjk = new MoScalarSetIjk();

      m_switchROIDataRangeFilter = new SoSwitch();
      m_moCellFilter = new MoCellFilter();
      m_switchROIDataRangeFilter.addChild(m_moCellFilter);

      m_switchSkin = new SoSwitch();
      m_switchSkin.setName("Skin");

      m_switchHullOutline = new SoSwitch();
      m_switchHullOutline.setName("Hull_Outline");

      m_switchHull = new SoSwitch();
      m_switchHull.setName("Hull");

      // Create the solid contour visualization node.
      m_skin = new MoMeshSkin();
      m_skin.setExtractorCallback(new CombinedCallback(m_timer, new TimerCallback("Skin")));
      m_skin.colorScalarSetId.setValue(0);

      m_hullSkin = new MoMeshSkin();
      m_hullSkin.setExtractorCallback(new CombinedCallback(m_timer, new TimerCallback("Hull")));
      m_hullSkin.colorScalarSetId.setValue(-1);

      m_switchPlaneSlice = new SoSwitch();
      m_switchPlaneSlice.setName("PlaneSlice");
      m_planeSlice = new MoMeshPlaneSlice();
      m_planeSlice.setExtractorCallback(new CombinedCallback(m_timer, new TimerCallback("Plane slice")));
      m_planeSlice.plane.connectFrom(m_clipPlaneManip.plane);

      m_hullOutline = new MoMeshOutline();
      m_hullOutline.setExtractorCallback(new CombinedCallback(m_timer, new TimerCallback("HullOutline")));
      m_hullOutline.colorScalarSetId.setValue(0);

      m_switchComposeLogicalSlice = new SoSwitch();
      m_switchComposeLogicalSlice.setName("ComposeLogicalSlice");

      m_switchLogicalSlice = new SoSwitch[allAxis.length];
      m_slabGroup = new SoGroup[allAxis.length];

      for ( int i = 0; i < allAxis.length; ++i )
      {
        m_switchLogicalSlice[i] = new SoSwitch();
        m_slabGroup[i] = new SoGroup();
        MoMeshSlab slab = new MoMeshSlab();
        m_slabGroup[i].addChild(slab);
        ((MoMeshSlab) m_slabGroup[i].getChild(0)).dimension.setValue(i);
        ((MoMeshSlab) m_slabGroup[i].getChild(0)).index.setValue(0);
        ((MoMeshSlab) m_slabGroup[i].getChild(0)).colorScalarSetId.setValue(0);
        ((MoMeshSlab) m_slabGroup[i].getChild(0)).setExtractorCallback(new CombinedCallback(m_timer, new TimerCallback(
            "Slab")));
        m_switchLogicalSlice[i].addChild(m_slabGroup[i]);
      }

      m_switchIsosurface = new SoSwitch();
      m_switchIsosurface.setName("Isosurface");

      m_isosurface = new MoMeshIsosurface();
      m_isosurface.setExtractorCallback(new CombinedCallback(m_timer, new TimerCallback("Isosurface")));

      assembleSceneGraph(skinMat, hullMat);
    }

    public void assembleSceneGraph(MoMaterial skinMaterial, MoMaterial hullMaterial)
    {
      Dimension[] allAxis = Dimension.values();

      this.addChild(m_moMesh);

      this.addChild(m_switchHull);
      {
        SoSeparator app = new SoSeparator();
        SoPickStyle pickStyle = new SoPickStyle();
        MoDrawStyle hullDrawStyle = new MoDrawStyle();
        hullDrawStyle.displayEdges.setValue(false);
        hullDrawStyle.displayPoints.setValue(false);
        pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
        m_switchHull.addChild(app);
        app.addChild(hullMaterial);
        app.addChild(pickStyle);
        app.addChild(hullDrawStyle);
        app.addChild(m_hullSkin);
      }

      this.addChild(m_switchHullOutline);
      {
        SoSeparator app = new SoSeparator();
        m_switchHullOutline.addChild(app);
        MoDrawStyle outlineDrawStyle = new MoDrawStyle();
        outlineDrawStyle.displayEdges.setValue(true);
        outlineDrawStyle.displayPoints.setValue(false);
        app.addChild(outlineDrawStyle);
        app.addChild(m_hullOutline);
      }

      this.addChild(m_moScalarSetIjk);

      this.addChild(m_switchROIDataRangeFilter);

      this.addChild(m_switchPlaneSlice);
      SoSeparator planeSliceSep = new SoSeparator();
      m_switchPlaneSlice.addChild(planeSliceSep);
      {
        planeSliceSep.addChild(m_planeSlice);
      }

      this.addChild(m_switchComposeLogicalSlice);
      m_switchComposeLogicalSlice.whichChild.setValue(SoSwitch.WhichChild.SO_SWITCH_ALL.getValue());
      SoSeparator sep = new SoSeparator();
      m_switchComposeLogicalSlice.addChild(sep);
      {
        SoDepthOffset depth = new SoDepthOffset();
        depth.offset.setValue((float) 0.001);
        sep.addChild(depth);
        for ( int i = 0; i < allAxis.length; ++i )
        {
          sep.addChild(m_switchLogicalSlice[i]);
        }
      }

      this.addChild(m_switchSkin);
      {
        SoSeparator app = new SoSeparator();
        m_switchSkin.addChild(app);
        app.addChild(skinMaterial);
        app.addChild(m_skin);
      }

      this.addChild(m_switchIsosurface);
      {
        SoSeparator app = new SoSeparator();
        m_switchIsosurface.addChild(app);
        MoDataBinding binding = new MoDataBinding();
        binding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);
        app.addChild(binding);
        app.addChild(m_isosurface);
      }
    }

    public void connectFrom(MeshSceneGraph sg)
    {
      Dimension[] allAxis = Dimension.values();
      m_planeSlice.plane.connectFrom(sg.m_planeSlice.plane);
      m_isosurface.isoValue.connectFrom(sg.m_isosurface.isoValue);

      m_switchSkin.whichChild.connectFrom(sg.m_switchSkin.whichChild);
      m_switchHullOutline.whichChild.connectFrom(sg.m_switchHullOutline.whichChild);
      m_switchHull.whichChild.connectFrom(sg.m_switchHull.whichChild);
      m_switchPlaneSlice.whichChild.connectFrom(sg.m_switchPlaneSlice.whichChild);
      m_switchComposeLogicalSlice.whichChild.connectFrom(sg.m_switchComposeLogicalSlice.whichChild);

      for ( int i = 0; i < allAxis.length; ++i )
      {
        m_switchLogicalSlice[i].whichChild.connectFrom(sg.m_switchLogicalSlice[i].whichChild);
      }
      m_switchROIDataRangeFilter.whichChild.connectFrom(sg.m_switchROIDataRangeFilter.whichChild);
      m_switchIsosurface.whichChild.connectFrom(sg.m_switchIsosurface.whichChild);
    }
  }

  public void updateDataBoundaries(BoundedProperty sset)
  {
    m_dataMapping.minValue.setValue((float) sset.getMin());
    m_dataMapping.maxValue.setValue((float) sset.getMax());
    m_lowResROIFilter.setDataRange((float) sset.getMin(), (float) sset.getMax());
    m_roiFilter.setDataRange((float) sset.getMin(), (float) sset.getMax());
  }

  public void updateLowResMesh()
  {
    m_lowResROIFilter.setMesh(m_lowResMesh);
  }
}
