package meshvizxlm.mapping.cellranges;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.actions.SoGLRenderAction.TransparencyTypes;
import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShapeHints;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.meshvizxlm.mapping.nodes.*;
import com.openinventor.meshvizxlm.mesh.Dimension;

import meshvizxlm.mesh.MbSampleMeshBuilder;
import meshvizxlm.mesh.data.MbScalarSetI;
import meshvizxlm.mesh.data.MbScalarSetIjk;
import meshvizxlm.mesh.volumes.MbHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVertexHexahedronMeshIjk;
import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  private static final java.awt.Dimension SLIDER_TEXT_DIM = new java.awt.Dimension(60, 20);

  private static final int NUM_CELL_I = 11;
  private static final int NUM_CELL_J = 15;
  private static final int NUM_CELL_K = 17;

  private static final String SCALARSET_I_NAME = "$Sphere";
  private static final String SCALARSET_IJK_NAME = "$CellId";

  private IViewerExaminer m_renderArea;
  private MbHexahedronMeshIjk m_meshHexaIjk;
  private MbVertexHexahedronMeshIjk m_meshVertexHexaIjk;

  private MoMesh m_moMesh;
  private MoPredefinedColorMapping m_colMap;
  private MoScalarSetI m_moScalarSetI;
  private MoScalarSetIjk m_moScalarSetIjk;
  private MoCellRanges m_globalCellRanges;
  private MoCellRanges m_internalCellRanges;
  private SoSwitch m_subskinSwitch;
  private MoMeshSkin m_subskin;
  private SoSwitch m_slabSwitch;

  public Main()
  {
    double[] min = { 0, 0, 0 };
    double[] max = { NUM_CELL_I, NUM_CELL_J, NUM_CELL_K };
    MbSampleMeshBuilder meshBuilder = new MbSampleMeshBuilder();

    m_meshHexaIjk = meshBuilder.getHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, min, max);
    m_meshVertexHexaIjk = meshBuilder.getVertexHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, min, max);

  }

  @Override
  public void start()
  {
    SoSeparator root = buildSceneGraph();

    m_renderArea = ViewerComponentsFactory.createViewerExaminer();
    m_renderArea.setSceneGraph(root);
    m_renderArea.getRenderArea().setTransparencyType(TransparencyTypes.DELAYED_BLEND);
    m_renderArea.viewAll();

    // SWING part
    final JCheckBox subSkinBox = new JCheckBox("SubSkin");
    subSkinBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( subSkinBox.isSelected() )
          m_subskinSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_subskinSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });
    subSkinBox.setSelected(true);

    final JCheckBox slabBox = new JCheckBox("Slabs");
    slabBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( slabBox.isSelected() )
          m_slabSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_slabSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });
    slabBox.setSelected(true);

    JPanel checkboxPanel = new JPanel();
    checkboxPanel.setLayout(new BoxLayout(checkboxPanel, BoxLayout.X_AXIS));
    checkboxPanel.add(subSkinBox);
    checkboxPanel.add(slabBox);

    JPanel slidersPanel1 = getPanel(m_globalCellRanges, "Global Range");
    JPanel slidersPanel2 = getPanel(m_internalCellRanges, "Internal Range");

    JPanel cellRangePanel = new JPanel();
    cellRangePanel.setLayout(new BoxLayout(cellRangePanel, BoxLayout.X_AXIS));
    cellRangePanel.add(slidersPanel1);
    cellRangePanel.add(slidersPanel2);

    JPanel meshPanel = new JPanel();
    meshPanel
        .setBorder(new TitledBorder(new EtchedBorder(), "Mesh", TitledBorder.LEADING, TitledBorder.TOP, null, null));
    meshPanel.setLayout(new BorderLayout());
    String[] items = new String[] { "Hexahedron IJK", "Vertex Hexahedron IJK" };
    final JComboBox<String> meshBox = new JComboBox<>(items);
    meshBox.setSelectedIndex(0);
    meshBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        int selectedIndex = meshBox.getSelectedIndex();

        if ( selectedIndex == 0 )
        { // hexahedron IJK
          m_moMesh.setMesh(m_meshHexaIjk);
          MbScalarSetI scalarSet = m_meshHexaIjk.getScalarSet(SCALARSET_I_NAME);
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          m_moScalarSetI.setScalarSet(scalarSet);
        }
        else
        { // Vertex hexahedron IJK
          m_moMesh.setMesh(m_meshVertexHexaIjk);
          MbScalarSetIjk scalarSetIjk = m_meshVertexHexaIjk.getScalarSetIjk(SCALARSET_IJK_NAME);
          m_colMap.minValue.setValue((float) scalarSetIjk.getMin());
          m_colMap.maxValue.setValue((float) scalarSetIjk.getMax());
          m_moScalarSetIjk.setScalarSet(scalarSetIjk);
        }
      }
    });
    meshPanel.add(meshBox, BorderLayout.CENTER);

    JPanel optionsPanel = new JPanel(new BorderLayout());
    optionsPanel.add(checkboxPanel, BorderLayout.NORTH);
    optionsPanel.add(cellRangePanel, BorderLayout.CENTER);
    optionsPanel.add(meshPanel, BorderLayout.SOUTH);

    final Component component = m_renderArea.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(optionsPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  private JPanel getPanel(final MoCellRanges cellRanges, String name)
  {
    JPanel slidersPanel = new JPanel();
    slidersPanel
        .setBorder(new TitledBorder(new EtchedBorder(), name, TitledBorder.LEADING, TitledBorder.TOP, null, null));
    slidersPanel.setLayout(new GridLayout(0, 1, 0, 0));

    SliderPanel iMinSliderPanel = new SliderPanel(0, NUM_CELL_I - 1, 1);
    ((FlowLayout) iMinSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    iMinSliderPanel.addInfoText("I min");
    iMinSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    iMinSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        cellRanges.min.set1Value(0, (long) value);
      }
    });
    slidersPanel.add(iMinSliderPanel);

    SliderPanel iMaxSliderPanel = new SliderPanel(0, NUM_CELL_I - 1, NUM_CELL_I - 2);
    ((FlowLayout) iMaxSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    iMaxSliderPanel.addInfoText("I max");
    iMaxSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    iMaxSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        cellRanges.max.set1Value(0, (long) value);
      }
    });
    slidersPanel.add(iMaxSliderPanel);

    SliderPanel jMinSliderPanel = new SliderPanel(0, NUM_CELL_J - 1, 1);
    ((FlowLayout) jMinSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    jMinSliderPanel.addInfoText("J min");
    jMinSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    jMinSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        cellRanges.min.set1Value(1, (long) value);
      }
    });
    slidersPanel.add(jMinSliderPanel);

    SliderPanel jMaxSliderPanel = new SliderPanel(0, NUM_CELL_J - 1, NUM_CELL_J - 2);
    ((FlowLayout) jMaxSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    jMaxSliderPanel.addInfoText("J max");
    jMaxSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    jMaxSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        cellRanges.max.set1Value(1, (long) value);
      }
    });
    slidersPanel.add(jMaxSliderPanel);

    SliderPanel kMinSliderPanel = new SliderPanel(0, NUM_CELL_K - 1, 1);
    ((FlowLayout) kMinSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    kMinSliderPanel.addInfoText("K min");
    kMinSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    kMinSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        cellRanges.min.set1Value(2, (long) value);
      }
    });
    slidersPanel.add(kMinSliderPanel);

    SliderPanel kMaxSliderPanel = new SliderPanel(0, NUM_CELL_K - 1, NUM_CELL_K - 2);
    ((FlowLayout) kMaxSliderPanel.getLayout()).setAlignment(FlowLayout.LEFT);
    kMaxSliderPanel.addInfoText("K max");
    kMaxSliderPanel.setInfoTextSize(SLIDER_TEXT_DIM);
    kMaxSliderPanel.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        cellRanges.max.set1Value(2, (long) value);
      }
    });
    slidersPanel.add(kMaxSliderPanel);
    return slidersPanel;
  }

  private SoSeparator buildSceneGraph()
  {
    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    m_moMesh = new MoMesh();
    m_moMesh.setMesh(m_meshHexaIjk);

    MbScalarSetI scalarSet = m_meshHexaIjk.getScalarSet(SCALARSET_I_NAME);

    m_colMap = new MoPredefinedColorMapping();
    m_colMap.minValue.setValue((float) scalarSet.getMin());
    m_colMap.maxValue.setValue((float) scalarSet.getMax());
    m_colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.STANDARD);

    MoDataBinding dataBinding = new MoDataBinding();
    dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);

    m_moScalarSetI = new MoScalarSetI();
    m_moScalarSetI.setScalarSet(scalarSet);

    m_moScalarSetIjk = new MoScalarSetIjk();

    m_globalCellRanges = new MoCellRanges();
    m_globalCellRanges.min.setNum(3);
    m_globalCellRanges.max.setNum(3);
    m_globalCellRanges.min.set1Value(0, 1);
    m_globalCellRanges.min.set1Value(1, 1);
    m_globalCellRanges.min.set1Value(2, 1);
    m_globalCellRanges.max.set1Value(0, NUM_CELL_I - 2);
    m_globalCellRanges.max.set1Value(1, NUM_CELL_J - 2);
    m_globalCellRanges.max.set1Value(2, NUM_CELL_K - 2);

    m_internalCellRanges = new MoCellRanges();
    m_internalCellRanges.min.setNum(3);
    m_internalCellRanges.max.setNum(3);
    m_internalCellRanges.min.set1Value(0, 1);
    m_internalCellRanges.min.set1Value(1, 1);
    m_internalCellRanges.min.set1Value(2, 1);
    m_internalCellRanges.max.set1Value(0, NUM_CELL_I - 2);
    m_internalCellRanges.max.set1Value(1, NUM_CELL_J - 2);
    m_internalCellRanges.max.set1Value(2, NUM_CELL_K - 2);

    SoLightModel lModel = new SoLightModel();
    lModel.model.setValue(SoLightModel.Models.BASE_COLOR);

    MoMeshOutline outline = new MoMeshOutline();
    outline.colorScalarSetId.setValue(0);

    // sub-skin
    m_subskin = new MoMeshSkin();
    m_subskin.colorScalarSetId.setValue(0);

    MoMesh subskinMesh = new MoMesh();
    subskinMesh.connectFrom(m_subskin);

    m_subskinSwitch = new SoSwitch();
    {
      m_subskinSwitch.addChild(m_subskin);
    }

    MoMeshSlab slabI = new MoMeshSlab();
    slabI.dimension.setValue(Dimension.I);
    slabI.index.setValue(NUM_CELL_I / 2);
    MoMeshSlab slabJ = new MoMeshSlab();
    slabJ.dimension.setValue(Dimension.J);
    slabJ.index.setValue(NUM_CELL_J / 2);
    MoMeshSlab slabK = new MoMeshSlab();
    slabK.dimension.setValue(Dimension.K);
    slabK.index.setValue(NUM_CELL_K / 2);

    m_slabSwitch = new SoSwitch();
    {
      m_slabSwitch.addChild(slabI);
      m_slabSwitch.addChild(slabJ);
      m_slabSwitch.addChild(slabK);
    }

    MoMaterial mat = new MoMaterial();
    mat.transparency.setValue((float) 0.5);

    SoSeparator root = new SoSeparator();
    {
      root.addChild(sh);
      root.addChild(m_moMesh);
      root.addChild(m_colMap);
      root.addChild(dataBinding);
      root.addChild(m_moScalarSetI);
      root.addChild(m_moScalarSetIjk);
      root.addChild(lModel);
      root.addChild(outline);
      root.addChild(m_globalCellRanges);
      root.addChild(mat);
      root.addChild(m_subskinSwitch);
      root.addChild(new MoMaterial());
      root.addChild(m_internalCellRanges);
      root.addChild(m_slabSwitch);
    }

    return root;
  }

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("CellRanges demo");
  }

}
