package meshvizxlm.mapping.cellshape;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Arrays;

import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import com.openinventor.inventor.nodes.SoLightModel;
import com.openinventor.inventor.nodes.SoPickStyle;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShapeHints;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.meshvizxlm.extractors.MiExtractorCallback;
import com.openinventor.meshvizxlm.mapping.nodes.*;
import com.openinventor.meshvizxlm.mesh.cell.MiCellFilterI;
import com.openinventor.meshvizxlm.mesh.cell.MiCellFilterIj;
import com.openinventor.meshvizxlm.mesh.cell.MiCellFilterIjk;

import meshvizxlm.mapping.MyVec3Set;
import meshvizxlm.mesh.MbSampleMeshBuilder;
import meshvizxlm.mesh.data.MbScalarSetI;
import meshvizxlm.mesh.data.MbScalarSetIj;
import meshvizxlm.mesh.data.MbScalarSetIjk;
import meshvizxlm.mesh.data.MbVec3SetI;
import meshvizxlm.mesh.surfaces.MbSurfaceMeshCurvilinear;
import meshvizxlm.mesh.surfaces.MbSurfaceMeshQuadrangle;
import meshvizxlm.mesh.volumes.MbHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVertexHexahedronMeshIjk;
import meshvizxlm.mesh.volumes.MbVolumeMeshHexahedron;
import meshvizxlm.mesh.volumes.MbVolumeMeshTetrahedron;
import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  static class MyExtractorCallback implements MiExtractorCallback
  {
    @Override
    public void beginCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("cellshape topology computation started...");
      if ( geomChanged )
        System.out.println("cellshape geometry computation started...");
      if ( dataSetChanged )
        System.out.println("dataset computation started...");
    }

    @Override
    public void endCallback(boolean geomChanged, boolean topoChanged, boolean dataSetChanged)
    {
      if ( topoChanged )
        System.out.println("cellshape topology computation finished...");
      if ( geomChanged )
        System.out.println("cellshape geometry computation finished...");
      if ( dataSetChanged )
        System.out.println("dataset computation finished...");
    }
  }

  static class CellFilterI implements MiCellFilterI
  {
    private static final int NUM_I_NUM_J = NUM_CELL_I * NUM_CELL_J;

    @Override
    public boolean acceptCell(long cellIndex)
    {
      int index = (int) cellIndex;
      int k = index / NUM_I_NUM_J;
      int ij = index % NUM_I_NUM_J;
      int j = ij / NUM_CELL_J;
      int i = ij % NUM_CELL_J;

      if ( i <= NUM_CELL_I / 3 && j <= NUM_CELL_J / 3 && k < NUM_CELL_K / 2 )
        return false;

      return true;
    }

    @Override
    public long getTimeStamp()
    {
      return 0;
    }
  }

  static class CellFilterIj implements MiCellFilterIj
  {
    @Override
    public boolean acceptCell(long i, long j)
    {
      if ( NUM_CELL_I / 3. < i && i <= 2 * NUM_CELL_I / 3. && NUM_CELL_J / 3. < j && j <= 2 * NUM_CELL_J / 3. )
        return false;

      return true;
    }

    @Override
    public long getTimeStamp()
    {
      return 0;
    }
  }

  static class CellFilterIjk implements MiCellFilterIjk
  {
    @Override
    public boolean acceptCell(int i, int j, int k)
    {
      if ( i <= NUM_CELL_I / 3 && j <= NUM_CELL_J / 3 && k < NUM_CELL_K / 2 )
        return false;

      return true;
    }

    @Override
    public long getTimeStamp()
    {
      return 0;
    }
  }

  enum DemoMesh
  {
    Tetrahedron("Tetrahedron"),
    Hexahedron("Hexahedron"),
    HexahedronIJK("Hexahedron IJK"),
    VertexHexahedronIJK("Vertex Hexahedron IJK"),
    Quadrangle("Unstructured quadrangle mesh"),
    CurvilinearSurface("Curvilinear surface mesh");

    private String name;

    private DemoMesh(String name)
    {
      this.name = name;
    }

    @Override
    public String toString()
    {
      return name;
    }
  }

  private static final DemoMesh DEFAULT_MESH = DemoMesh.Hexahedron;
  private static final int NUM_CELL_I = 10;
  private static final int NUM_CELL_J = 10;
  private static final int NUM_CELL_K = 10;
  private static final double[] MIN = { 0, 0, 0 };
  private static final double[] MAX = { NUM_CELL_I, NUM_CELL_J, NUM_CELL_K };
  private static final long[] VALUES = { 0, 3, 5, 7, 9 };
  private static final MbVec3SetI VEC3_SET_I = new MyVec3Set(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K);
  private static final CellFilterI CELL_FILTER_I = new CellFilterI();
  private static final CellFilterIj CELL_FILTER_IJ = new CellFilterIj();
  private static final CellFilterIjk CELL_FILTER_IJK = new CellFilterIjk();

  private final long[] m_valuesIJK;
  private final long[] m_valuesIJ;
  private IViewerExaminer m_renderArea;
  private MbSampleMeshBuilder m_meshBuilder;
  private MbVolumeMeshTetrahedron m_meshTetra;
  private MbVolumeMeshHexahedron m_meshHexa;
  private MbHexahedronMeshIjk m_meshHexaIjk;
  private MbVertexHexahedronMeshIjk m_meshVertexHexaIjk;
  private MbSurfaceMeshQuadrangle m_meshSurfaceQuadrangle;
  private MbSurfaceMeshCurvilinear m_meshSurfaceCurvilinear;
  private MoMesh m_moMesh;
  private SoSwitch m_filterSwitch;
  private MoCellFilter m_cellFilter;
  private SoSwitch m_vectorSwitch;
  private MoMeshVector m_moVector;
  private MoScalarSetI m_moScalarSetI;
  private MoScalarSetIj m_moScalarSetIj;
  private MoScalarSetIjk m_moScalarSetIjk;
  private MoVec3SetI m_moVec3SetI;
  private MoVec3SetIj m_moVec3SetIj;
  private MoVec3SetIjk m_moVec3SetIjk;
  private MoPredefinedColorMapping m_colMap;
  private MoMeshCellShape m_cellShape;
  private SoSwitch m_displaySwitch;

  public Main()
  {
    m_valuesIJK = new long[VALUES.length * 3];
    for ( int iValues = 0, iValuesIjk; iValues < VALUES.length; ++iValues )
    {
      iValuesIjk = iValues * 3;
      m_valuesIJK[iValuesIjk] = VALUES[iValues];
      m_valuesIJK[iValuesIjk + 1] = VALUES[iValues];
      m_valuesIJK[iValuesIjk + 2] = VALUES[iValues];
    }

    m_valuesIJ = new long[VALUES.length * 2];
    for ( int iValues = 0, iValuesIj; iValues < VALUES.length; ++iValues )
    {
      iValuesIj = iValues * 2;
      m_valuesIJ[iValuesIj] = VALUES[iValues];
      m_valuesIJ[iValuesIj + 1] = VALUES[iValues];
    }

    m_meshBuilder = new MbSampleMeshBuilder();
    m_meshHexa = null;
    m_meshHexaIjk = null;
    m_meshVertexHexaIjk = null;
    m_meshTetra = null;
    m_meshSurfaceQuadrangle = null;
    m_meshSurfaceCurvilinear = null;
  }

  @Override
  public void start()
  {
    SoSeparator root = buildSceneGraph();

    // SWING part
    final JCheckBox inclusiveBox = new JCheckBox("Inclusive list");
    inclusiveBox.setSelected(m_cellShape.inclusiveList.getValue());
    inclusiveBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        m_cellShape.inclusiveList.setValue(inclusiveBox.isSelected());
      }
    });

    final JCheckBox filterBox = new JCheckBox("Cell filter");
    filterBox.setSelected(m_filterSwitch.whichChild.getValue() != SoSwitch.SO_SWITCH_NONE);
    filterBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( filterBox.isSelected() )
          m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });

    final JCheckBox vectorBox = new JCheckBox("Display vectors");
    vectorBox.setSelected(m_vectorSwitch.whichChild.getValue() != SoSwitch.SO_SWITCH_NONE);
    vectorBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        if ( vectorBox.isSelected() )
          m_vectorSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
        else
          m_vectorSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
      }
    });

    final JCheckBox showCellNameBox = new JCheckBox("Show cell names");
    showCellNameBox.setSelected(m_cellShape.showCellName.getValue());
    showCellNameBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        m_cellShape.showCellName.setValue(showCellNameBox.isSelected());
      }
    });

    final JCheckBox showNodeNameBox = new JCheckBox("Show node names");
    showNodeNameBox.setSelected(m_cellShape.showNodeName.getValue());
    showNodeNameBox.addItemListener(new ItemListener()
    {
      @Override
      public void itemStateChanged(ItemEvent e)
      {
        m_cellShape.showNodeName.setValue(showNodeNameBox.isSelected());
      }
    });

    JPanel boxesPanel = new JPanel();
    GridBagLayout boxesGridBag = new GridBagLayout();
    GridBagConstraints boxesConstraints = new GridBagConstraints();
    boxesPanel.setLayout(boxesGridBag);

    boxesConstraints.weightx = 0.;
    boxesConstraints.weighty = 0.;
    boxesConstraints.anchor = GridBagConstraints.LINE_START;
    boxesConstraints.insets = new Insets(0, 20, 0, 0);
    boxesGridBag.setConstraints(inclusiveBox, boxesConstraints);
    boxesGridBag.setConstraints(vectorBox, boxesConstraints);

    boxesConstraints.gridwidth = GridBagConstraints.REMAINDER;
    boxesGridBag.setConstraints(showCellNameBox, boxesConstraints);
    boxesGridBag.setConstraints(showNodeNameBox, boxesConstraints);
    boxesGridBag.setConstraints(filterBox, boxesConstraints);

    boxesPanel.add(inclusiveBox);
    boxesPanel.add(showCellNameBox);
    boxesPanel.add(vectorBox);
    boxesPanel.add(showNodeNameBox);
    boxesPanel.add(filterBox);

    final SliderPanel nbIndSlider = new SliderPanel(0, VALUES.length, VALUES.length);
    nbIndSlider.setBorder(new TitledBorder(new EtchedBorder(), "Nb of indices"));
    nbIndSlider.setSliderSize(new Dimension(300, 20));
    nbIndSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        int nb = (int) value;
        switch ( m_moMesh.getMeshType() )
        {
        case VOLUME_MESH_HEXAHEDRON_IJK :
        case VOLUME_MESH_VERTEX_HEXAHEDRON_IJK :
          m_cellShape.cellIndices.setNum(nb * 3);
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(m_valuesIJK, nb * 3));
          break;
        case SURFACE_MESH_CURVILINEAR :
          m_cellShape.cellIndices.setNum(nb * 2);
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(m_valuesIJ, nb * 2));
          break;
        default:
          m_cellShape.cellIndices.setNum(nb);
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(VALUES, nb));
          break;
        }
      }
    });

    SliderPanel factorSlider = new SliderPanel(0.f, 2.f, m_cellShape.factor.getValue(), 1);
    factorSlider.setBorder(new TitledBorder(new EtchedBorder(), "Cell scale factor"));
    factorSlider.setSliderSize(new Dimension(300, 20));
    factorSlider.setTextFieldColumns(nbIndSlider.getTextFieldColumns());
    factorSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_cellShape.factor.setValue(value);
      }
    });

    JLabel label3 = new JLabel("Mesh : ");
    final DemoMesh[] meshes = DemoMesh.values();
    String[] meshItems = new String[meshes.length];
    for ( int i = 0; i < meshes.length; i++ )
    {
      meshItems[i] = meshes[i].toString();
    }
    final JComboBox<String> meshBox = new JComboBox<>(meshItems);
    meshBox.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        DemoMesh selectedMesh = meshes[meshBox.getSelectedIndex()];
        switch ( selectedMesh )
        {
        case Tetrahedron :
          if ( m_meshTetra == null )
            m_meshTetra = m_meshBuilder.getMeshTetrahedron();
          MbScalarSetI scalarSet = m_meshTetra.getScalarSet("$ScalarSetX");
          m_moScalarSetI.setScalarSet(scalarSet);
          m_moVec3SetI.setVec3Set(VEC3_SET_I);
          m_moVector.scaleFactor.setValue(1.f);
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);
          filterBox.setEnabled(false);
          m_moMesh.setMesh(m_meshTetra);
          m_displaySwitch.whichChild.setValue(0);
          m_cellShape.cellIndices.setNum((int) nbIndSlider.getSliderValue());
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(VALUES, (int) nbIndSlider.getSliderValue()));
          break;
        case Hexahedron :
          if ( m_meshHexa == null )
            m_meshHexa = m_meshBuilder.getMeshHexahedron(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, MIN, MAX);
          scalarSet = m_meshHexa.getScalarSet("$ScalarSetX");
          m_moScalarSetI.setScalarSet(scalarSet);
          m_moVec3SetI.setVec3Set(VEC3_SET_I);
          m_moVector.scaleFactor.setValue(1.f);
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          m_cellFilter.setCellFilter(CELL_FILTER_I);
          if ( filterBox.isSelected() )
            m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          filterBox.setEnabled(true);
          m_moMesh.setMesh(m_meshHexa);
          m_displaySwitch.whichChild.setValue(0);
          m_cellShape.cellIndices.setNum((int) nbIndSlider.getSliderValue());
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(VALUES, (int) nbIndSlider.getSliderValue()));
          break;
        case HexahedronIJK :
          if ( m_meshHexaIjk == null )
            m_meshHexaIjk = m_meshBuilder.getHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, MIN, MAX);
          scalarSet = m_meshHexaIjk.getScalarSet("$ScalarSetX");
          m_moScalarSetI.setScalarSet(scalarSet);
          m_moVec3SetI.setVec3Set(VEC3_SET_I);
          m_moVector.scaleFactor.setValue(1.f);
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          m_cellFilter.setCellFilter(CELL_FILTER_IJK);
          if ( filterBox.isSelected() )
            m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          filterBox.setEnabled(true);
          m_moMesh.setMesh(m_meshHexaIjk);
          m_displaySwitch.whichChild.setValue(0);
          m_cellShape.cellIndices.setNum((int) nbIndSlider.getSliderValue() * 3);
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(m_valuesIJK, (int) nbIndSlider.getSliderValue() * 3));
          break;
        case VertexHexahedronIJK :
          if ( m_meshVertexHexaIjk == null )
            m_meshVertexHexaIjk =
                m_meshBuilder.getVertexHexahedronMeshIjk(NUM_CELL_I, NUM_CELL_J, NUM_CELL_K, MIN, MAX);
          MbScalarSetIjk scalarsetIjk = m_meshVertexHexaIjk.getScalarSetIjk("$CellId");
          m_moScalarSetIjk.setScalarSet(scalarsetIjk);
          m_moVec3SetIjk.setVec3Set(m_meshVertexHexaIjk.getVec3SetIjk("$CellCenter"));
          m_moVector.scaleFactor.setValue(0.1f);
          m_colMap.minValue.setValue((float) scalarsetIjk.getMin());
          m_colMap.maxValue.setValue((float) scalarsetIjk.getMax());
          m_cellFilter.setCellFilter(CELL_FILTER_IJK);
          if ( filterBox.isSelected() )
            m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          filterBox.setEnabled(true);
          m_moMesh.setMesh(m_meshVertexHexaIjk);
          m_displaySwitch.whichChild.setValue(0);
          m_cellShape.cellIndices.setNum((int) nbIndSlider.getSliderValue() * 3);
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(m_valuesIJK, (int) nbIndSlider.getSliderValue() * 3));
          break;
        case Quadrangle :
          if ( m_meshSurfaceQuadrangle == null )
            m_meshSurfaceQuadrangle = m_meshBuilder.getSurfaceMeshQuadrangle(NUM_CELL_I, NUM_CELL_J, MIN,
                new double[] { 10. / NUM_CELL_I, 0, 0 }, new double[] { 0, 10. / NUM_CELL_J, 0 }, null);
          scalarSet = m_meshSurfaceQuadrangle.getScalarSet("$ScalarSetX");
          m_moScalarSetI.setScalarSet(scalarSet);
          m_moVec3SetI.setVec3Set(VEC3_SET_I);
          m_moVector.scaleFactor.setValue(1.f);
          m_colMap.minValue.setValue((float) scalarSet.getMin());
          m_colMap.maxValue.setValue((float) scalarSet.getMax());
          m_cellFilter.setCellFilter(CELL_FILTER_I);
          if ( filterBox.isSelected() )
            m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          filterBox.setEnabled(true);
          m_moMesh.setMesh(m_meshSurfaceQuadrangle);
          m_displaySwitch.whichChild.setValue(1);
          m_cellShape.cellIndices.setNum((int) nbIndSlider.getSliderValue());
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(VALUES, (int) nbIndSlider.getSliderValue()));
          break;
        case CurvilinearSurface :
          if ( m_meshSurfaceCurvilinear == null )
          {
            double[] origin = { 0, 0, 0 };
            double[] stepVecI = { 1, 0, 0 };
            double[] stepVecJ = { 0, 1, 0 };
            m_meshSurfaceCurvilinear =
                m_meshBuilder.getSurfaceMeshCurvilinear(NUM_CELL_I, NUM_CELL_J, origin, stepVecI, stepVecJ);
          }
          MbScalarSetIj scalarSetIj = m_meshSurfaceCurvilinear.getScalarSetIj("$ScalarSetX");
          m_moScalarSetIj.setScalarSet(scalarSetIj);
          m_moVec3SetIj.setVec3Set(m_meshSurfaceCurvilinear.getVec3SetIj("$MyGeometry"));
          m_moVector.scaleFactor.setValue(.2f);
          m_colMap.minValue.setValue((float) scalarSetIj.getMin());
          m_colMap.maxValue.setValue((float) scalarSetIj.getMax());
          m_cellFilter.setCellFilter(CELL_FILTER_IJ);
          if ( filterBox.isSelected() )
            m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_ALL);
          filterBox.setEnabled(true);
          m_moMesh.setMesh(m_meshSurfaceCurvilinear);
          m_displaySwitch.whichChild.setValue(1);
          m_cellShape.cellIndices.setNum((int) nbIndSlider.getSliderValue() * 2);
          m_cellShape.cellIndices.setValues(0, Arrays.copyOf(m_valuesIJ, (int) nbIndSlider.getSliderValue() * 2));
          break;
        default:
          break;
        }
      }
    });
    meshBox.setSelectedIndex(DEFAULT_MESH.ordinal());

    JPanel meshPanel = new JPanel();
    GridBagLayout gridBag = new GridBagLayout();
    GridBagConstraints constraints = new GridBagConstraints();
    meshPanel.setLayout(gridBag);

    constraints.fill = GridBagConstraints.NONE;
    constraints.weightx = 0.;
    constraints.weighty = 0.;
    constraints.anchor = GridBagConstraints.LINE_END;
    constraints.ipadx = 10;
    constraints.ipady = 30;
    gridBag.setConstraints(label3, constraints);

    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 1.;
    constraints.weighty = 0.;
    constraints.ipady = 0;
    constraints.gridwidth = GridBagConstraints.REMAINDER;
    gridBag.setConstraints(nbIndSlider, constraints);
    gridBag.setConstraints(factorSlider, constraints);
    gridBag.setConstraints(meshBox, constraints);

    meshPanel.add(label3);
    meshPanel.add(meshBox);
    meshPanel.add(nbIndSlider);
    meshPanel.add(factorSlider);

    JPanel panel2 = new JPanel(new BorderLayout());
    panel2.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED, null, null));
    panel2.add(boxesPanel, BorderLayout.EAST);
    panel2.add(meshPanel, BorderLayout.CENTER);

    m_renderArea = ViewerComponentsFactory.createViewerExaminer();
    m_renderArea.setSceneGraph(root);
    m_renderArea.viewAll();

    final Component component = m_renderArea.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(panel2, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    m_renderArea.dispose();
  }

  private SoSeparator buildSceneGraph()
  {
    SoShapeHints sh = new SoShapeHints();
    sh.vertexOrdering.setValue(SoShapeHints.VertexOrderings.CLOCKWISE);

    m_moMesh = new MoMesh();
    m_moMesh.setMesh(m_meshHexa);

    m_colMap = new MoPredefinedColorMapping();
    m_colMap.predefColorMap.setValue(MoPredefinedColorMapping.PredefColorMapping.STANDARD);

    m_moScalarSetI = new MoScalarSetI();
    m_moScalarSetIj = new MoScalarSetIj();
    m_moScalarSetIjk = new MoScalarSetIjk();

    m_filterSwitch = new SoSwitch();
    m_filterSwitch.whichChild.setValue(SoSwitch.SO_SWITCH_NONE);

    m_cellFilter = new MoCellFilter();

    MoDataBinding dataBinding = new MoDataBinding();
    dataBinding.dataBinding.setValue(MoDataBinding.DataBinding.PER_NODE);

    m_cellShape = new MoMeshCellShape();
    m_cellShape.setExtractorCallback(new MyExtractorCallback());
    m_cellShape.factor.setValue(0.7f);
    m_cellShape.offset.setValue(0.05f);
    m_cellShape.relativeOffset.setValue(true);
    m_cellShape.showCellName.setValue(true);
    m_cellShape.showNodeName.setValue(true);
    m_cellShape.colorScalarSetId.setValue(0);

    // Transparent Mesh Skin
    MoDrawStyle style = new MoDrawStyle();
    style.displayFaces.setValue(false);
    style.displayEdges.setValue(true);
    SoPickStyle pickStyle = new SoPickStyle();
    pickStyle.style.setValue(SoPickStyle.Styles.UNPICKABLE);
    SoLightModel lModel = new SoLightModel();
    lModel.model.setValue(SoLightModel.Models.BASE_COLOR);
    MoMeshSkin skin = new MoMeshSkin();
    skin.colorScalarSetId.setValue(0);
    SoSeparator skinSep = new SoSeparator(4);
    {
      skinSep.addChild(style);
      skinSep.addChild(pickStyle);
      skinSep.addChild(lModel);
      skinSep.addChild(skin);
    }

    // Surface
    MoDrawStyle moStyle = new MoDrawStyle();
    moStyle.displayEdges.setValue(true);
    moStyle.displayFaces.setValue(false);
    MoMeshSurface surface = new MoMeshSurface();
    surface.colorScalarSetId.setValue(0);
    SoSeparator surfaceSep = new SoSeparator(2);
    {
      surfaceSep.addChild(moStyle);
      surfaceSep.addChild(surface);
    }

    m_displaySwitch = new SoSwitch(2);
    {
      m_displaySwitch.addChild(skinSep);
      m_displaySwitch.addChild(surfaceSep);
    }

    // This mesh is used to connect the cells generated mesh as input of the
    // vectors
    MoMesh cellMesh = new MoMesh();
    cellMesh.connectFrom(m_cellShape);

    // Vector field
    m_moVec3SetI = new MoVec3SetI();
    m_moVec3SetIj = new MoVec3SetIj();
    m_moVec3SetIjk = new MoVec3SetIjk();

    // Display vector on grid
    m_vectorSwitch = new SoSwitch();
    m_moVector = new MoMeshVector();

    SoSeparator root = new SoSeparator();
    {
      m_filterSwitch.addChild(m_cellFilter);
      m_vectorSwitch.addChild(m_moVector);

      root.addChild(sh);
      root.addChild(m_moMesh);
      root.addChild(m_colMap);
      root.addChild(m_moScalarSetI);
      root.addChild(m_moScalarSetIj);
      root.addChild(m_moScalarSetIjk);
      root.addChild(m_filterSwitch);
      root.addChild(dataBinding);
      root.addChild(m_cellShape);

      root.addChild(m_displaySwitch);

      root.addChild(cellMesh);
      root.addChild(m_moVec3SetI);
      root.addChild(m_moVec3SetIj);
      root.addChild(m_moVec3SetIjk);
      root.addChild(m_vectorSwitch);
    }

    return root;
  }

  public static void main(String[] argv)
  {
    Main example = new Main();
    example.demoMain("Cell shape Example");
  }
}
