package meshvizxlm.mesh.cell;

import java.util.ArrayList;
import java.util.Arrays;

import com.openinventor.meshvizxlm.extractors.cell.MxCellExtract;
import com.openinventor.meshvizxlm.extractors.cell.MxPolyhedronCellExtract;
import com.openinventor.meshvizxlm.mesh.cell.MiVolumeCell;
import com.openinventor.meshvizxlm.mesh.geometry.MiGeometryI;

public class MbPolyhedricCell implements MiVolumeCell
{
  private long[][] m_facesList;
  private long[] m_nodeIds;
  protected long m_beginNodeId;
  protected long m_endNodeId;

  public MbPolyhedricCell()
  {}

  public MbPolyhedricCell(long[] nodeIds)
  {
    assignNodeIndices(nodeIds);
  }

  public MbPolyhedricCell(long[] nodeIds, long[][] faceList)
  {
    assignNodeIndices(nodeIds);
    defineFaces(faceList);
  }

  public MbPolyhedricCell(ArrayList<Long> nodeIds, long[][] faceList)
  {
    assignNodeIndices(nodeIds);
    defineFaces(faceList);
  }

  @Override
  public int getNumNodes()
  {
    return m_nodeIds.length;
  }

  public long getBeginNodeId()
  {
    return m_beginNodeId;
  }

  public long getEndNodeId()
  {
    return m_endNodeId;
  }

  public void assignNodeIndices(long[] nodeIds)
  {
    m_nodeIds = new long[nodeIds.length];

    System.arraycopy(nodeIds, 0, m_nodeIds, 0, nodeIds.length);
    for ( int i = 0; i < m_nodeIds.length; i++ )
    {
      if ( m_nodeIds[i] < m_beginNodeId )
        m_beginNodeId = m_nodeIds[i];

      if ( m_nodeIds[i] + 1 > m_endNodeId )
        m_endNodeId = m_nodeIds[i] + 1;
    }

  }

  public void assignNodeIndices(ArrayList<Long> nodeIds)
  {
    m_nodeIds = new long[nodeIds.size()];
    m_beginNodeId = Integer.MAX_VALUE;
    m_endNodeId = 0;

    for ( int i = 0; i < m_nodeIds.length; i++ )
    {
      m_nodeIds[i] = nodeIds.get(i);

      if ( m_nodeIds[i] < m_beginNodeId )
        m_beginNodeId = m_nodeIds[i];

      if ( m_nodeIds[i] + 1 > m_endNodeId )
        m_endNodeId = m_nodeIds[i] + 1;
    }
  }

  public void defineFaces(long[][] faceList)
  {
    int faceListSize = faceList.length;
    m_facesList = new long[faceListSize][];

    for ( int i = 0; i < faceListSize; i++ )
      m_facesList[i] = Arrays.copyOf(faceList[i], faceList[i].length);
  }

  @Override
  public long getNodeIndex(int node)
  {
    return m_nodeIds[node];
  }

  @Override
  public int getNumFacets()
  {
    return m_facesList.length;
  }

  @Override
  public long[] getNodesIndexOfFacet(int facet, long[] facetNodes)
  {
    long[] face = m_facesList[facet];
    int size = face.length;

    if ( facetNodes == null || facetNodes.length != size )
      facetNodes = new long[size];

    for ( int i = 0; i < size; i++ )
      facetNodes[i] = m_nodeIds[(int) face[i]];

    return facetNodes;
  }

  @Override
  public int[] getIsosurfTopology(int caseId, int[] edges)
  {
    // not supported by this kind of cell. Must return null
    return null;
  }

  @Override
  public int getMarchingCaseId(boolean[] nodesSign, long beginNodeId)
  {
    // not supported by this kind of cell. Must return 0
    return 0;
  }

  @Override
  public double[] getFacetCenter(int facet, MiGeometryI geometry)
  {
    long[] facetNodeInds = m_facesList[facet];
    double[] facetCenter = { 0, 0, 0 };
    double[] coord;
    int nbNodes = facetNodeInds.length;

    for ( int i = 0; i < nbNodes; ++i )
    {
      coord = geometry.getCoord(m_nodeIds[(int) facetNodeInds[i]]);
      facetCenter[0] += coord[0];
      facetCenter[1] += coord[1];
      facetCenter[2] += coord[2];
    }

    facetCenter[0] /= nbNodes;
    facetCenter[1] /= nbNodes;
    facetCenter[2] /= nbNodes;

    return facetCenter;
  }

  @Override
  public boolean isPointInsideCell(MiGeometryI meshGeometry, double[] point, double[] weights)
  {
    return MxPolyhedronCellExtract.isPointInsideCell(meshGeometry, this, point, weights);
  }

  @Override
  public double[] getCenter(MiGeometryI geometry)
  {
    return MxCellExtract.getCenter(geometry, this);
  }

  @Override
  public double getRelativeSize(MiGeometryI meshGeometry)
  {
    return 1;
  }
}
