package meshvizxlm.mesh.topology;

import com.openinventor.meshvizxlm.mesh.StorageLayoutIJK;
import com.openinventor.meshvizxlm.mesh.topology.MiHexahedronTopologyExplicitIjk;

/**
 * List of volume cells' indices.
 * <p>
 * A topology for a unstructured explicit hexahedron mesh Ijk.<br>
 * Hexahedrons are defines as follows:
 *
 * <PRE>
 *                        J
 *                        |
 *                        |
 *
 *                       n3----------n2   facet 0 = 0321
 *                       /|          /|   facet 1 = 4567
 *                     /  |        /  |   facet 2 = 0473
 *                   /    |      /    |   facet 3 = 1265
 *                 n7---------n6      |   facet 4 = 0154
 *                  |     |    |      |   facet 5 = 3762
 *                  |    n0----|-----n1    --- I
 *                  |    /     |     /
 *                  |  /       |   /
 *                  |/         | /
 *                 n4---------n5
 *
 *               /
 *             /
 *           K
 * </PRE>
 */
public class MbHexahedronTopologyExplicitIjk extends MbTopologyIjk implements MiHexahedronTopologyExplicitIjk
{

  protected int[] m_indices;
  protected int m_beginNodeId;
  protected int m_endNodeId;
  private StorageLayoutIJK m_storageLayout;

  public MbHexahedronTopologyExplicitIjk(int numCellI, int numCellJ, int numCellK, StorageLayoutIJK storageLayout)
  {
    super(numCellI, numCellJ, numCellK, storageLayout);

    m_beginNodeId = Integer.MAX_VALUE;
    m_endNodeId = 0;
    int size = numCellK * numCellJ * numCellI * 8;
    m_indices = new int[size];

    m_storageLayout = storageLayout;
  }

  /**
   * Direct access to the indices array. Useful for non indexed hexahedron Ijk
   * mesh built from an hexahendron Ijk mesh, use with caution!
   */
  public int[] getIndices()
  {
    return m_indices;
  }

  @Override
  public long[] getCellNodeIndices(int i, int j, int k, long[] nodeIndices)
  {
    if ( nodeIndices == null || nodeIndices.length < 8 )
      nodeIndices = new long[8];

    int index = (int) m_adaptor.getI(i, j, k) * 8;
    nodeIndices[0] = m_indices[index];
    nodeIndices[1] = m_indices[index + 1];
    nodeIndices[2] = m_indices[index + 2];
    nodeIndices[3] = m_indices[index + 3];
    nodeIndices[4] = m_indices[index + 4];
    nodeIndices[5] = m_indices[index + 5];
    nodeIndices[6] = m_indices[index + 6];
    nodeIndices[7] = m_indices[index + 7];

    return nodeIndices;
  }

  @Override
  public long getEndNodeId()
  {
    return m_endNodeId;
  }

  @Override
  public long getBeginNodeId()
  {
    if ( m_endNodeId == 0 )
      return 0;
    else
      return m_beginNodeId;
  }

  @Override
  public StorageLayoutIJK getStorageLayout()
  {
    return m_storageLayout;
  }

  /**
   * Set the nodes indices of cell (i,j,k)
   *
   * @param i
   * @param j
   * @param k
   * @param nodeIds
   */
  public void setCellNodeIndices(int i, int j, int k, int[] nodeIds)
  {
    assert (nodeIds.length == 8);

    int index = (int) m_adaptor.getI(i, j, k) * 8;

    for ( int id : nodeIds )
    {
      if ( id >= m_endNodeId )
        m_endNodeId = id + 1;
      if ( id < m_beginNodeId )
        m_beginNodeId = id;
      m_indices[index] = id;
      index++;
    }

    touch();
  }

  public long getNumCells()
  {
    return m_numK * m_numJ * m_numI;
  }

}
