package remoteviz.independentservice.independentrenderingservice;

import java.awt.image.BufferedImage;

import com.jogamp.opengl.DefaultGLCapabilitiesChooser;
import com.jogamp.opengl.GL2;
import com.jogamp.opengl.GLAutoDrawable;
import com.jogamp.opengl.GLCapabilities;
import com.jogamp.opengl.GLDrawableFactory;
import com.jogamp.opengl.GLProfile;

import com.openinventor.inventor.events.*;

import com.jogamp.opengl.util.awt.AWTGLReadBufferUtil;

/**
 * Render engine
 */
public class Engine
{
  private Cube cube;
  private Transform trans;
  private int prevMouseX;
  private int prevMouseY;
  private int width;
  private int height;
  private boolean isLeftButtonDown;
  private GLAutoDrawable drawable;
  private Frame frame;

  public class Frame
  {
    boolean isInteractive = false;
    BufferedImage bufferedImage = null;
    long id = 0;
  }

  /**
   * Default Constructor
   */
  public Engine(int width, int height)
  {
    this.width = width;
    this.height = height;

    this.isLeftButtonDown = false;

    // initialize the cube
    this.cube = new Cube(new Point3d(5, 5, 0), 4);

    // Setting values for transform matrix
    this.trans = new Transform();

    this.frame = new Frame();
  }

  /**
   * Initializes the render engine
   */
  public synchronized void init()
  {
    GLProfile glp = GLProfile.getDefault();
    GLCapabilities caps = new GLCapabilities(glp);
    caps.setHardwareAccelerated(true);
    caps.setDoubleBuffered(false);
    caps.setAlphaBits(8);
    caps.setRedBits(8);
    caps.setBlueBits(8);
    caps.setGreenBits(8);
    caps.setOnscreen(false);
    GLDrawableFactory factory = GLDrawableFactory.getFactory(glp);

    drawable =
        factory.createOffscreenAutoDrawable(factory.getDefaultDevice(), caps, new DefaultGLCapabilitiesChooser(),
            width, height);
    drawable.display();
    drawable.getContext().makeCurrent();

    // Set the camera
    setCamera();

    // First render
    display(false);
  }

  /**
   * Sets the render engine camera
   */
  private synchronized void setCamera()
  {
    GL2 gl = drawable.getGL().getGL2();
    // Use the Projection Matrix
    gl.glMatrixMode(GL2.GL_PROJECTION);
    gl.glLoadIdentity();

    gl.glOrtho(0f, 10f, 0f, 10f, -10f, 10f);

    // Change back to model view matrix.
    gl.glMatrixMode(GL2.GL_MODELVIEW);
    gl.glLoadIdentity();
  }

  /**
   * Displays the cube
   */
  private synchronized void display(boolean isInteractiveFrame)
  {
    GL2 gl = drawable.getGL().getGL2();
    // Clear Color and Depth Buffers
    gl.glClear(GL2.GL_COLOR_BUFFER_BIT | GL2.GL_DEPTH_BUFFER_BIT);
    // Reset transformations
    gl.glLoadIdentity();
    // draw cube
    cube.drawCube(gl);
    // apply transformation matrix after displaying
    cube.updateCubeLocation(trans);
    // Reset transformations
    trans.identity();

    // read the drawable's pixels
    AWTGLReadBufferUtil util = new AWTGLReadBufferUtil(drawable.getGLProfile(), false);
    frame.bufferedImage = util.readPixelsToBufferedImage(drawable.getGL(), 0, 0, width, height, true);

    // check for errors, at least once per frame
    int error = gl.glGetError();
    if ( error != GL2.GL_NO_ERROR )
    {
      System.out.println("OpenGL Error");
      System.exit(1);
    }

    frame.isInteractive = isInteractiveFrame;
    frame.id++;
  }

  /**
   * Returns frame
   */
  public synchronized Frame getFrame()
  {
    return frame;
  }

  /**
   * Mouse pressed event
   */
  public synchronized void mousePressed(int x, int y, SoMouseButtonEvent.Buttons button)
  {
    // Update mouse position
    prevMouseX = x;
    prevMouseY = y;

    // Update left mouse button state
    if ( button == SoMouseButtonEvent.Buttons.BUTTON1 )
    {
      isLeftButtonDown = true;
    }
  }

  /**
   * Mouse released event
   */
  public synchronized void mouseReleased(int x, int y, SoMouseButtonEvent.Buttons button)
  {
    // Update left mouse button state
    if ( button == SoMouseButtonEvent.Buttons.BUTTON1 )
    {
      isLeftButtonDown = false;

      display(false);
    }
  }

  /**
   * Mouse dragged event
   */
  public synchronized void mouseDragged(int x, int y)
  {
    if ( isLeftButtonDown )
    {
      float thetaY = 360f * ((float) (prevMouseX - x) / (float) width);
      float thetaX = 360f * ((float) (prevMouseY - y) / (float) height);

      // Update the transformation matrix
      trans.translate(5f, 5f, 0);
      trans.rotateX(thetaX);
      trans.rotateY(thetaY);
      trans.translate(-5f, -5f, 0);

      display(true);
    }

    // Update mouse position
    prevMouseX = x;
    prevMouseY = y;
  }
}
