package remoteviz.independentservice.independentrenderingservice;

import com.openinventor.remoteviz.rendering.Connection;
import com.openinventor.remoteviz.rendering.RenderArea;
import com.openinventor.remoteviz.rendering.RenderAreaListener;

import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.openinventor.inventor.devices.SoBufferObject;
import com.openinventor.inventor.devices.SoCpuBufferObject;
import com.openinventor.inventor.events.*;
import com.openinventor.inventor.image.SbRasterImage;

public class IndependentRenderArea extends RenderAreaListener
{

  private Engine engine;
  private Map<String, Long> lastConnectionFrame;

  /**
   * Default Constructor
   */
  public IndependentRenderArea(RenderArea renderArea)
  {
    engine = new Engine(renderArea.getWidth(), renderArea.getHeight());

    // Initialize the render engine
    engine.init();

    lastConnectionFrame = Collections.synchronizedMap(new HashMap<String, Long>());
  }

  @Override
  /**
   * Triggered when a new frame is requested by a connection.
   */
  public boolean onRequestedFrame(RenderArea renderArea, Connection sender, SbRasterImage rasterImage)
  {
    Engine.Frame frame = engine.getFrame();
    boolean isInteractive = frame.isInteractive;

    if ( lastConnectionFrame.containsKey(sender.getId()) )
    {
      // Check if a new frame is available
      if ( lastConnectionFrame.get(sender.getId()) != frame.id )
      {
        lastConnectionFrame.put(sender.getId(), frame.id);
        bufferedImageToRasterImage(rasterImage, frame.bufferedImage);
      }
    }
    else // new connection
    {
      lastConnectionFrame.put(sender.getId(), frame.id);

      bufferedImageToRasterImage(rasterImage, frame.bufferedImage);
    }

    return isInteractive;
  }

  /**
   *  Triggered when a client disconnects.
   */
  @Override
  public void onClosedConnection(RenderArea renderArea, String connectionId, boolean aborted)
  {
    lastConnectionFrame.remove(connectionId);
    super.onClosedConnection(renderArea, connectionId, aborted);
  }

  /**
   * Convert a BufferedImage to a SbRasterImage
   */
  private void bufferedImageToRasterImage(SbRasterImage rasterImage, BufferedImage bufferedImage)
  {
    SoCpuBufferObject cpuBuffer = new SoCpuBufferObject();

    cpuBuffer.setSize(bufferedImage.getWidth() * bufferedImage.getHeight() * 3);

    ByteBuffer data = cpuBuffer.map(SoBufferObject.AccessModes.SET);

    int[] bufferedImageData = ((DataBufferInt) bufferedImage.getRaster().getDataBuffer()).getData();

    int pixel = 0;
    for ( int i = 0; i < data.capacity(); i += 3 )
    {
      pixel = bufferedImageData[i / 3];
      data.put(i, (byte) ((pixel & 0xff0000) >> 16)); // R
      data.put(i + 1, (byte) ((pixel & 0xff00) >> 8)); // G
      data.put(i + 2, (byte) (pixel & 0xff)); // B
    }

    cpuBuffer.unmap();
    rasterImage.setBuffer(cpuBuffer);
  }

  @Override
  /**
   * Triggered when a MouseDown event is received from the client.
   */
  public boolean onMouseDown(RenderArea renderArea, Connection sender, int x, int y, SoMouseButtonEvent.Buttons button)
  {
    // Update engine
    engine.mousePressed(x, y, button);

    return true;
  }

  @Override
  /**
   * Triggered when a MouseUp event is received from the client.
   */
  public boolean onMouseUp(RenderArea renderArea, Connection sender, int x, int y, SoMouseButtonEvent.Buttons button)
  {
    // Update engine
    engine.mouseReleased(x, y, button);

    return true;
  }

  @Override
  /**
   * Triggered when a MouseMove event is received from the client.
   */
  public boolean onMouseMove(RenderArea renderArea, Connection sender, int x, int y)
  {
    // Update engine
    engine.mouseDragged(x, y);

    return true;
  }

}
