package volumeviz.advanced.volRend;

/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volumeData and volume geometry.
          A standard manipulator allows the geometry to be moved through the volume.
author : Jerome Hummel
August 2002

Note:
  Correct rendering of volume geometry requires OpenGL 3D texturing.

  All boards that report OpenGL version 1.2 or higher are required to
  support 3D texturing and are capable of rendering the correct image.

  However on some boards, e.g. GeForce2, this support is not hardware
  accelerated and may be very slow.  Especially for actual volume
  rendering using SoVolumeRender.  By default VolumeViz tries to
  detect this situation and automatically *disable* use of 3D textures.
  In this case you may see volume geometry rendered with no texture
  (i.e. just gray triangles).

  To enable use of 3D textures in VolumeViz in all cases, set the
  environment variable IVVR_USE_TEX3D to 1 (see SoPreferences).
----------------------------------------------------------------------------------------*/

import com.openinventor.volumeviz.nodes.*;
import com.openinventor.inventor.manips.*;
import com.openinventor.inventor.nodes.*;

public class VolumeGeometry {
  // Positions of all of the vertices
  private static float[][] vertexPositions = {
      {-0.9f,-0.9f,-0.3f},
      {-0.3f,-0.9f, 0.0f},
      {0.3f,-0.9f, 0.0f},
      {0.6f,-0.9f,-0.3f},
      {-0.9f,-0.3f,-0.1f},
      {-0.3f,-0.3f, 0.0f},
      {0.3f,-0.3f, 0.0f},
      {0.6f,-0.3f,-0.1f},
      {-0.9f, 0.3f, 0.1f},
      {-0.3f, 0.3f, 0.0f},
      {0.3f, 0.3f, 0.0f},
      {0.6f, 0.3f, 0.1f},
      {-0.9f, 0.9f, 0.3f},
      {-0.3f, 0.9f, 0.0f},
      {0.3f, 0.9f, 0.0f},
      {0.6f, 0.9f, 0.3f}
  };

  // Number of vertices in each strip.
  static int[] coordIndex = {
      0, 4, 1, 5, 2, 6, 3, 7, -1,
      4, 8, 5, 9, 6,10, 7,11, -1,
      8, 12, 9, 13, 10, 14, 11, 15, -1
  };

  SoVolumeIndexedTriangleStripSet m_volGeom;

  public VolumeGeometry() {
    m_volGeom = new SoVolumeIndexedTriangleStripSet();
  }

  public SoSeparator createVolumeGeometryGraph() {
    // Define coordinates and base color for volume geometry
    SoVertexProperty myVertexProperty = new SoVertexProperty();
    myVertexProperty.orderedRGBA.set1Value(0, 0xFFFFFFFF);
    myVertexProperty.vertex.setValues(0, vertexPositions);

    // Volume geometry node
    m_volGeom.coordIndex.setValues(0, coordIndex);
    m_volGeom.vertexProperty.setValue(myVertexProperty);
    m_volGeom.offset.setValue(0.0f);
    m_volGeom.clipGeometry.setValue(false);

    // This ShapeHints will enable two-sided lighting for the geometry
    SoShapeHints pHints = new SoShapeHints();
    pHints.vertexOrdering.setValue(SoShapeHints.VertexOrderings.COUNTERCLOCKWISE);
    pHints.creaseAngle.setValue(3.14f);

    // Assemble the scene graph
    // Note: Volume geometry must appear after the SoVolumeData node.
    SoSeparator root = new SoSeparator();
    {
      root.addChild(pHints);                 // Shape hints
      root.addChild(new SoTransformerManip()); // Manip to move geometry
      root.addChild(m_volGeom);               // Volume geometry
    }

    return root;
  }

  void setVolumeGeometryClipping(boolean clipGeometry)
  {
    m_volGeom.clipGeometry.setValue(clipGeometry);
  }

  void setVolumeGeometryOffset(float offset)
  {
    m_volGeom.offset.setValue(offset);
  }

  void setVolumeGeometryInterpolation(boolean interpolation)
  {
    if (interpolation)
      m_volGeom.interpolation.setValue(SoVolumeShape.Interpolations.LINEAR);
    else
      m_volGeom.interpolation.setValue(SoVolumeShape.Interpolations.NEAREST);
  }
}
