package volumeviz.sample.bonesMuscles;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.regex.Pattern;

import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoGLRenderAction;
import com.openinventor.inventor.events.SoLocation2Event;
import com.openinventor.inventor.events.SoMouseButtonEvent;
import com.openinventor.inventor.nodes.*;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.nodes.SoDataRange;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{
  // Data Set
  static final String FILENAME = "$OIVJHOME/data/volumeviz/medicalFoot.ldm";
  // VSG Logo
  static final String IMG_FILENAME = "$OIVJHOME/data/textures/png/VSGBanner60.png";
  // color map that can be edited in Avizo.
  static final String COLORMAP_FILENAME = "/data/volumeviz/bonesMuscles.col.am";

  private IViewerExaminer myViewer;

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    SoSeparator root = new SoSeparator();

    // Separator for data.
    SoSeparator dataSep = new SoSeparator();

    // Increase the rendering quality when you stop to move the dataset.
    SoInteractiveComplexity cplx = new SoInteractiveComplexity();

    // All data under this group will be shadowed.
    SoShadowGroup data = new SoShadowGroup();
    data.method.setValue(SoShadowGroup.ShadowingMethods.VARIANCE_SHADOW_MAP);
    data.quality.setValue(3.0f);
    data.intensity.setValue(0.7f);
    data.smoothFactor.setValue(4);
    data.isActive.setValue(true);

    // Separator for the foot support.
    SoSeparator suppportSep = new SoSeparator();
    SoTranslation suppTrans = new SoTranslation();
    suppTrans.translation.setValue(10, 0f, 0f);

    // The support.
    SoCube support = new SoCube();
    support.depth.setValue(50.0f);
    support.width.setValue(50.0f);
    support.height.setValue(0.05f);

    // Node to hold the volume data
    SoVolumeData volData = new SoVolumeData();
    volData.storageHint.setValue(SoVolumeData.StorageHints.TEX3D);
    // Load the model in the SoVolumeData
    volData.fileName.setValue(FILENAME);

    // Load the colorMap from an Avizo colormap.
    SoTransferFunction transFunc = new SoTransferFunction();
    float[] values;
    values = loadColorTable(SoPreferences.getValue("OIVJHOME") + COLORMAP_FILENAME);
    transFunc.colorMap.setValues(0, values);

    // Set data default range.
    SoDataRange localDataRange = new SoDataRange();
    localDataRange.min.setValue(-868.93f);
    localDataRange.max.setValue(2659.00f);

    // Property node which allows SoVolumeRender to draw High Quality volumes.
    SoVolumeRenderingQuality vRVolQuality = new SoVolumeRenderingQuality();
    vRVolQuality.lighting.setValue(true);
    vRVolQuality.preIntegrated.setValue(true);
    vRVolQuality.edgeColoring.setValue(true);
    vRVolQuality.jittering.setValue(true);
    vRVolQuality.gradientQuality.setValue(SoVolumeRenderingQuality.GradientQualities.MEDIUM);

    // Node in charge of drawing the volume
    SoVolumeRender volRender = new SoVolumeRender();
    volRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.MANUAL);
    volRender.numSlices.setValue(512);

    // Set the ambient, diffuse, transparency, specular and shininess of the
    // material
    SoMaterial material = new SoMaterial();
    material.ambientColor.setValue(.0f, .0f, .0f);
    material.diffuseColor.setValue(1, 1, 1);
    material.transparency.setValue(0.0f);
    material.specularColor.setValue(1, 1, 1);
    material.shininess.setValue(0.5f);

    // Define background color
    SoGradientBackground bg = new SoGradientBackground();

    // Define VSG logo
    SoImageBackground imgBg = new SoImageBackground();
    imgBg.filename.setValue(IMG_FILENAME);
    imgBg.style.setValue(SoImageBackground.Styles.UPPER_RIGHT);

    SoRotationXYZ rotX = new SoRotationXYZ();
    rotX.axis.setValue(SoRotationXYZ.AxisType.X);
    rotX.angle.setValue(-3.14f / 2);
    SoRotationXYZ rotZ = new SoRotationXYZ();
    rotZ.axis.setValue(SoRotationXYZ.AxisType.Z);
    rotZ.angle.setValue(-3.14f / 2);

    // Mouse event callback to manage data range.
    SoEventCallback mouseMoveEvent = new SoEventCallback();
    SoEventCallback mouseKeyEvent = new SoEventCallback();

    // Text 2D to inform the user.
    SoSeparator localAnnotation = new SoSeparator();
    // Text 2D font.
    SoFont textFont = new SoFont();
    textFont.size.setValue(12);

    SoTranslation textTrans = new SoTranslation();
    textTrans.translation.setValue(-0.98f, 0.9f, 0);

    SoText2 instruction = new SoText2();
    instruction.string.set1Value(0, "In selection mode :");
    instruction.string.set1Value(1, "  . Drag horizontally with left mouse button pressed to change the min data range.");
    instruction.string.set1Value(2, "  . Drag vertically with left mouse button pressed to change the max data range.");

    // Add Children
    root.addChild(dataSep);

    dataSep.addChild(cplx);
    cplx.fieldSettings.set1Value(0, "SoVolumeRender numSlices 200 1000 1500");
    cplx.fieldSettings.set1Value(1, "SoVolumeRender lowScreenResolutionScale 2 1 -1");
    cplx.fieldSettings.set1Value(2, "SoVolumeRenderingQuality gradientQuality LOW MEDIUM");

    dataSep.addChild(material);
    dataSep.addChild(bg);
    dataSep.addChild(imgBg);
    dataSep.addChild(data);

    data.addChild(suppportSep);
    suppportSep.addChild(suppTrans);
    suppportSep.addChild(support);

    data.addChild(rotX);
    data.addChild(rotZ);
    data.addChild(volData);
    data.addChild(localDataRange);
    data.addChild(transFunc);
    data.addChild(vRVolQuality);
    data.addChild(volRender);
    data.addChild(mouseMoveEvent);
    data.addChild(mouseKeyEvent);

    root.addChild(localAnnotation);
    // Camera for Text 2D.
    localAnnotation.addChild(new SoOrthographicCamera());
    localAnnotation.addChild(textFont);
    localAnnotation.addChild(textTrans);
    localAnnotation.addChild(instruction);

    // get mouse events to change the colormap ranges.
    MouseEventsCB mouseEventsCB = new MouseEventsCB(localDataRange);
    mouseMoveEvent.addEventCallback(SoLocation2Event.class, mouseEventsCB, null);
    mouseKeyEvent.addEventCallback(SoMouseButtonEvent.class, mouseEventsCB, null);

    // Set up viewer:
    myViewer.setSceneGraph(root);
    myViewer.getRenderArea().setTransparencyType(SoGLRenderAction.TransparencyTypes.DELAYED_BLEND);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private float[] loadColorTable(String colorFileName)
  {
    // Open the color file
    FileReader fr = null;
    try
    {
      fr = new FileReader(colorFileName);
    }
    catch (FileNotFoundException e)
    {
      System.err.println("Can't open the color map file " + colorFileName);
      return null;
    }

    // Get first line of file and try to identify it.
    BufferedReader reader = new BufferedReader(fr);
    String line = null;
    try
    {
      line = reader.readLine();
    }
    catch (IOException e)
    {
      e.printStackTrace();
      return null;
    }

    // If Amira format, use specific file loader
    if ( line != null )
    {
      if ( !line.matches("^\\s*#\\s*AmiraMesh.*") )
      {
        System.err.println("# AmiraMesh not found in file " + colorFileName);
        return null;
      }
    }
    else
    {
      return null;
    }

    // Get the rest of the header

    // Find "define Lattice <nb>"
    int numEntries = 0;

    while ( numEntries == 0 )
    {
      try
      {
        line = reader.readLine();
      }
      catch (IOException e)
      {
        e.printStackTrace();
        return null;
      }

      if ( line != null )
      {
        if ( line.matches("^\\s*define\\s*Lattice\\s*[0-9]+.*") )
        {
          Pattern p = Pattern.compile("\\s+");
          String[] result = p.split(line.trim());
          try
          {
            numEntries = new Integer(result[2]);
          }
          catch (NumberFormatException e)
          {
            e.printStackTrace();
            return null;
          }
        }
      }
      else
      {
        System.err.println("define Lattice not found in file " + colorFileName);
        return null;
      }
    }

    // Find "ContentType "Colormap""
    boolean isColorMapFile = false;

    while ( !isColorMapFile )
    {
      try
      {
        line = reader.readLine();
      }
      catch (IOException e)
      {
        e.printStackTrace();
        return null;
      }

      if ( line != null )
      {
        if ( line.matches("^\\s*ContentType\\s*\"Colormap\".*") )
        {
          isColorMapFile = true;
        }
      }
      else
      {
        System.err.println("ContentType \"Colormap\" not found in file " + colorFileName);
        return null;
      }
    }

    // Find "@1" at beginning of line
    boolean foundDataSection = false;

    while ( !foundDataSection )
    {
      try
      {
        line = reader.readLine();
      }
      catch (IOException e)
      {
        e.printStackTrace();
        return null;
      }

      if ( line != null )
      {
        if ( line.matches("^\\s*@1.*") )
        {
          foundDataSection = true;
        }
      }
      else
      {
        System.err.println("@1 not found in file " + colorFileName);
        return null;
      }
    }

    // Read the values
    ArrayList<Float> rgba = new ArrayList<Float>();
    Pattern p = Pattern.compile("\\s+");
    int count = 0;
    while ( count < numEntries )
    {
      try
      {
        line = reader.readLine();
      }
      catch (IOException e)
      {
        e.printStackTrace();
        return null;
      }

      if ( (line != null) && !line.isEmpty() )
      {
        String[] floats = p.split(line.trim());

        for ( int i = 0; i < 4; i++ )
        {
          try
          {
            rgba.add(new Float(floats[i]));
          }
          catch (NumberFormatException e)
          {
            e.printStackTrace();
            return null;
          }
        }
        count++;
      }
    }

    float[] rgbaArray = new float[rgba.size()];
    for ( int i = 0; i < rgba.size(); i++ )
    {
      rgbaArray[i] = rgba.get(i);
    }

    try
    {
      reader.close();
    }
    catch (IOException e)
    {
      e.printStackTrace();
      return null;
    }
    return rgbaArray;
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("Bones  Muscles");
  }
}
