/*----------------------------------------------------------------------------------------
 * Example program.
 * Purpose : Demonstrate how to use a custom combining feature
 * author : Jean-Michel Godinaud
 /*----------------------------------------------------------------------------------------*/
package volumeviz.sample.dataCompositor.Custom;

import java.awt.BorderLayout;
import java.awt.Component;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbDataType.DataTypes;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.nodes.SoDirectionalLight;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.nodes.SoDataCompositor;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  static final boolean HighRenderingQuality = true;

  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Custom Data compositor");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    String fileName = "$OIVJHOME/data/volumeviz/3DHEAD.ldm";

    //data 1
    SoVolumeData pVolData1 = new SoVolumeData();
    pVolData1.fileName.setValue(fileName);
    SbBox3f size = pVolData1.extent.getValue();
    SbVec3i32 vol1dim = pVolData1.data.getSize();

    //data 2: mask volume
    int testXdim = vol1dim.getX();
    int testYdim = vol1dim.getY();
    int testZdim = vol1dim.getZ();

    int numTestBytes = testXdim * testYdim * testZdim;
    ByteBuffer testData = ByteBuffer.allocateDirect(numTestBytes);
    testData.clear();
    testData.order(ByteOrder.nativeOrder());

    for (int k = 0; k < testZdim; k++) {
      for (int j = 0; j < testYdim; j++) {
        for (int i = 0; i < testXdim; i++) {
          testData.put( (k % 20) < 10 ? (byte)0x00 : 0x01);
        }
      }
    }

    SoVolumeData pVolData = new SoVolumeData();
    pVolData.data.setValue(vol1dim, new SbDataType(DataTypes.UNSIGNED_BYTE), testData);
    pVolData.dataSetId.setValue(2);
    pVolData.extent.setValue(size);

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction pTransFunc = new SoTransferFunction();
    pTransFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.GLOW);
    pTransFunc.minValue.setValue(40);
    pTransFunc.maxValue.setValue(250);

    // Node in charge of drawing the volume
    SoVolumeRender pVolRender = new SoVolumeRender();
    pVolRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.ALL);

    //Custom compositor:
    // Alternatevily compose a "null" tile or a multiplication tile
    // That produces an animated effect : some parts of the volume seem
    // to be loaded and next time unloaded
    SoDataCompositor dataCompositor = new SoDataCompositor() {
      @Override
      public void compose(SbVec3i32 tile_dimension,
                          int[] volume_ids,
                          Buffer[] input_buffer,
                          int[] data_types,
                          Buffer output_buffer)
      {
          // compose by multiplying input buffers
          byte b;
          while (output_buffer.hasRemaining()) {
            b = 0x01;
            for (int j = 0; j < input_buffer.length; j++) {
              b *= ((ByteBuffer)input_buffer[j]).get();
            }
            ((ByteBuffer)output_buffer).put(b);
          }
      }

    };
    dataCompositor.dataType.setValue(SoDataCompositor.DataTypes.UNSIGNED_BYTE);
    dataCompositor.preDefCompositor.setValue(SoDataCompositor.PreDefCompositors.NONE);

    // Property node which allows SoVolumeRender to draw lighted volumes
    SoVolumeRenderingQuality volQuality;
    if (HighRenderingQuality) {
      volQuality = new SoVolumeRenderingQuality();
      volQuality.lighting.setValue(true);
      volQuality.preIntegrated.setValue(true);
    }

    //Set the ambient, diffuse, transparency, specular and shininess of the material
    SoMaterial material;
    if (HighRenderingQuality) {
      material = new SoMaterial();
      material.ambientColor.setValue(.0f, .0f, .0f);
      material.diffuseColor.setValue(1, 1, 1);
      material.transparency.setValue(0.0f);
      material.specularColor.setValue(1, 1, 1);
      material.shininess.setValue(0.5f);
    }

    SbColor bkgColor = new SbColor(new SbColor(.5f, .5f, .5f));
    SoGradientBackground background = new SoGradientBackground();
    background.color0.setValue(bkgColor);
    background.color1.setValue(bkgColor);

    // Assemble the scene graph
    SoSeparator root = new SoSeparator();
    {
      root.addChild(background);
      root.addChild(dataCompositor);
      root.addChild(pVolData1);
      root.addChild(pVolData);
      root.addChild(pTransFunc);

      if (HighRenderingQuality) {
        //for rendering quality only
        root.addChild(new SoDirectionalLight());
        root.addChild(material);
        root.addChild(volQuality);
      }

      root.addChild(pVolRender);
    }

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

}
