/*----------------------------------------------------------------------------------------
 * Example program.
 * Purpose : Demonstrate how to use CPU combining feature
 * author : Jerome Hummel
 /*----------------------------------------------------------------------------------------*/
package volumeviz.sample.dataCompositor.Multiply;
import java.awt.BorderLayout;
import java.awt.Component;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import com.openinventor.inventor.SbBox3f;
import com.openinventor.inventor.SbColor;
import com.openinventor.inventor.SbDataType;
import com.openinventor.inventor.SbDataType.DataTypes;
import com.openinventor.inventor.SbVec3i32;
import com.openinventor.inventor.nodes.SoDirectionalLight;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoMaterial;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.ldm.SoLDMGlobalResourceParameters;
import com.openinventor.ldm.SoLDMGlobalResourceParameters.LoadListener;
import com.openinventor.ldm.nodes.SoDataCompositor;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoVolumeData;
import com.openinventor.volumeviz.nodes.SoVolumeRender;
import com.openinventor.volumeviz.nodes.SoVolumeRenderingQuality;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  static final boolean HighRenderingQuality = true;

  private IViewerExaminer myViewer;

  public static void main(String[] args) {
    Main example = new Main();
    example.demoMain("Data compositor");
  }

  @Override
  public void start() {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    String fileName = "$OIVJHOME/data/volumeviz/3DHEAD.ldm";

    //data 1
    SoVolumeData pVolData1 = new SoVolumeData();
    pVolData1.fileName.setValue(fileName);
    SbBox3f size = pVolData1.extent.getValue();
    SbVec3i32 vol1dim = pVolData1.getDimension();

    //data 2: mask volume
    int testXdim = vol1dim.getX();
    int testYdim = vol1dim.getY();
    int testZdim = vol1dim.getZ();

    int numTestBytes = testXdim * testYdim * testZdim;
    ByteBuffer testData = ByteBuffer.allocateDirect(numTestBytes);
    testData.clear();
    testData.order(ByteOrder.nativeOrder());

    for (int k = 0; k < testZdim; k++) {
      for (int j = 0; j < testYdim; j++) {
        for (int i = 0; i < testXdim; i++) {
          testData.put( (k % 20) < 10 ? (byte)0x00 : 0x01);
        }
      }
    }

    SoVolumeData pVolData = new SoVolumeData();
    pVolData.data.setValue(vol1dim, new SbDataType(DataTypes.UNSIGNED_BYTE), testData);
    pVolData.dataSetId.setValue(2);
    pVolData.extent.setValue(size);

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction pTransFunc = new SoTransferFunction();
    pTransFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.GLOW);
    pTransFunc.minValue.setValue(40);
    pTransFunc.maxValue.setValue(250);

    // Node in charge of drawing the volume
    SoVolumeRender pVolRender = new SoVolumeRender();
    pVolRender.numSlicesControl.setValue(SoVolumeRender.NumSlicesControls.ALL);

    //CPU compositor:
    SoDataCompositor dataCompositor = new SoDataCompositor();
    dataCompositor.dataType.setValue(SoDataCompositor.DataTypes.UNSIGNED_BYTE);
    dataCompositor.preDefCompositor.setValue(SoDataCompositor.PreDefCompositors.MULTIPLY);

    // Property node which allows SoVolumeRender to draw lighted volumes
    SoVolumeRenderingQuality volQuality;
    if (HighRenderingQuality) {
      volQuality = new SoVolumeRenderingQuality();
      volQuality.lighting.setValue(true);
      volQuality.preIntegrated.setValue(true);
    }

    //Set the ambient, diffuse, transparency, specular and shininess of the material
    SoMaterial material;
    if (HighRenderingQuality) {
      material = new SoMaterial();
      material.ambientColor.setValue(.0f, .0f, .0f);
      material.diffuseColor.setValue(1, 1, 1);
      material.transparency.setValue(0.0f);
      material.specularColor.setValue(1, 1, 1);
      material.shininess.setValue(0.5f);
    }

    SbColor bkgColor = new SbColor(new SbColor(.5f, .5f, .5f));
    SoGradientBackground background = new SoGradientBackground();
    background.color0.setValue(bkgColor);
    background.color1.setValue(bkgColor);

    // Assemble the scene graph
    SoSeparator root = new SoSeparator();
    {
      root.addChild(background);
      root.addChild(dataCompositor);
      root.addChild(pVolData1);
      root.addChild(pVolData);
      root.addChild(pTransFunc);

      if (HighRenderingQuality) {
        //for rendering quality only
        root.addChild(new SoDirectionalLight());
        root.addChild(material);
        root.addChild(volQuality);
      }

      root.addChild(pVolRender);
    }

    // define tasks
    SoLDMGlobalResourceParameters.addLoadListener(new LoadListener()
		{

			@Override
			public void loadStarted()
			{
				 System.out.println("start loading");
			}

			@Override
			public void loadFinished()
			{
				System.out.println("finish loading");
			}
		});

    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

}
