package volumeviz.sample.editingPicking;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayDeque;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import com.openinventor.inventor.SbEventListener;
import com.openinventor.inventor.SbVec2f;
import com.openinventor.inventor.SoDB;
import com.openinventor.inventor.SoInput;
import com.openinventor.inventor.SoPickedPoint;
import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.actions.SoHandleEventAction;
import com.openinventor.inventor.actions.SoRayPickAction;
import com.openinventor.inventor.details.SoDetail;
import com.openinventor.inventor.drawers.SoPolyLineScreenDrawer;
import com.openinventor.inventor.drawers.SoPolyLineScreenDrawer.EventArg;
import com.openinventor.inventor.drawers.SoPolygonScreenDrawer;
import com.openinventor.inventor.errors.SoError;
import com.openinventor.inventor.nodes.SoFaceSet;
import com.openinventor.inventor.nodes.SoNode;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoSwitch;
import com.openinventor.inventor.nodes.SoVertexProperty;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.volumeviz.details.SoSliceDetail;
import com.openinventor.volumeviz.details.SoVolumeDetail;
import com.openinventor.volumeviz.details.SoVolumeDetail.TransparentDoubleValue;
import com.openinventor.volumeviz.nodes.SoVolumeData;

import util.Example;
import util.ViewerComponentsFactory;

/**
 * Interactive volume editing with picking.
 * <p>
 * This demo use {@link SoPolygonScreenDrawer} and gpuPicking to draw a polygon
 * in volumeData. The polygon is then written to dataSet using the first clicked
 * voxel color. Use the selection mode (arrow), and draw a polygon on the
 * volume. You must click on the visible part of the volume. To finalize
 * polygon, double-click or hit enter key. Hit escape to cancel.
 *
 */
public class Main extends Example
{
  private IViewerExaminer myViewer;

  // root of scene graph
  private SoSeparator m_root;
  // shapes generated by extrusion
  private SoSwitch m_generatedShapes;
  // dataset
  private SoVolumeData m_dataSet;

  // thickness value
  private float m_thickness;
  // list of transaction ids that are currently applied
  private ArrayDeque<Integer> m_editions;
  // list of transaction ids that are currently undone
  private ArrayDeque<Integer> m_undoneEditions;

  class LineDrawerListener implements SbEventListener<EventArg>
  {
    @Override
    public void onEvent(EventArg event)
    {
      // Called when the polygon is finalized.
      // Compute intersection of each polygon point with volume,
      // convert to data space, generate a surface and voxelize it.

      SoPolyLineScreenDrawer drawer = event.getSource();
      SoHandleEventAction action = event.getAction();

      // generated surface
      SoVertexProperty vertexProperty = new SoVertexProperty();
      SoFaceSet surface = new SoFaceSet();
      surface.vertexProperty.setValue(vertexProperty);

      // value of first clicked point
      double firstValue = Double.NaN;

      // for each point of polygon
      final int numPoints = drawer.point.getNum();
      for ( int i = 0; i < numPoints; ++i )
      {
        SbVec2f point = drawer.point.getValueAt(i);

        // retrieve clicked point
        SoRayPickAction pickAction = new SoRayPickAction(action.getViewportRegion());
        // [-1, 1] to [0, 1]
        SbVec2f normPoint = new SbVec2f((point.getX() + 1.f) * 0.5f, (point.getY() + 1.f) * 0.5f);
        pickAction.setNormalizedPoint(normPoint);
        pickAction.setSceneManager(action.getSceneManager());
        pickAction.apply(myViewer.getRenderArea().getRootSceneGraph());

        SoPickedPoint pickedPoint = pickAction.getPickedPoint();
        if ( pickedPoint == null )
        {
          SoError.post("Polygon point " + i + " is not on dataset. Skip it.");
          continue;
        }

        vertexProperty.vertex.set1Value(vertexProperty.vertex.getNum(), pickedPoint.getPoint());

        // If we still not have 1st point value, compute it.
        if ( Double.isNaN(firstValue) )
        {
          // depending on what was clicked, retrieve value of clicked voxel if
          // possible.
          SoDetail detail = pickedPoint.getDetail();
          if ( detail instanceof SoVolumeDetail )
          {
            TransparentDoubleValue value = ((SoVolumeDetail) detail).getFirstNonTransparentDoubleValue();
            if ( value != null )
              firstValue = value.value;
          }
          else if ( detail instanceof SoSliceDetail )
          {
            firstValue = ((SoSliceDetail) detail).getValueD();
          }
          else
          {
            SoError.post("Cannot retrieve value of clicked point using current rendering mode.");
            drawer.clear();
            action.setHandled();
            return;
          }
        }
      }

      // Surface needs at least 3 points.
      if ( vertexProperty.vertex.getNum() < 3 )
      {
        SoError.post("Polygon needs at least 3 points");
        drawer.clear();
        action.setHandled();
        return;
      }

      surface.numVertices.set1Value(0, vertexProperty.vertex.getNum());

      // Edit dataset with generated surface.
      int editionId = m_dataSet.startEditing();
      m_dataSet.editSurfaceShape(surface, m_thickness, firstValue);
      m_dataSet.finishEditing(editionId);
      m_editions.push(editionId);

      if ( m_generatedShapes != null )
        m_generatedShapes.addChild(surface);

      drawer.clear();

      action.setHandled();
    }

  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    m_thickness = 1;
    m_editions = new ArrayDeque<Integer>();
    m_undoneEditions = new ArrayDeque<Integer>();

    m_root = new SoSeparator();

    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    String filePrefix =
        SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator;
    SoNode node = readSceneGraph(filePrefix + "scene.iv");
    m_root.addChild(node);

    m_dataSet = (SoVolumeData) SoNode.getByName("EditedDataSet");
    m_generatedShapes = (SoSwitch) SoNode.getByName("GeneratedShapes");
    final SoSwitch rendererSwitch = (SoSwitch) SoNode.getByName("RendererSwitch");

    SoPolygonScreenDrawer drawer = (SoPolygonScreenDrawer) SoNode.getByName("ScreenDrawer");
    drawer.onFinish.addEventListener(new LineDrawerListener());

    myViewer.setSceneGraph(m_root);
    myViewer.viewAll();

    JPanel optionsPanel = new JPanel();
    GridBagLayout gridBagLayout = new GridBagLayout();
    optionsPanel.setLayout(gridBagLayout);

    JButton undoButton = new JButton("Undo");
    undoButton.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( m_editions.isEmpty() )
        {
          SoError.post("Nothing to undo.");
          return;
        }
        int id = m_editions.pop();
        m_dataSet.undoEditing(id);
        m_undoneEditions.push(id);

      }
    });
    GridBagConstraints gbc_undoButton = new GridBagConstraints();
    gbc_undoButton.anchor = GridBagConstraints.WEST;
    gbc_undoButton.insets = new Insets(0, 0, 5, 5);
    gbc_undoButton.gridx = 0;
    gbc_undoButton.gridy = 0;
    optionsPanel.add(undoButton, gbc_undoButton);

    JButton redoButton = new JButton("Redo");
    redoButton.addActionListener(new ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent e)
      {
        if ( m_undoneEditions.isEmpty() )
        {
          SoError.post("Nothing to redo.");
          return;
        }
        int id = m_undoneEditions.pop();
        m_dataSet.redoEditing(id);
        m_editions.push(id);
      }
    });
    GridBagConstraints gbc_redoButton = new GridBagConstraints();
    gbc_redoButton.anchor = GridBagConstraints.WEST;
    gbc_redoButton.insets = new Insets(0, 0, 5, 5);
    gbc_redoButton.gridx = 0;
    gbc_redoButton.gridy = 1;
    optionsPanel.add(redoButton, gbc_redoButton);

    String[] renderers = new String[rendererSwitch.getNumChildren()];
    for ( int i = 0; i < renderers.length; i++ )
      renderers[i] = rendererSwitch.getChild(i).getName();

    final JComboBox<String> rendererComboBox = new JComboBox<>(renderers);
    rendererComboBox.addActionListener(new java.awt.event.ActionListener()
    {
      @Override
      public void actionPerformed(ActionEvent evt)
      {
        rendererSwitch.whichChild.setValue(rendererComboBox.getSelectedIndex());
      }
    });

    JPanel rendererPanel = new JPanel();
    rendererPanel.add(new JLabel("Show "));
    rendererPanel.add(rendererComboBox);
    GridBagConstraints gbc_rendererPanel = new GridBagConstraints();
    gbc_rendererPanel.anchor = GridBagConstraints.WEST;
    gbc_rendererPanel.insets = new Insets(0, 0, 5, 5);
    gbc_rendererPanel.gridx = 1;
    gbc_rendererPanel.gridy = 1;
    optionsPanel.add(rendererPanel, gbc_rendererPanel);

    SliderPanel thicknessSlider = new SliderPanel(1.f, 10.f, m_thickness, 1);
    thicknessSlider.addInfoText("Thickness : ");
    thicknessSlider.setSliderSize(new Dimension(200, 20));
    thicknessSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        m_thickness = value;
      }
    });
    GridBagConstraints gbc_thicknessSlider = new GridBagConstraints();
    gbc_thicknessSlider.anchor = GridBagConstraints.WEST;
    gbc_thicknessSlider.insets = new Insets(0, 0, 5, 5);
    gbc_thicknessSlider.gridx = 1;
    gbc_thicknessSlider.gridy = 0;
    optionsPanel.add(thicknessSlider, gbc_thicknessSlider);

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(optionsPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private static SoNode readSceneGraph(String fileName)
  {
    SoInput input = new SoInput();

    if ( !input.openFile(fileName) )
    {
      System.err.println("Cannot open file " + fileName);
      return null;
    }

    SoSeparator node = SoDB.readAll(input);
    if ( node == null )
    {
      System.err.println("Problem reading file");
      input.closeFile();
      return null;
    }

    input.closeFile();
    return node;
  }

  public static void main(String argv[])
  {
    Main example = new Main();
    example.demoMain("Editing Picking");
  }

}
