///////////////////////////////////////////////////////////////////////
//
// This example shows how to use custom shaders with HeightFieldRender.
//
// Sliders allow to modify uniform variables used in fragment shader.
// When firstIsoline parameter changes the height of first isoline drawn on height field while
// isolineStep change the step between each isolines.
//
///////////////////////////////////////////////////////////////////////

package volumeviz.sample.horizonIsolines;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.io.File;

import javax.swing.BoxLayout;
import javax.swing.JPanel;

import com.openinventor.inventor.SoPreferences;
import com.openinventor.inventor.nodes.SoFragmentShader;
import com.openinventor.inventor.nodes.SoGradientBackground;
import com.openinventor.inventor.nodes.SoScale;
import com.openinventor.inventor.nodes.SoSeparator;
import com.openinventor.inventor.nodes.SoShaderParameter1f;
import com.openinventor.inventor.nodes.SoShaderParameter1i;
import com.openinventor.inventor.viewercomponents.awt.IViewerExaminer;
import com.openinventor.inventor.viewercomponents.awt.tools.SliderPanel;
import com.openinventor.ldm.nodes.SoMultiDataSeparator;
import com.openinventor.ldm.nodes.SoTransferFunction;
import com.openinventor.volumeviz.nodes.SoHeightFieldGeometry;
import com.openinventor.volumeviz.nodes.SoHeightFieldProperty;
import com.openinventor.volumeviz.nodes.SoHeightFieldRender;
import com.openinventor.volumeviz.nodes.SoVolumeShader;

import util.Example;
import util.ViewerComponentsFactory;

public class Main extends Example
{

  private SoHeightFieldGeometry geometry;
  private SoShaderParameter1f isolineStep;
  private SoShaderParameter1f firstIsoline;
  private IViewerExaminer myViewer;

  public static void main(String[] args)
  {
    Main example = new Main();
    example.demoMain("HorizonIsolines");
  }

  public SoVolumeShader configureShaders(String shader)
  {
    // Initialize and set the fragment shader program.
    SoVolumeShader shaderProgram = new SoVolumeShader();

    SoFragmentShader fragment =
        shaderProgram.setFragmentShader(SoVolumeShader.ShaderPositions.FRAGMENT_COMPUTE_COLOR.getValue(), shader);

    SoShaderParameter1i propertyID = new SoShaderParameter1i();
    propertyID.name.setValue("propertyID");
    propertyID.value.setValue(2);

    // Set the uniform variable which allow to change the step between isolines.
    isolineStep = new SoShaderParameter1f();
    isolineStep.name.setValue("isolineStep");
    isolineStep.value.setValue(10);

    // Set the uniform variable which allow to change the height of isoline.
    firstIsoline = new SoShaderParameter1f();
    firstIsoline.name.setValue("firstIsoline");
    firstIsoline.value.setValue(1);

    // Set the uniform variable which allow to convert height values.
    long minMax[] = geometry.getMinMax();
    SoShaderParameter1i heightRange = new SoShaderParameter1i();
    heightRange.name.setValue("heightRange");
    heightRange.value.setValue((int) (minMax[1] - minMax[0]));

    fragment.parameter.set1Value(0, propertyID);
    fragment.parameter.set1Value(1, isolineStep);
    fragment.parameter.set1Value(2, firstIsoline);
    fragment.parameter.set1Value(3, heightRange);

    return shaderProgram;
  }

  @Override
  public void start()
  {
    myViewer = ViewerComponentsFactory.createViewerExaminer();

    String pkgName = this.getClass().getPackage().getName();
    pkgName = pkgName.replace('.', File.separatorChar);
    // Path to the fragment shader.
    String shader =
        SoPreferences.getValue("OIVJHOME") + File.separator + "examples" + File.separator + pkgName + File.separator
            + "composition.glsl";
    // Path to the DataSet.
    String data =
        SoPreferences.getValue("OIVJHOME") + File.separator + "data" + File.separator + "volumeviz" + File.separator
            + "horizon.ldm";

    SoMultiDataSeparator separator = new SoMultiDataSeparator();

    SoScale scale = new SoScale();
    scale.scaleFactor.setValue(1.f, 1.f, 0.2f);

    // Node containing height values.
    geometry = new SoHeightFieldGeometry();
    geometry.fileName.setValue(data);
    geometry.dataSetId.setValue(1);

    // Node containing the property and is used to compute color.
    SoHeightFieldProperty heightFieldproperty = new SoHeightFieldProperty();
    heightFieldproperty.fileName.setValue(data);
    heightFieldproperty.dataSetId.setValue(2);

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction pTransFunc = new SoTransferFunction();
    pTransFunc.predefColorMap.setValue(SoTransferFunction.PredefColorMaps.STANDARD);

    // Node in charge of drawing the height field.
    SoHeightFieldRender pVolRender = new SoHeightFieldRender();

    SoSeparator root = new SoSeparator();
    { // Assemble the scene graph
      root.addChild(new SoGradientBackground());
      root.addChild(separator);
      separator.addChild(configureShaders(shader));
      separator.addChild(scale);
      separator.addChild(geometry);
      separator.addChild(pTransFunc);
      separator.addChild(heightFieldproperty);
      separator.addChild(pVolRender);
    }

    // Set up viewer:
    myViewer.setSceneGraph(root);
    myViewer.viewAll();

    // SWING part
    // Add GUI allowing to interact with uniform variables.
    JPanel sliderPanel = createSliderPanel();

    final Component component = myViewer.getComponent();
    component.setPreferredSize(new java.awt.Dimension(600, 500));
    setLayout(new BorderLayout());
    add(component);
    add(sliderPanel, BorderLayout.SOUTH);
  }

  @Override
  public void stop()
  {
    myViewer.dispose();
  }

  private JPanel createSliderPanel()
  {
    long minMax[] = geometry.getMinMax();
    SliderPanel levelInnerSlider = new SliderPanel(minMax[0], minMax[1], 1.f, 1);
    levelInnerSlider.addInfoText("First isoline : ");
    levelInnerSlider.setSliderSize(new Dimension(200, 20));
    levelInnerSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        firstIsoline.value.setValue(value);
      }
    });

    SliderPanel levelOuterSlider = new SliderPanel(1.f, 50.f, 10.f, 1);
    levelOuterSlider.addInfoText("Isolines step : ");
    levelOuterSlider.setSliderSize(new Dimension(200, 20));
    levelOuterSlider.addSliderPanelListener(new SliderPanel.Listener()
    {
      @Override
      public void stateChanged(float value)
      {
        isolineStep.value.setValue(value);
      }
    });

    JPanel sliderPanel = new JPanel();
    sliderPanel.setLayout(new BoxLayout(sliderPanel, BoxLayout.Y_AXIS));
    sliderPanel.add(levelInnerSlider);
    sliderPanel.add(levelOuterSlider);

    return sliderPanel;
  }

}
